#ifndef _IAPLISTENER_H_
#define _IAPLISTENER_H_

#include <buy/BuyItemInformation.h>
#include <buy/BuyPurchasedItemInformation.h>
#include <buy/BuyPurchaseTicket.h>

/**
 * The listener interface to retrieve the responses for the requests sent from
 * IapClient. To retrieve information from IapClient, this interface should be
 * implemented and set to the IapClient instance.
 *
 */

using namespace Tizen::Base;

class BuyListener
{
public:
    enum StatusCode
    {
        Unknown = -1,               /*!< The status code for unkown status. */
        Succeed = 0,                /*!< The status code for success. */
        Cancel = 100,               /*!< The status code if the user cancels. */
        NetworkError = 200,         /*!< The status code for network errors. */
        ProcessError = 9000,        /*!< The status code for process errors. */
        ServiceUnavailable = 9200,  /*!< The status code for service errors. */
        ItemGroupIdNotFound = 9201, /*!< The status code if the item group ID is not found. */
        PaymentIdNotFound = 9203,   /*!< The status code if the payment ID is not found. */
        ItemIdNotFound = 9207       /*!< The status code if the item ID is not found. */
    };

public:
    virtual ~BuyListener()
    {
    }

    /**
     * Returns string describing StatusCode value
     */
    static String GetErrorString( StatusCode statusCode )
    {
    	if ( statusCode == Succeed )
    		return String("Success");
    	else if (statusCode == Cancel)
    		return String("Canceled");
    	else if (statusCode == NetworkError)
    		return String("Network error");
    	else if (statusCode == ProcessError)
    		return String("Process error");
    	else if (statusCode == ServiceUnavailable)
    		return String("Service unavailable");
    	else if (statusCode == ItemGroupIdNotFound)
    		return String("Item group Id not found");
    	else if (statusCode == PaymentIdNotFound)
    		return String("Payment Id not found");
    	else if (statusCode == ItemGroupIdNotFound)
    		return String("Item group id not found");
    	else if (statusCode == ItemIdNotFound)
    		return String("Item Id not found");
    	else if (statusCode == Unknown)
    		return String("Unknown");
    	else
    		return String("Unknown status code returned.");
	}

    /**
     * This method will be called when the item information is retrieved.
     *
     * @see BuyLib#requestItemInformationList
     * @param transactionId
     *            The transaction ID which was sent by application
     * @param statusCode
     *            Result of the request. Success if
     *            {@link BuyLib#StatusCode}
     * @param itemInformationList
     *            The item information list
     */
    virtual void OnItemInformationListReceived(int aTransactionId, StatusCode aStatusCode,
            const BuyItemInformationList & aItemInformationList) { AppLog("Not implemented"); };

    /**
     * This method will be called when the purchased item information is
     * retrieved.
     *
     * @see BuyLib#requestPurchasedItemInformationList
     * @param transactionId
     *            The transaction ID which was sent by application
     * @param statusCode
     *            Result of the request. Success if
     *            {@link BuyLib#StatusCode}
     * @param purchasedItemInformationList
     *            The purchased item information list
     */
    virtual void OnPurchasedItemInformationListReceived(int aTransactionId, StatusCode aStatusCode,
            const BuyPurchasedItemInformationList & aPurchasedItemInformationList) { AppLog("Not implemented"); };

    /**
     * This method will be called when purchasing the item is started.
     *
     * @see BuyLib#requestPurchaseItem
     * @see BuyLib#onPurchaseItemFinished
     * @param transactionId
     *            The transaction ID which was sent by application
     * @param statusCode
     *            Result of the request. Success if
     *            {@link BuyLib#StatusCode}
     * @param purchaseTicket
     *            The information of the transaction which is purchasing
     */
    virtual void OnPurchaseItemInitialized(int aTransactionId, StatusCode aStatusCode,
            const BuyPurchaseTicket & aPurchaseTicket) { AppLog("Not implemented"); };

    /**
     * This method will be called when purchasing the item is finished.
     *
     * @see BuyLib#requestPurchaseItem
     * @see BuyLib#onPurchaseItemInitialized
     * @param transactionId
     *            The transaction ID which was sent by application
     * @param statusCode
     *            Result of the request. Success if
     *            {@link BuyLib#StatusCode}
     * @param purchasedItemInformation
     *            The purchased item information
     */
    virtual void OnPurchaseItemFinished(int aTransactionId, StatusCode aStatusCode,
            const BuyPurchasedItemInformation & aPurchasedItemInformation) { AppLog("Not implemented"); };

    /**
     * This method will be called when purchasing the item is finished.
     *
     * @see BuyLib#requestPurchaseItem
     * @see BuyLib#onPurchaseItemInitialized
     * @param transactionId
     *            The transaction ID which was sent by application
     * @param statusCode
     *            Result of the request. Success if
     *            {@link BuyLib#StatusCode}
     * @param purchasedItemInformation
     *            The purchased item information
     */
    virtual void OnCountryListReceived(int aTransactionId, StatusCode aStatusCode,
            const Tizen::Base::Collection::HashMap * countryList) { AppLog("Not implemented"); };
};

#endif // _IAPLISTENER_H_
