#include "buy/BuyAppControlSerialization.h"

#include <buy/BuyUtils.h>
#include <buy/BuyItemInformation.h>
#include <buy/BuyPurchasedItemInformation.h>
#include <buy/BuyPurchaseTicket.h>

#include "buy/BuyAppControlTypes.h"

using namespace Tizen::Base;

namespace BuyAppControlSerialization {

namespace Helpers {

    result ReadItemListData(const Collection::IMap &aMap, int &aStartNumOut, int &aEndNumOut, int &aTotalCountOut)
    {
        result retval = BuyHashMapUtils::GetIntegerValue(&aMap, BUY_APPCTRL_KEY_START_NUMBER, aStartNumOut);
        TryReturn(retval == E_SUCCESS, retval, "%S: %s", BUY_APPCTRL_KEY_START_NUMBER, GetErrorMessage(retval));

        retval = BuyHashMapUtils::GetIntegerValue(&aMap, BUY_APPCTRL_KEY_END_NUMBER, aEndNumOut);
        TryReturn(retval == E_SUCCESS, retval, "%S: %s", BUY_APPCTRL_KEY_END_NUMBER, GetErrorMessage(retval));

        retval = BuyHashMapUtils::GetIntegerValue(&aMap, BUY_APPCTRL_KEY_TOTAL_COUNT, aTotalCountOut);
        TryReturn(retval == E_SUCCESS, retval, "%S: %s", BUY_APPCTRL_KEY_TOTAL_COUNT, GetErrorMessage(retval));

        return retval;
    }

    template<class T>
    result ReadItemList(result (*itemReader)(const Collection::IMap &, T *, const String &), const Collection::IMap &map, BuyItemList<T> *infoList)
    {
        result retval = E_INVALID_ARG;

        if (infoList && itemReader) {
            int startNum = -1;
            int endNum = -1;
            int totalCount = -1;

            retval = Helpers::ReadItemListData(map, startNum, endNum, totalCount);
            TryReturn(retval == E_SUCCESS, retval, GetErrorMessage(retval));

            BuyItemList<T> tmpInfoList;
            retval = tmpInfoList.Construct(startNum, endNum, totalCount);
            TryReturn(retval == E_SUCCESS, retval, "Constructing list: %s", GetErrorMessage(retval));

            for(int i = startNum; i <= endNum; ++i) {
                String prefix;
                prefix.Append(i);
                T item;
                retval = itemReader(map, &item, prefix);
                TryReturn(retval == E_SUCCESS, retval, GetErrorMessage(retval));

                tmpInfoList.Add(item);
            }

            retval = infoList->Construct(tmpInfoList);
            TryReturn(retval == E_SUCCESS, retval, GetErrorMessage(retval));
        }

        return retval;
    }

    String BuildKey(const String &prefix, const wchar_t *key)
    {
        String retval = prefix;
        result res = retval.Append(key);

        if (IsFailed(res)) {
            retval.Clear();
        }

        return retval;
    }

    result AddToMapWithPrefix(Collection::HashMap *map, const String &keyPrefix, const String &key, const String &value)
    {
        result retval = E_INVALID_ARG;

        if (map) {
            String * pKey = new String(keyPrefix);
            pKey->Append(key);
            String * pValue = new String(value);
            retval = map->Add(pKey, pValue);

            if (IsFailed(retval)) {
                delete pKey;
                delete pValue;
            }
        }

        return retval;
    }

    result ItemInformationFromHashMap(const Collection::IMap &map, BuyItemInformation *info, const String &keyPrefix = String())
    {
        result retval = E_INVALID_ARG;

        if (info) {
            String itemId;
            String itemGroupId;
            String itemName;
            String currencyUnit;
            bool unitPreceeds = false;
            bool hasPenny = false;
            double itemPrice = 0.0;
            String downloadUrl;
            String imageUrl;
            String desc;
            String res1;
            String res2;

            retval = BuyHashMapUtils::GetStringValue(&map, BuildKey(keyPrefix, BUY_APPCTRL_KEY_ITEM_ID), itemId);
            TryReturn(retval == E_SUCCESS, retval, "%S: %s", BUY_APPCTRL_KEY_ITEM_ID, GetErrorMessage(retval));

            retval = BuyHashMapUtils::GetStringValue(&map, BuildKey(keyPrefix, BUY_APPCTRL_KEY_ITEM_GROUP_ID), itemGroupId);
            TryReturn(retval == E_SUCCESS, retval, "%S: %s", BUY_APPCTRL_KEY_ITEM_GROUP_ID, GetErrorMessage(retval));

            retval = BuyHashMapUtils::GetStringValue(&map, BuildKey(keyPrefix, BUY_APPCTRL_KEY_ITEM_NAME), itemName);
            TryReturn(retval == E_SUCCESS, retval, "%S: %s", BUY_APPCTRL_KEY_ITEM_NAME, GetErrorMessage(retval));

            retval = BuyHashMapUtils::GetStringValue(&map, BuildKey(keyPrefix, BUY_APPCTRL_KEY_CURRENCY_UNIT), currencyUnit);
            TryReturn(retval == E_SUCCESS, retval, "%S: %s", BUY_APPCTRL_KEY_CURRENCY_UNIT, GetErrorMessage(retval));

            retval = BuyHashMapUtils::GetBoolValue(&map, BuildKey(keyPrefix, BUY_APPCTRL_KEY_UNIT_PRECEDES), unitPreceeds);
            TryReturn(retval == E_SUCCESS, retval, "%S: %s", BUY_APPCTRL_KEY_UNIT_PRECEDES, GetErrorMessage(retval));

            retval = BuyHashMapUtils::GetBoolValue(&map, BuildKey(keyPrefix, BUY_APPCTRL_KEY_HAS_PENNY), hasPenny);
            TryReturn(retval == E_SUCCESS, retval, "%S: %s", BUY_APPCTRL_KEY_HAS_PENNY, GetErrorMessage(retval));

            retval = BuyHashMapUtils::GetDoubleValue(&map, BuildKey(keyPrefix, BUY_APPCTRL_KEY_ITEM_PRICE), itemPrice);
            TryReturn(retval == E_SUCCESS, retval, "%S: %s", BUY_APPCTRL_KEY_ITEM_PRICE, GetErrorMessage(retval));

            retval = BuyHashMapUtils::GetStringValue(&map, BuildKey(keyPrefix, BUY_APPCTRL_KEY_ITEM_DOWNLOAD_URL), downloadUrl);
            TryReturn(retval == E_SUCCESS, retval, "%S: %s", BUY_APPCTRL_KEY_ITEM_DOWNLOAD_URL, GetErrorMessage(retval));

            BuyHashMapUtils::GetStringValue(&map, BuildKey(keyPrefix, BUY_APPCTRL_KEY_ITEM_IMAGE_URL), imageUrl);
            TryReturn(retval == E_SUCCESS, retval, "%S: %s", BUY_APPCTRL_KEY_ITEM_IMAGE_URL, GetErrorMessage(retval));

            BuyHashMapUtils::GetStringValue(&map, BuildKey(keyPrefix, BUY_APPCTRL_KEY_ITEM_DESCRIPTION), desc);
            TryReturn(retval == E_SUCCESS, retval, "%S: %s", BUY_APPCTRL_KEY_ITEM_DESCRIPTION, GetErrorMessage(retval));

            BuyHashMapUtils::GetStringValue(&map, BuildKey(keyPrefix, BUY_APPCTRL_KEY_RESERVED1), res1);
            TryReturn(retval == E_SUCCESS, retval, "%S: %s", BUY_APPCTRL_KEY_RESERVED1, GetErrorMessage(retval));

            BuyHashMapUtils::GetStringValue(&map, BuildKey(keyPrefix, BUY_APPCTRL_KEY_RESERVED2), res2);
            TryReturn(retval == E_SUCCESS, retval, "%S: %s", BUY_APPCTRL_KEY_RESERVED2, GetErrorMessage(retval));

            info->SetItemId(itemId);
            info->SetItemGroupId(itemGroupId);
            info->SetItemName(itemName);
            info->SetCurrencyUnit(currencyUnit);
            info->SetCurrencyUnitHasPenny(hasPenny);
            info->SetCurrencyUnitPrecedes(unitPreceeds);
            info->SetItemPrice(itemPrice);
            info->SetItemDownloadUrl(downloadUrl);
            info->SetItemImageUrl(imageUrl);
            info->SetItemDescription(desc);
            info->SetReserved1(res1);
            info->SetReserved2(res2);
        }

        return retval;
    }

    result PurchasedItemInformationFromHashMap(const Collection::IMap &map, BuyPurchasedItemInformation *info, const String &keyPrefix = String())
    {
        result retval = ItemInformationFromHashMap(map, info, keyPrefix);
        TryReturn(retval == E_SUCCESS, retval, GetErrorMessage(retval));

        String paymentId;
        String purchaseDateString;
        DateTime purchaseDate;

        retval = BuyHashMapUtils::GetStringValue(&map, BuildKey(keyPrefix, BUY_APPCTRL_KEY_PAYMENT_ID), paymentId);
        TryReturn(retval == E_SUCCESS, retval, "%S: %s", BUY_APPCTRL_KEY_PAYMENT_ID, GetErrorMessage(retval));

        retval = BuyHashMapUtils::GetStringValue(&map, BuildKey(keyPrefix, BUY_APPCTRL_KEY_PURCHASE_DATE), purchaseDateString);
        TryReturn(retval == E_SUCCESS, retval, "%S: %s", BUY_APPCTRL_KEY_PURCHASE_DATE, GetErrorMessage(retval));

        retval = DateTime::Parse(purchaseDateString, purchaseDate);
        TryReturn(retval == E_SUCCESS, retval, "%S conversion: %s", BUY_APPCTRL_KEY_PURCHASE_DATE, GetErrorMessage(retval));

        info->SetPaymentId(paymentId);
        info->SetPurchaseDate(purchaseDate);
        return retval;
    }
} // namespace Helpers

result PurchaseTicket::FromIMap(const Collection::IMap &map, BuyPurchaseTicket *ticket)
{
    result retval = E_INVALID_ARG;

    if (ticket) {
        String itemId;
        String purchaseId;
        String verifyUrl;
        String param1;
        String param2;
        String param3;

        retval = BuyHashMapUtils::GetStringValue(&map, BUY_APPCTRL_KEY_ITEM_ID, itemId);
        TryReturn(retval == E_SUCCESS, retval, "%S: %s", BUY_APPCTRL_KEY_ITEM_ID, GetErrorMessage(retval));

        retval = BuyHashMapUtils::GetStringValue(&map, BUY_APPCTRL_KEY_TICKET_PURCHASE_ID, purchaseId);
        TryReturn(retval == E_SUCCESS, retval, "%S: %s", BUY_APPCTRL_KEY_TICKET_PURCHASE_ID, GetErrorMessage(retval));

        retval = BuyHashMapUtils::GetStringValue(&map, BUY_APPCTRL_KEY_TICKET_VERIFY_URL, verifyUrl);
        TryReturn(retval == E_SUCCESS, retval, "%S: %s", BUY_APPCTRL_KEY_TICKET_VERIFY_URL, GetErrorMessage(retval));

        retval = BuyHashMapUtils::GetStringValue(&map, BUY_APPCTRL_KEY_TICKET_PARAM1, param1);
        TryReturn(retval == E_SUCCESS, retval, "%S: %s", BUY_APPCTRL_KEY_TICKET_PARAM1, GetErrorMessage(retval));

        retval = BuyHashMapUtils::GetStringValue(&map, BUY_APPCTRL_KEY_TICKET_PARAM2, param2);
        TryReturn(retval == E_SUCCESS, retval, "%S: %s", BUY_APPCTRL_KEY_TICKET_PARAM2, GetErrorMessage(retval));

        retval = BuyHashMapUtils::GetStringValue(&map, BUY_APPCTRL_KEY_TICKET_PARAM3, param3);
        TryReturn(retval == E_SUCCESS, retval, "%S: %s", BUY_APPCTRL_KEY_TICKET_PARAM3, GetErrorMessage(retval));

        ticket->SetItemId(itemId);
        ticket->SetPurchaseId(purchaseId);
        ticket->SetVerifyUrl(verifyUrl);
        ticket->SetParam1(param1);
        ticket->SetParam2(param2);
        ticket->SetParam3(param3);
    }

    return retval;
}

result PurchasedItemInformation::FromIMap(const Collection::IMap &map, BuyPurchasedItemInformation *info)
{
    return Helpers::PurchasedItemInformationFromHashMap(map, info);
}

result PurchasedItemInformationList::FromIMap(const Collection::IMap &map, BuyPurchasedItemInformationList *infoList)
{
    return Helpers::ReadItemList<BuyPurchasedItemInformation>(Helpers::PurchasedItemInformationFromHashMap, map, infoList);
}

result ItemInformationList::FromIMap(const Collection::IMap &map, BuyItemInformationList *infoList)
{
    return Helpers::ReadItemList<BuyItemInformation>(Helpers::ItemInformationFromHashMap, map, infoList);
}

} // namespace BuyAppControlSerialization
