﻿/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Xamarin.Forms; // we added this namespace as a reference

namespace VolumeController
{
    public class MainPage : ContentPage
    {
        private Dictionary<AudioVolumeTypeShare, Label> controllerLabelDictionary;
        private Dictionary<AudioVolumeTypeShare, Slider> controllerSliderDictionary;

        private bool _contentLoaded;
        void InitializeComponent()
        {
            if (_contentLoaded)
            {
                return;
            }

            _contentLoaded = true;

            // UI Logic Development
            #region UI Logic

            this.Content = new ScrollView()
            {
                HorizontalOptions = LayoutOptions.FillAndExpand,
                Orientation = ScrollOrientation.Vertical,
                Content = CreateAllControllers()
            };
            #endregion
        }


        public MainPage()
        {
            InitializeComponent();
            // created Initialization and user logic

            //  AudioManager.VolumeController.Changed += VolumeController_Changed;
        }

        /*  private void VolumeController_Changed(object sender, VolumeChangedEventArgs e)
          {
              UpdateControllerFromEvent(e.Type, e.Level);
          }*/





        private StackLayout CreateAllControllers()
        {
            controllerLabelDictionary = new Dictionary<AudioVolumeTypeShare, Label>();
            controllerSliderDictionary = new Dictionary<AudioVolumeTypeShare, Slider>();

            StackLayout controllers = new StackLayout();
            controllers.HorizontalOptions = LayoutOptions.FillAndExpand;
            controllers.Padding = new Thickness(0, 30, 0, 0);

            foreach (AudioVolumeTypeShare type in Enum.GetValues(typeof(AudioVolumeTypeShare)))
            {
                if (type != AudioVolumeTypeShare.None && type != AudioVolumeTypeShare.Voip && type != AudioVolumeTypeShare.Voice)
                {
                    controllers.Children.Add(CreateController(type));
                }
            }

            return controllers;
        }

        private StackLayout CreateController(AudioVolumeTypeShare type)
        {
            Label soundType = new Label();
            soundType.Text = type.ToString();
            soundType.FontSize = 50.0;
            soundType.TextColor = new Color(0, 0, 0);
            soundType.HorizontalOptions = LayoutOptions.StartAndExpand;

            Label soundLevel = new Label();

            soundLevel.Text = DependencyService.Get<IAudioManager>().LevelType(type).ToString();
            soundLevel.FontSize = 50.0;
            soundLevel.HorizontalOptions = LayoutOptions.EndAndExpand;
            soundLevel.TextColor = new Color(0, 0, 0);
            soundLevel.LineBreakMode = LineBreakMode.NoWrap;
            controllerLabelDictionary.Add(type, soundLevel);

            Layout typeAndLevel = new StackLayout()
            {
                Orientation = StackOrientation.Horizontal,
                HorizontalOptions = LayoutOptions.FillAndExpand,
                Padding = new Thickness(30, 0, 30, 30),
                Children =
                {
                    soundType,
                    soundLevel
                }
            };

            Slider slider = new Slider();
            slider.HorizontalOptions = LayoutOptions.FillAndExpand;
            slider.Minimum = 0;
            slider.Maximum = DependencyService.Get<IAudioManager>().MaxLevel(type);
            slider.Value = DependencyService.Get<IAudioManager>().LevelType(type);
            switch (type)
            {
                case AudioVolumeTypeShare.System:
                    slider.ValueChanged += SystemSliderDragged;
                    break;
                case AudioVolumeTypeShare.Notification:
                    slider.ValueChanged += NotificationSliderDragged;
                    break;
                case AudioVolumeTypeShare.Alarm:
                    slider.ValueChanged += AlarmSliderDragged;
                    break;
                case AudioVolumeTypeShare.Ringtone:
                    slider.ValueChanged += RingtoneSliderDragged;
                    break;
                case AudioVolumeTypeShare.Media:
                    slider.ValueChanged += MediaSliderDragged;
                    break;
                case AudioVolumeTypeShare.Call:
                    slider.ValueChanged += CallSliderDragged;
                    break;
                default:
                    break;
            }

            controllerSliderDictionary.Add(type, slider);

            StackLayout controller = new StackLayout();
            controller.HorizontalOptions = LayoutOptions.FillAndExpand;
            controller.VerticalOptions = LayoutOptions.StartAndExpand;
            controller.Children.Add(typeAndLevel);
            controller.Children.Add(slider);

            return controller;
        }

        private void SystemSliderDragged(object sender, EventArgs e)
        {
            UpdateVolumeAndLabelFromDrag(AudioVolumeTypeShare.System, sender);
        }

        private void NotificationSliderDragged(object sender, EventArgs e)
        {
            UpdateVolumeAndLabelFromDrag(AudioVolumeTypeShare.Notification, sender);
        }

        private void AlarmSliderDragged(object sender, EventArgs e)
        {
            UpdateVolumeAndLabelFromDrag(AudioVolumeTypeShare.Alarm, sender);
        }

        private void RingtoneSliderDragged(object sender, EventArgs e)
        {
            UpdateVolumeAndLabelFromDrag(AudioVolumeTypeShare.Ringtone, sender);
        }

        private void MediaSliderDragged(object sender, EventArgs e)
        {
            UpdateVolumeAndLabelFromDrag(AudioVolumeTypeShare.Media, sender);
        }

        private void CallSliderDragged(object sender, EventArgs e)
        {
            UpdateVolumeAndLabelFromDrag(AudioVolumeTypeShare.Call, sender);
        }

        private void UpdateVolumeAndLabelFromDrag(AudioVolumeTypeShare type, object sliderObject)
        {
            Slider slider = (Slider)sliderObject;
            slider.Value = Math.Round(slider.Value);
            DependencyService.Get<IAudioManager>().ApplyAudioType(type, (int)slider.Value);

            Label sliderLabel = controllerLabelDictionary[type];
            sliderLabel.Text = slider.Value.ToString();
            // Tizen.Log.Info("volume-app", "drag changed label for " + type + " to " + slider.Value);
        }

        private void UpdateControllerFromEvent(AudioVolumeTypeShare type, uint level)
        {
            Slider slider = controllerSliderDictionary[type];
            slider.Value = level;

            Label sliderLabel = controllerLabelDictionary[type];
            sliderLabel.Text = level.ToString();
            //Tizen.Log.Info("volume-app", "event changed controller for " + type + " to " + level);
        }

    }
}