#include "usinglame.h"

void
init_base_player(appdata_s *ad)
{
   int error_code = 0;
   error_code = player_create(&ad->player);
   if (error_code != PLAYER_ERROR_NONE)
      dlog_print(DLOG_ERROR, LOG_TAG, "failed to create");
}

void
release_base_player(appdata_s *ad)
{
	if(player_unprepare(ad->player) == PLAYER_ERROR_NONE)
	{
		player_destroy(ad->player);
		ad->player = NULL;
	}
}

void
play_audio_file(appdata_s *ad)
{
	if(player_set_uri(ad->player, get_file_path()) != PLAYER_ERROR_NONE)
	{
		LOGE("FILED TO SET FILE URI");
	}

	if(player_prepare(ad->player) != PLAYER_ERROR_NONE)
	{
		LOGE("FILED TO PREPARE PLAYER");
	}

	player_set_play_position(ad->player, 0, true, _seek_completed_cb, ad);
}

static void
win_delete_request_cb(void *data , Evas_Object *obj , void *event_info)
{
	ui_app_exit();
}

static void
win_back_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = data;
	elm_win_lower(ad->win);
}

void set_file_name(appdata_s* ad)
{
	if(file_exist())
	{
		char file_name_text[PATH_MAX];
		snprintf(file_name_text, sizeof(file_name_text), "<color=#FFFFFF><font_size=24><align=center>%s</align></font_size></color>", get_file_path());
		elm_object_text_set(ad->filename, file_name_text);
		return;
	}
	elm_object_text_set(ad->filename, "");
}

void
set_timer(appdata_s* ad, int minutes, int seconds)
{
	char* color;
	char time_text[PATH_MAX];

	if(!ad->recording)
	{
		color = "#FFFFFF";
	}
	else
	{
		color = "#FF0000";
	}

	snprintf(time_text, sizeof(time_text), "<color=%s><font_size=120><align=center>%02d:%02d</align></font_size></color>", color, minutes, seconds);
	elm_object_text_set(ad->time, time_text);
}

float
get_player_progress(appdata_s* ad)
{
	int position;
	int duration;

	player_get_play_position(ad->player, &position);
	player_get_duration(ad->player, &duration);

	if(duration <= 0)
	{
		return 0;
	}
	return round(((float)position/(float)duration)*10)/10;
}

void
set_player_progress(appdata_s* ad, float progress)
{
	evas_object_resize(ad->progress, progress * 0.8 * screen_width, 5);
}

void
stop_playing(appdata_s* ad)
{
	player_stop(ad->player);
	player_unprepare(ad->player);
	play_button_state_set(PAUSE, ad);
}

Eina_Bool
timer_cb(void *data)
{
	appdata_s *ad = data;
	float progress;

	if(!ad->recording && !ad->playing)
	{
		return false;
	}

	int minutes =  (time(NULL) - ad->start_rec_timestamp) / 60;
	int seconds =  (time(NULL) - ad->start_rec_timestamp) % 60;

	set_timer(ad, minutes, seconds);

	if(!ad->recording)
	{
		progress = get_player_progress(ad);
		set_player_progress(ad, get_player_progress(ad));

		if(progress == 1.0)
		{
			stop_playing(ad);
			return false;
		}
	}

	return true;
}

void
_seek_completed_cb(void *data)
{
	appdata_s *ad = data;
	if(player_start(ad->player) != PLAYER_ERROR_NONE)
	{
		LOGE("FILED TO PLAY");
	}

	ecore_timer_add (TIMER_REFRESH_FREQUENCY, timer_cb, ad);
}

void
play_button_state_set(play_button_state state,appdata_s* ad)
{
	if(state == PLAY)
	{
		ad->playing = true;
		elm_object_text_set(ad->play_button, "<font_size=30>ll</font_size>");
	}
	if(state == PAUSE)
	{
		ad->playing = false;
		elm_object_text_set(ad->play_button, "<font_size=30>></font_size>");
	}
}

void
_record_button_clicked(void *data, Evas *e, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = data;

	ad->playing = false;
	if(ad->recording || !init_encoder(SAMPLE_RATE, CHANNELS, BITRATE, QUALITY))
	{
		LOGE("FILED TO INIT ENCODERS");
		return;
	}

	if(!ad->recording)
	{
		ad->recording = init_audio_IO();
		set_file_name(ad);
	}

	ad->start_rec_timestamp = time(NULL);
	ecore_timer_add (TIMER_REFRESH_FREQUENCY, timer_cb, data);

	set_timer(ad, 0, 0);
	play_button_state_set(PAUSE,ad);
}

void
stop_recording(appdata_s* ad)
{
	if(!ad->recording)
	{
		LOGE("NO RECORDINGS TO STOP");
		return;
	}

	if(!close_audio_IO())
	{
		LOGE("FILED TO STOP AUDIO IO");
		return;
	}
	ad->recording = false;

	flush_mp3_buffer();
}

void
_stop_button_clicked(void *data, Evas *e, Evas_Object *obj, void *event_info)
{
	stop_recording((appdata_s*) data);
}

void
_cancel_button_clicked(void *data, Evas *e, Evas_Object *obj, void *event_info)
{
	if(((appdata_s*)data)->recording)
	{
		stop_recording((appdata_s*) data);

		// remove the audio file
		int ret = remove(get_file_path());
		if(ret != 0){
			LOGE("UNABLE TO DELETE AUDIO FILE");
		}
		set_timer((appdata_s*) data, 0, 0);
		set_file_name((appdata_s*) data);
	}
}

void
_play_button_clicked(void *data, Evas *e, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = data;

	if(!ad->recording && file_exist())
	{
		if(ad->playing)
		{
			stop_playing(ad);
		}
		else
		{
			play_button_state_set(PLAY, ad);
			ad->start_rec_timestamp = time(NULL);
			play_audio_file(ad);
		}
		set_timer((appdata_s*) data, 0, 0);
	}
}

static void
create_base_gui(appdata_s *ad)
{
	/* Window */
	ad->win = elm_win_util_standard_add(PACKAGE, PACKAGE);
	elm_win_autodel_set(ad->win, EINA_TRUE);

	Evas_Object *background = elm_bg_add(ad->win);
	elm_win_resize_object_add(ad->win, background);
	elm_bg_color_set(background, 0, 0, 0);
	evas_object_show(background);

	evas_object_smart_callback_add(ad->win, "delete,request", win_delete_request_cb, NULL);
	eext_object_event_callback_add(ad->win, EEXT_CALLBACK_BACK, win_back_cb, ad);

	elm_win_screen_size_get (ad->win, NULL, NULL, &screen_width, &screen_height);

	/* Rectangle */
	Evas_Object *title_background = evas_object_rectangle_add(evas_object_evas_get(ad->win));
	evas_object_color_set(title_background, 0, 132, 202, 255);
	evas_object_resize(title_background, screen_width, 0.1*screen_height);
	evas_object_move(title_background, 0, 0);
	evas_object_show(title_background);

	/* Label*/
	Evas_Object *title_text  = elm_label_add(ad->win);
	elm_object_text_set(title_text, "<color=#FFFFFF><font_size=40><align=center>VoiceRecorder</align></font_size></color>");
	evas_object_size_hint_weight_set(title_text, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_resize(title_text, screen_width, 0.075*screen_height); // covers full canvas
	evas_object_move(title_text, 0, 0.02*screen_height);
	evas_object_show(title_text);

	/* Filename*/
	ad->filename  = elm_label_add(ad->win);
	evas_object_size_hint_weight_set(ad->filename, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_resize(ad->filename, screen_width, 0.075*screen_height);
	evas_object_move(ad->filename, 0, 0.5*screen_height);
	evas_object_show(ad->filename);

	/* time*/
	ad->time  = elm_label_add(ad->win);
	elm_object_text_set(ad->time, "<color=#FFFFFF><font_size=120><align=center>00:00</align></font_size></color>");
	evas_object_size_hint_weight_set(ad->time, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_resize(ad->time, screen_width, 120); // covers full canvas
	evas_object_move(ad->time, 0, 0.35*screen_height);
	evas_object_show(ad->time);

	/* Record button */
	ad->record_button = elm_button_add(ad->win);
	elm_object_text_set(ad->record_button, "<font_size=30>Record</font_size>");
	evas_object_resize(ad->record_button, 0.47 * screen_width, 0.28 * screen_height);
	evas_object_move(ad->record_button, 0.02 * screen_width, 0.71 * screen_height);
	evas_object_show(ad->record_button);
	evas_object_event_callback_add(ad->record_button, EVAS_CALLBACK_MOUSE_DOWN, _record_button_clicked, ad);

	/* Stop button */
	ad->stop_button = elm_button_add(ad->win);
	elm_object_text_set(ad->stop_button, "<font_size=30>Stop</font_size>");
	evas_object_resize(ad->stop_button, 0.47 * screen_width, 0.135 * screen_height);
	evas_object_move(ad->stop_button, 0.51 * screen_width, 0.71 * screen_height);
	evas_object_show(ad->stop_button);
	evas_object_event_callback_add(ad->stop_button, EVAS_CALLBACK_MOUSE_DOWN, _stop_button_clicked, ad);

	/* Cancel button */
	ad->cancel_button = elm_button_add(ad->win);
	elm_object_text_set(ad->cancel_button, "<font_size=30>Cancel</font_size>");
	evas_object_resize(ad->cancel_button, 0.47 * screen_width, 0.135 * screen_height);
	evas_object_move(ad->cancel_button, 0.51 * screen_width, 0.855 * screen_height);
	evas_object_show(ad->cancel_button);
	evas_object_event_callback_add(ad->cancel_button, EVAS_CALLBACK_MOUSE_DOWN, _cancel_button_clicked, ad);

	/* timeline */
	Evas_Object *timeline = evas_object_rectangle_add(evas_object_evas_get(ad->win));
	evas_object_color_set(timeline, 200, 200, 200, 255);
	evas_object_resize(timeline, 0.8 * screen_width, 5);
	evas_object_move(timeline, 0.16*screen_width, 0.605 * screen_height);
	evas_object_show(timeline);

	/* playing progress */
	ad->progress = evas_object_rectangle_add(evas_object_evas_get(ad->win));
	evas_object_color_set(ad->progress, 255, 0, 0, 255);
	evas_object_move(ad->progress, 0.16*screen_width, 0.605 * screen_height);
	evas_object_show(ad->progress);

	/* play button */
	ad->play_button = elm_button_add(ad->win);
	elm_object_text_set(ad->play_button, "<font_size=30>></font_size>");
	evas_object_resize(ad->play_button, 0.11 * screen_width, 0.067 * screen_height);
	evas_object_move(ad->play_button, 0.05 * screen_width, 0.573 * screen_height);
	evas_object_show(ad->play_button);
	evas_object_event_callback_add(ad->play_button, EVAS_CALLBACK_MOUSE_DOWN, _play_button_clicked, ad);

	/* Show window after base gui is set up */
	evas_object_show(ad->win);
}

static bool
app_create(void *data)
{
	appdata_s *ad = data;

	init_base_player(ad);

	get_external_storage();
	create_base_gui(ad);

	return true;
}

static void
app_control(app_control_h app_control, void *data)
{
	/* Handle the launch request. */
}

static void
app_pause(void *data)
{
	/* Take necessary actions when application becomes invisible. */
}

static void
app_resume(void *data)
{
	/* Take necessary actions when application becomes visible. */
}

static void
app_terminate(void *data)
{
	/* Release all resources. */
	release_base_player( (appdata_s*)data);
	free_buffers();
}

static void
ui_app_lang_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LANGUAGE_CHANGED*/
	char *locale = NULL;
	system_settings_get_value_string(SYSTEM_SETTINGS_KEY_LOCALE_LANGUAGE, &locale);
	elm_language_set(locale);
	free(locale);
	return;
}

static void
ui_app_orient_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_DEVICE_ORIENTATION_CHANGED*/
	return;
}

static void
ui_app_region_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_REGION_FORMAT_CHANGED*/
}

static void
ui_app_low_battery(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LOW_BATTERY*/
}

static void
ui_app_low_memory(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LOW_MEMORY*/
}

int
main(int argc, char *argv[])
{
	appdata_s ad = {0,};
	int ret = 0;

	ui_app_lifecycle_callback_s event_callback = {0,};
	app_event_handler_h handlers[5] = {NULL, };

	event_callback.create = app_create;
	event_callback.terminate = app_terminate;
	event_callback.pause = app_pause;
	event_callback.resume = app_resume;
	event_callback.app_control = app_control;

	ui_app_add_event_handler(&handlers[APP_EVENT_LOW_BATTERY], APP_EVENT_LOW_BATTERY, ui_app_low_battery, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_LOW_MEMORY], APP_EVENT_LOW_MEMORY, ui_app_low_memory, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_DEVICE_ORIENTATION_CHANGED], APP_EVENT_DEVICE_ORIENTATION_CHANGED, ui_app_orient_changed, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_LANGUAGE_CHANGED], APP_EVENT_LANGUAGE_CHANGED, ui_app_lang_changed, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_REGION_FORMAT_CHANGED], APP_EVENT_REGION_FORMAT_CHANGED, ui_app_region_changed, &ad);
	ui_app_remove_event_handler(handlers[APP_EVENT_LOW_MEMORY]);

	ret = ui_app_main(argc, argv, &event_callback, &ad);
	if (ret != APP_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "app_main() is failed. err = %d", ret);
	}

	return ret;
}
