/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Xamarin.Forms;

namespace UIComponents
{
    /// <summary>
    /// a TabbedPage contains various NamedColorPage
    /// </summary>
    class TabbedPageDemoPage : TabbedPage
    {
        /// <summary>
        /// The constructor of TabbedPageDemoPage class
        /// </summary>
        public TabbedPageDemoPage()
        {
            this.Title = "One ContentPage";

            this.ItemsSource = new NamedColor[]
            {
                new NamedColor("Red", Color.Red),
                new NamedColor("Yellow", Color.Yellow),
                new NamedColor("Green", Color.Green),
                new NamedColor("Aqua", Color.Aqua),
                new NamedColor("Blue", Color.Blue),
                new NamedColor("Purple", Color.Purple)
            };

            this.ItemTemplate = new DataTemplate(() =>
            {
                return new NamedColorPage();
            });
        }
    }

    /// <summary>
    /// Define NamedColor class
    /// </summary>
    class NamedColor
    {
        public NamedColor(string name, Color color)
        {
            this.Name = name;
            this.Color = color;
        }

        public string Name { private set; get; }

        public Color Color { private set; get; }

        public override string ToString()
        {
            return Name;
        }
    }

    /// <summary>
    /// a ContentPage to show colored box
    /// </summary>
    class NamedColorPage : ContentPage
    {
        public NamedColorPage()
        {
            /// <summary>
            /// This binding is necessary to label the tabs in the TabbedPage.
            /// </summary>
            this.SetBinding(ContentPage.TitleProperty, "Name");

            BoxView boxView = new BoxView
            {
                WidthRequest = 100,
                HeightRequest = 100,
                HorizontalOptions = LayoutOptions.Center
            };
            boxView.SetBinding(BoxView.ColorProperty, "Color");
            this.Content = boxView;
        }
    }

    /// <summary>
    /// Two ContentPage added as children of TabbedPage
    /// </summary>
    class TabbedPageDemoPage2 : TabbedPage
    {
        public TabbedPageDemoPage2()
        {
            this.Title = "Two ContentPage";
            this.Children.Add(new ContentPage
            {
                Title = "Blue",
                Content = new BoxView
                {
                    Color = Color.Blue,
                    HeightRequest = 100f,
                    VerticalOptions = LayoutOptions.Center
                },
            });

            this.Children.Add(new ContentPage
            {
                Title = "Blue and Red",
                Content = new StackLayout
                {
                    Children =
                    {
                    new BoxView { Color = Color.Blue },
                    new BoxView { Color = Color.Red}
                }
                }
            });
        }
    }

    /// <summary>
    /// Two samples for TabbedPage usage
    /// </summary>
    public class TabbedPageView : ContentPage
    {
        public TabbedPageView()
        {
            /// <summary>
            /// Set title of this page.
            /// </summary>
            Title = "TabbedPage";

            var list = new ListView();
            list.ItemsSource = new List<String>()
            {
                "One ContentPage sample",
                "Two ContentPage sample"
            };

            /// <summary>
            /// List item event handler
            /// </summary>
            list.ItemTapped += async (sender, e) =>
            {
                if (e.Item.ToString() == "One ContentPage sample")
                {
                    await Navigation.PushAsync(new TabbedPageDemoPage());
                }
                else if (e.Item.ToString() == "Two ContentPage sample")
                {
                    await Navigation.PushAsync(new TabbedPageDemoPage2());
                }
            };

            Content = list;
        }
    }
}

