/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Xamarin.Forms;

namespace UIComponents
{
    /// <summary>
    /// A simple view contains various buttons
    /// </summary>
    public class ButtonView : ContentPage
    {
        public ButtonView()
        {
            /// <summary>
            /// Set title of this page.
            /// </summary>
            Title = "Button";

            string imgpath = "T05_capture_call.png";

            int count = 0;
            /// <summary>
            /// Init a color array
            /// </summary>
            Color[] colors = new Color[]
            {
                new Color(0, 0, 0),
                new Color(255, 0, 0),
                new Color(0, 255, 0),
                new Color(0, 0, 255),
                new Color(255, 255, 0),
                new Color(255, 0, 255),
                new Color(0, 255, 255)
            };

            /// <summary>
            /// User can change button color by clicking it
            /// </summary>
            Button button = new Button
            {
                BorderWidth = 3,
                HorizontalOptions = LayoutOptions.CenterAndExpand,
                VerticalOptions = LayoutOptions.CenterAndExpand,
                Text = "Color Change",
                TextColor = new Color(0, 0, 255),
            };

            /// <summary>
            /// button click event handler, to change button text color
            /// </summary>
            button.Clicked += (sender, e) =>
            {
                button.TextColor = colors[count % 7];
                count++;
            };

            StackLayout layout = new StackLayout
            {
                Children =
                {
                    button,

                    /// <summary>
                    /// A button with specified width is placed in the center of screen
                    /// </summary>
                    new Button
                    {
                        WidthRequest = 300,
                        HorizontalOptions = LayoutOptions.Center,
                        VerticalOptions = LayoutOptions.CenterAndExpand,
                        Text = "Center",
                    },

                    /// <summary>
                    /// A button with specified width is placed in the left of screen
                    /// </summary>
                    new Button
                    {
                        WidthRequest = 300,
                        HorizontalOptions = LayoutOptions.Start,
                        VerticalOptions = LayoutOptions.CenterAndExpand,
                        Text = "Start",
                    },

                    /// <summary>
                    /// A button with specified width is placed in the right of screen
                    /// </summary>
                    new Button
                    {
                        WidthRequest = 300,
                        HorizontalOptions = LayoutOptions.End,
                        VerticalOptions = LayoutOptions.CenterAndExpand,
                        Text = "End",
                    },

                    /// <summary>
                    /// A button is placed in the center of screen and expands its width according to the text
                    /// </summary>
                    new Button
                    {
                        HorizontalOptions = LayoutOptions.CenterAndExpand,
                        VerticalOptions = LayoutOptions.CenterAndExpand,
                        Text = "CenterAndExpand",
                    },

                    /// <summary>
                    /// A button is placed in the left of screen and expands its width according to the text
                    /// </summary>
                    new Button
                    {
                        HorizontalOptions = LayoutOptions.StartAndExpand,
                        VerticalOptions = LayoutOptions.CenterAndExpand,
                        Text = "StartAndExpand",
                    },

                    /// <summary>
                    /// A button is placed in the right of screen and expands its width according to the text
                    /// </summary>
                    new Button
                    {
                        HorizontalOptions = LayoutOptions.EndAndExpand,
                        VerticalOptions = LayoutOptions.CenterAndExpand,
                        Text = "EndAndExpand",
                    },

                    /// <summary>
                    /// A button fills the screen in vertical
                    /// </summary>
                    new Button
                    {
                        HorizontalOptions = LayoutOptions.Fill,
                        VerticalOptions = LayoutOptions.CenterAndExpand,
                        Text = "Fill",
                    },

                    /// <summary>
                    /// A button fills the screen and extends its width in vertical
                    /// </summary>
                    new Button
                    {
                        HorizontalOptions = LayoutOptions.FillAndExpand,
                        VerticalOptions = LayoutOptions.CenterAndExpand,
                        Text = "FillAndExpand",
                    },

                    /// <summary>
                    /// A button is disabled
                    /// </summary>
                    new Button
                    {
                        HorizontalOptions = LayoutOptions.CenterAndExpand,
                        VerticalOptions = LayoutOptions.CenterAndExpand,
                        Text = "Disabled",
                        IsEnabled = false,
                    },

                    /// <summary>
                    /// A button which has image and text
                    /// </summary>
                    new Button
                    {
                        HorizontalOptions = LayoutOptions.CenterAndExpand,
                        VerticalOptions = LayoutOptions.CenterAndExpand,
                        Image = new FileImageSource
                        {
                            File = imgpath,
                        },
                    },
                }
            };

            /// <summary>
            /// Add a ScrollView if content is too long
            /// </summary>
            ScrollView scrollView = new ScrollView
            {
                HeightRequest = 200,
                VerticalOptions = LayoutOptions.FillAndExpand,
                HorizontalOptions = LayoutOptions.FillAndExpand,
                Orientation = ScrollOrientation.Vertical,
                Content = layout,
            };

            /// <summary>
            /// Set scrollView as the content.
            /// </summary>
            this.Content = scrollView;
        }
     }
}
