(function () {
    var wordsList = [
        'tizen',
        'conference',
        'wifi direct',
        'phone',
        'mobile',
        'application',
        'java script',
        'platform',
        'samsung',
        'computer',
        'internet',
        'database',
        'html5',
        'css',
        'input',
        'widget',
        'tag',
        'header',
        'textarea',
        'button',
        'canvas',
        'api',
        'connection',
        'protocole',
        'communication',
        'hybrid',
        'message',
        'error',
        'port',
        'method',
        'function',
        'interface',
        'native',
        'web',
        'data',
        'library',
        'game',
        'type race',
        'debug',
        'demonstration',
        'development',
        'connect',
        'disconnect',
        'group',
        'user interface',
        'server',
        'client',
        'host',
        'node',
        'drag and drop',
        'boolean',
        'float',
        'integer',
        'broadcast',
        'ip address',
        'uppercase',
        'lowercase',
        'click',
        'listener',
        'callback'
    ];

    var Game = Class.extend({
        currentScreenName: null,

        activated: false,
        loading: false,
        leaving: false,
        countingDown: false,
        winner: null,

        isServer: false,
        wordsIndexes: null,
        wordsCount: 6,
        started: false,
        gamesList: null,

        ui: null,
        data: null,

        init: function () {
            this.gamesList = [];

            this.ui = {
                screens: {
                    start: document.getElementById('start-screen'),
                    menu: document.getElementById('menu-screen'),
                    games: document.getElementById('games-screen'),
                    game: document.getElementById('game-screen')
                },
                loading: document.getElementById('loading'),
                createGame: document.getElementById('create-game'),
                leaveGame: document.getElementById('leave-game'),
                joinGame: document.getElementById('join-game'),
                startGame: document.getElementById('start-game'),
                gamesList: document.getElementById('games-list'),
                countDown: document.getElementById('count-down'),
                winner: document.getElementById('winner'),
                loser: document.getElementById('loser'),
                waitingForOpp: document.getElementById('waiting-for-opp'),
                you: {
                    wordImage: document.querySelector('#you img'),
                    textField: document.querySelector('#you input'),
                    context: document.querySelector('#you canvas').getContext('2d'),
                    bar: document.querySelector('#you .bar')
                },
                opp: {
                    wordImage: document.querySelector('#opp img'),
                    textField: document.querySelector('#opp input'),
                    context: document.querySelector('#opp canvas').getContext('2d'),
                    bar: document.querySelector('#opp .bar')
                }
            };

            // UI methods.
            this.ui.openScreen = function (screenName) {
                if (!this.ui.screens[screenName] || this.currentScreenName === screenName) return;

                _.each(this.ui.screens, function (screen) {
                    screen.setAttribute('hidden', 'hidden');
                }.bind(this));

                this.ui.screens[screenName].removeAttribute('hidden');
                this.ui.screens[screenName].focus();
                this.currentScreenName = screenName;

                if (screenName === 'menu') {
                    this.resetPlayersData(true);
                }
            }.bind(this);

            this.ui.closeScreen = function () {
                switch (this.currentScreenName) {
                    case 'game':
                        this.leave();
                        break;
                    case 'games':
                        this.ui.openScreen('menu');
                        break;
                    case 'menu':
                        tizen.application.getCurrentApplication().exit();
                        break;
                }
            }.bind(this);

            this.ui.show = function (element) {
                if (!this.ui[element]) return;

                this.ui[element].removeAttribute('hidden');
            }.bind(this);
            
            this.ui.hide = function (element) {
                if (!this.ui[element]) return;

                this.ui[element].setAttribute('hidden', 'hidden');
            }.bind(this);

            this.ui.enable = function (element) {
                if (!element) return;

                if (_.isString(element)) {
                    if (!this.ui[element]) return;
                    this.ui[element].removeAttribute('disabled');
                } else if (_.isObject(element)) {
                    element.removeAttribute('disabled');
                }
            }.bind(this);
            
            this.ui.disable = function (element) {
                if (!element) return;

                if (_.isString(element)) {
                    if (!this.ui[element]) return;
                    this.ui[element].setAttribute('disabled', 'disabled');
                } else if (_.isObject(element)) {
                    element.setAttribute('disabled', 'disabled');
                }
            }.bind(this);

            // Players's data.
            this.data = {
                you: {
                    wordIndex: 0,
                    typedWord: '',
                    finished: false,
                    started: false,
                    connected: false,
                    winner: false
                },
                opp: {
                    wordIndex: 0,
                    typedWord: '',
                    finished: false,
                    started: false,
                    connected: false,
                    winner: false
                }
            };

            window.addEventListener('tizenhwkey', function (e) {
                if (e.keyName === 'back') {
                    this.ui.closeScreen();
                }
            }.bind(this));

            this.ui.you.textField.addEventListener('keydown', function (e) {
                if (!this.data.you.started) {
                    e.preventDefault();
                }
            }.bind(this));

            this.ui.you.textField.addEventListener('keyup', function (e) {
                var typedWord, wordIndex, currWord;

                if (this.data.you.started) {
                    typedWord = e.target.value;
                    this.setPlayerData('you', 'typedWord', typedWord);

                    wordIndex = this.getPlayerData('you', 'wordIndex');
                    currWord = this.getPlayerData('you', 'currWord');

                    if (typedWord === currWord) {
                        wordIndex++;

                        // Go to the next word and clear input field.
                        this.setPlayerData('you', 'wordIndex', wordIndex);
                        this.setPlayerData('you', 'typedWord', '');

                        // Update word's image.
                        this.printWord(this.getPlayerData('you', 'currWord'), 'you');

                        // Send data to the opponent.
                        this.sendPlayerData('wordIndex');

                        // Finished.
                        if (wordIndex === this.wordsCount) {
                            this.setPlayerData('you', 'finished', true);
                            this.sendPlayerData('finished');

                            this.finish(true);
                        }
                    }

                    if (this.hasError('you')) {
                        this.playSound('error');
                    }

                    this.sendPlayerData('typedWord');
                    this.update();
                }
            }.bind(this));

            this.ui.createGame.addEventListener('click', function (e) {
                this.create();
            }.bind(this));

            this.ui.joinGame.addEventListener('click', function (e) {
                this.scan();
            }.bind(this));

            this.ui.startGame.addEventListener('click', function (e) {
                this.start();
            }.bind(this));

            this.ui.winner.addEventListener('click', function (e) {
                this.ui.hide('winner');
            }.bind(this));

            this.ui.loser.addEventListener('click', function (e) {
                this.ui.hide('loser');
            }.bind(this));

            this.ui.countDown.addEventListener('webkitAnimationEnd', this.onCountDownFinished.bind(this));

            tizen.wifidirect.addEventListener('groupCreated', this.onCreateSuccess.bind(this));
            tizen.wifidirect.addEventListener('groupLeft', this.onLeaveSuccess.bind(this));
            tizen.wifidirect.addEventListener('activated', this.onActivateSuccess.bind(this));
            tizen.wifidirect.addEventListener('scanCompleted', this.onScanSuccess.bind(this));
            tizen.wifidirect.addEventListener('connected', this.onConnectSuccess.bind(this));
            tizen.wifidirect.addEventListener('disconnected', this.onDisconnectSuccess.bind(this));
            tizen.wifidirect.addEventListener('socketInitialized', this.onInitSocketSuccess.bind(this));
            tizen.wifidirect.addEventListener('messageReceived', this.onDataReceived.bind(this));

            tizen.wifidirect.init(this.onInit.bind(this));
        },

        playSound: function (soundName) {
            this.sounds = this.sounds || {};
            if (soundName === 'error') {
                if (!this.sounds[soundName]) {
                    this.sounds[soundName] = new Audio('/sounds/' + soundName + '.mp3');
                }
                this.sounds[soundName].play();
            }
        },

        printWord: function (text, player) {
            text = text || 'WINNER!';
            if (player !== 'you' && player !== 'opp') return;

            var context = this.ui[player].context;

            var font = {
              color: '#ffffff',
              weight: 'bold',
              size: 80,
              family: 'Arial'
            };

            context.save();

            context.font = font.weight + ' ' + font.size + 'px ' + font.family;
            context.clearRect(0, 0, 720, 150);
            context.shadowBlur = 4;
            context.shadowColor = '#000000';
            context.shadowOffsetY = 2;
            context.fillStyle = font.color;
            context.fillText(text, (720 - context.measureText(text).width) / 2, (150 / 2) + (font.size / 2.5));

            context.restore();
        },

        setPlayerData: function (player, key, value) {
            if (player !== 'you' && player !== 'opp') return;

            this.data[player][key] = value;
        },

        getPlayerData: function (player, key) {
            if (player !== 'you' && player !== 'opp') return;

            switch (key) {
            case 'currWord':
                return wordsList[this.wordsIndexes[this.getPlayerData(player, 'wordIndex')]];
            default:
                return this.data[player][key];
            }
        },

        hasError: function (player) {
            var typedWord, currWord;

            typedWord = this.getPlayerData(player, 'typedWord');
            currWord = this.getPlayerData(player, 'currWord');

            // Check if there are any errors in the typed word.
            for (var i = 0; i < typedWord.length; i++) {
                if (typedWord[i] !== currWord[i]) {
                    return true;
                }
            }

            return false;
        },

        updateOne: function (player) {
            var typedWord = this.getPlayerData(player, 'typedWord'),
                wordIndex = this.getPlayerData(player, 'wordIndex');

            if (this.hasError(player)) {
                this.ui[player].textField.classList.add('error');
            } else {
                this.ui[player].textField.classList.remove('error');
            }

            // Update user's text field if it has different value than it should.
            if (this.ui[player].textField.value !== typedWord) {
                this.ui[player].textField.value = typedWord;
            }

            this.ui[player].bar.style.width = (wordIndex / this.wordsCount * 100) + '%';
        },

        update: function () {
            if (this.loading) {
                this.ui.show('loading');
            } else {
                this.ui.hide('loading');
            }

            // YOU CONNECTED / DISCONNECTED - BEG
            if (this.getPlayerData('you', 'connected')) {
                this.ui.show('waitingForOpp');

                // OPPONENT CONNECTED / DISCONNECTED - BEG
                if (this.getPlayerData('opp', 'connected')) {
                    this.ui.hide('waitingForOpp');
                    this.ui.enable('startGame');

                    // YOU STARTED - BEG
                    if (this.getPlayerData('you', 'started')) {
                        this.ui.disable('startGame');

                        // OPPONENT STARTED - BEG
                        if (this.getPlayerData('opp', 'started')) {
                            if (this.countingDown) {
                                this.ui.show('countDown');
                                this.ui.disable(this.ui.you.textField);
                            } else {
                                this.ui.hide('countDown');
                                this.ui.enable(this.ui.you.textField);
                                this.ui.you.textField.focus();

                                // BOTH PLAYERS STARTED - BEG
                                this.updateOne('you');
                                this.updateOne('opp');
                                // BOTH PLAYERS STARTED - END
                            }
                            this.ui.hide('waitingForOpp');
                        } else {
                            this.ui.show('waitingForOpp');
                        }
                        // OPPONENT STARTED - END
                    } else {
                        this.ui.enable('startGame');
                        this.ui.disable(this.ui.you.textField);
                    }
                    // YOU STARTED - END

                    // FINISHED - BEG
                    if (this.getPlayerData('you', 'finished') || this.getPlayerData('opp', 'finished')) {
                        this.updateOne('you');
                        this.updateOne('opp');

                        this.ui.disable(this.ui.you.textField);

                        if (this.getPlayerData('you', 'finished')) {
                            this.ui.show('winner');
                        } else if (this.getPlayerData('opp', 'finished')) {
                            this.ui.show('loser');
                        }
                    } else {
                        this.ui.hide('winner');
                        this.ui.hide('loser');
                    }
                    // FINISHED - END
                } else {
                    this.ui.show('waitingForOpp');
                    this.ui.disable('startGame');
                    this.ui.hide('winner');
                    this.ui.hide('loser');
                    this.ui.disable(this.ui.you.textField);
                }
                // OPPONENT CONNECTED / DISCONNECTED - END
            } else {
                this.ui.hide('winner');
                this.ui.hide('loser');
                this.ui.hide('waitingForOpp');
                this.ui.disable(this.ui.you.textField);
            }
            // YOU CONNECTED / DISCONNECTED - END

            // GROUPS LIST - BEG
            if (this.gamesList.length !== this.ui.gamesList.children.length) {
                this.ui.gamesList.innerHTML = '';

                var joinListener = function (e) {
                    var li = e.target,
                        index;

                    index = li.getAttribute('data-index');

                    if (this.gamesList[index]) {
                        this.join(this.gamesList[index]);
                    }
                }.bind(this);

                if (this.ui.gamesList.children.length === 0) {
                    this.gamesList.forEach(function (device, index) {
                        var li;

                        li = document.createElement('li');
                        li.innerText = device.deviceName;
                        li.setAttribute('data-index', index);
                        li.addEventListener('click', joinListener);

                        this.ui.gamesList.appendChild(li);
                    }.bind(this));
                }
            }
            // GROUPS LIST - END
        },

        create: function () {
            this.loading = true;
            this.isServer = true;
            this.resetPlayersData(true);
            tizen.wifidirect.createGroup(this.onCreate.bind(this));
            this.update();
        },

        leave: function () {
            if (this.data.you.connected) {
                this.leaving = true;
                tizen.wifidirect.leaveGroup(this.onLeave.bind(this));
                this.update();
                this.ui.openScreen('menu');
            }
        },

        scan: function () {
            this.loading = true;
            tizen.wifidirect.scan(this.onScan.bind(this));
            this.update();
            this.ui.openScreen('games');
        },

        join: function (device) {
            this.loading = true;
            this.isServer = false;
            tizen.wifidirect.connect(device, this.onConnect.bind(this));

            this.gamesList = [];
            this.update();
        },

        initSocket: function () {
            tizen.wifidirect.initSocket(this.onInitSocket.bind(this));
        },

        onInit: function (error) {
            tizen.wifidirect.isActivated(function (error, isActivated) {
                if (error) {
                    this.onError(error);
                    return;
                }

                if (isActivated) {
                    this.onActivateSuccess();
                } else {
                    tizen.wifidirect.activate(this.onActivate.bind(this));
                }
            }.bind(this));

            this.update();
        },

        onActivate: function (error) {
            if (error) {
                this.onError(error);
                return;
            }
        },

        onActivateSuccess: function (error) {
            if (error) {
                this.onError(error);
                return;
            }

            this.activated = true;
            this.ui.openScreen('menu');
        },

        onError: function (error) {
            console.log(error.code);
            this.update();
        },

        onCreate: function (error) {
            if (error) {
                this.loading = false;
                this.onError(error);
            }
        },

        onCreateSuccess: function (error) {
            this.loading = false;

            if (error) {
                this.onError(error);
                return;
            }

            if (this.isServer) {
                this.data.you.connected = true;
                this.initSocket();
                this.ui.openScreen('game');
            }                

            this.update();
        },

        onLeave: function (error) {
            if (error) {
                this.leaving = false;
                this.onError(error);
            }
        },

        onLeaveSuccess: function (error) {
            this.leaving = false;

            if (error) {
                this.onError(error);
                return;
            }

            this.data.you.connected = false;
            this.isServer = null;

            this.update();
        },

        onScan: function (error) {
            if (error) {
                this.loading = false;
                this.onError(error);
            }
        },

        onScanSuccess: function (error, devices) {
            this.loading = false;

            if (error) {
                this.onError(error);
                return;
            }

            this.gamesList = devices;

            this.update();
        },

        onConnect: function (error) {
            if (error) {
                this.loading = false;
                this.onError(error);
            }
        },

        onConnectSuccess: function (error, device) {
            this.loading = false;

            if (error) {
                this.onError(error);
                return;
            }

            if (this.isServer) { // Server
                this.data.opp.connected = true;
            } else { // Client
                this.data.you.connected = true;
                this.data.opp.connected = true;
                this.gamesList = [];
                this.initSocket();
                this.resetPlayersData();
                this.ui.openScreen('game');
            }

            this.update();
        },

        onDisconnectSuccess: function (error) {
            if (error) {
                this.onError(error);
                return;
            }

            this.data.you.connected = false;
            this.ui.openScreen('menu');

            this.isServer = null;

            this.update();
        },

        onInitSocket: function (error) {
            if (error) {
                this.onError(error);
                return;
            }
        },

        onInitSocketSuccess: function (error) {
        },

        onDataReceived: function (error, data) {
            if (error) {
                this.onError(error);
                return;
            }

            data = JSON.parse(data.message);

            switch (data.command) {
            case 'words':
                this.wordsIndexes = data.data;
                this.update();
                break;
            case 'wordIndex':
                this.setPlayerData('opp', data.command, data.data);
                // Update word's image.
                this.printWord(this.getPlayerData('opp', 'currWord'), 'opp');
                this.update();
                break;
            case 'typedWord':
                this.setPlayerData('opp', data.command, data.data);
                this.update();
                break;
            case 'finished':
                this.setPlayerData('opp', data.command, data.data);
                this.finish(false);
                break;
            case 'started':
                this.setPlayerData('opp', data.command, data.data);
                if (this.checkIfReadyToStart()) {
                    this.countDown();
                }
                break;
            }
        },

        resetPlayersData: function (hard) {
            hard = hard || false;

            this.data.you.wordIndex = 0;
            this.data.you.typedWord = '';
            this.data.you.finished = false;
            this.data.you.winner = false;
        
            this.data.opp.wordIndex = 0;
            this.data.opp.typedWord = '';
            this.data.opp.finished = false;
            this.data.opp.winner = false;

            if (hard) {
                this.data.you.connected = false;
                this.data.you.started = false;

                this.data.opp.connected = false;
                this.data.opp.started = false;
            }
        },

        start: function () {

            this.resetPlayersData();

            if (this.isServer) {
                this.drawWords();
                this.sendWords();
            }

            this.data.you.started = true;
            this.winner = null;
            this.sendPlayerData('started');

            if (this.checkIfReadyToStart()) {
                this.countDown();
            }
            this.update();
        },

        checkIfReadyToStart: function () {
            return this.getPlayerData('you', 'started') && this.getPlayerData('opp', 'started');
        },

        countDown: function () {
            this.countingDown = true;
            this.ui.countDown.classList.add('count-down');
            this.update();
        },

        onCountDownFinished: function (e) {
            if (e.target.innerText === '1') {
                this.ui.countDown.classList.remove('count-down');
                this.countingDown = false;
                this.data.you.started = true;

                this.printWord(this.getPlayerData('you', 'currWord'), 'you');
                this.printWord(this.getPlayerData('opp', 'currWord'), 'opp');

                this.update();
            }
        },

        finish: function (winner) {
            this.setPlayerData('you', 'started', false);
            this.setPlayerData('opp', 'started', false);

            if (winner) {
                this.setPlayerData('you', 'winner', true);
                this.setPlayerData('opp', 'winner', false);
            } else {
                this.setPlayerData('you', 'winner', false);
                this.setPlayerData('opp', 'winner', true);
            }

            this.update();
        },

        drawWords: function () {
            var index;

            this.wordsIndexes = [];
            while (this.wordsIndexes.length < this.wordsCount) {
                index = Math.floor(Math.random() * wordsList.length);
                if (this.wordsIndexes.indexOf(index) >= 0) continue;
                this.wordsIndexes.push(index);
            }
        },

        sendPlayerData: function (propertyName) {
            if (!_.has(this.data.you, propertyName)) return;

            var data = JSON.stringify({
                command: propertyName,
                data: this.data.you[propertyName]
            });

            tizen.wifidirect.sendBroadcast(data, this.onSendBroadcast.bind(this));
        },

        sendWords: function () {
            var data = JSON.stringify({
                command: 'words',
                data: this.wordsIndexes
            });

            tizen.wifidirect.sendBroadcast(data, this.onSendBroadcast.bind(this));
        },

        onSendBroadcast: function (error) {
            if (error) {
                this.onError(error);
                return;
            }
        }
    });

    var game = new Game();

    game.ui.openScreen('start');

    window.game = game;
})();