/*******************************************************************************
 *
 * @author Zaneta Szymanska <<ahref='mailto:z.szymanska@samsung.com'>z.szymanska@samsung.com</a>>
 *
 * **************************************************************************************
 *
 * Copyright (c) 2012 Samsung Electronics All Rights Reserved.
 *
 ******************************************************************************/
"use strict";
var tizenPaint = function() {

    tlib.logger.changeLogLevel(3);
    /**
     * Instance of fabric.Canvas
     */
    var fabricCanvas;
    /**
     * Name of actual sellected shape
     */
    var currShape = '';
    /**
     * Resolved directory
     */
    var gDocumentsDir;
    /**
     * Serialized canvas
     */
    var canvasSerialized;

    var svgFileName = '';

    /**
     * Closes application if possible
     */
    var exit = function() {
        if (typeof tizen !== "undefined" && tizen.application) {
            if (confirm("Exit?")) {
                var app = tizen.application.getCurrentApplication();
                app.exit();
                tlib.logger.info("Application closed");
            }
        } else {
            alert("Not supported");
        }
    };

    function init() {

        // Prevents scrolling effect
        $('body').bind('touchmove', function(e) {
            e.preventDefault();
        });

        fabricCanvas = new fabric.Canvas('paint-canvas');

        var str = '';

        for ( var i = 0; i < 16; i++) {
            str += '<input id="color' + (i + 1)
                    + '" class="colorInput" type="submit" value="" />';
        }

        $('#colors').append(str).trigger("create");
        // color1 is selected by default
        $('#color1').prop("checked", true);
        actualColor = $('#color1');

        // Default properties: edit mode, pencil
        drawMode = false;
        fabricCanvas.isDrawingMode = false;
        currShape = $('#pencil').val();
        fabricCanvas.freeDrawingLineWidth = parseInt($('#line-width').text(),
                10);

        /**
         * Changes selected color on click
         */
        $('.colorInput').siblings().unbind().click(function(event) {
            setColors($(this));
        });

        $('#add-figure').unbind().click(function() {
            /**
             * Draws line or adds figure on click
             */
            if ($('#shape-menu').hasClass('hidden')) {
                drawMode = true;
                if (currShape === 'pencil') {
                    fabricCanvas.isDrawingMode = true;
                } else {
                    $('#stroke-menu').toggleClass('hidden', false);
                }
                if (currShape === 'polygon') {
                    $('#vertex').toggleClass('hidden', false);
                }
                if (!($('#colors-menu').hasClass('hidden'))) {
                    $('#colors-menu').addClass('hidden');
                    $('#choose-color').removeClass('selected');
                }
                $('#shape-menu').toggleClass('hidden', false);
                $('#edit-figure').removeClass('selected');
                $('#add-figure').addClass('selected');

                fabricCanvas.deactivateAll();
                for ( var i = 0; i < fabricCanvas.getObjects().length; i++) {
                    fabricCanvas.item(i).selectable = false;
                }

                fabricCanvas.renderAll();

            } else {
                $('#shape-menu').toggleClass('hidden', true);
                $('#stroke-menu').toggleClass('hidden', true);
                $('#vertex').toggleClass('hidden', true);
            }
        });

        $('#edit-figure').unbind().click(editModeOn);

        $('#choose-color').unbind().click(function() {
            /**
             * Shows menu with available colors
             */
            if ($('#colors-menu').hasClass('hidden')) {
                $('#colors-menu').toggleClass('hidden', false);
                $('#choose-color').addClass('selected');
                if (!($('#shape-menu').hasClass('hidden'))) {
                    $('#shape-menu').toggleClass('hidden', true);
                    $('#stroke-menu').toggleClass('hidden', true);
                    $('#vertex').toggleClass('hidden', true);
                    $('#add-figure').removeClass('selected');
                }
            } else {
                $('#colors-menu').toggleClass('hidden', true);
                $('#choose-color').removeClass('selected');
            }
        });

        $('#svg-file1').parent().children().unbind().click(function() {
            /**
             * Adds selected SVG file to canvas
             */
            if (!(svgFileName.trim())) {
                editModeOn();
                $(this).addClass('selected');
                svgFileName = $(this).val();
            } else {
                $(this).parent().children().removeClass('selected');
                $(this).addClass('selected');
                svgFileName = $(this).val();
            }
        });

        fabricCanvas.observe('object:selected', function() {
            /**
             * Handles selection of object event In drawMode objects have to be
             * no-selectable and inactive
             */
            if (drawMode) {
                fabricCanvas.getActiveObject().set('selectable', false);
                fabricCanvas.getActiveObject().set('active', false);
            } else {
                if (fabricCanvas.getActiveObject().get('strokeWidth')) {
                    $('#line-width').text(
                            fabricCanvas.getActiveObject().get('strokeWidth'));
                    fabricCanvas.getActiveObject().set('cornersize', 30);
                }
            }
        });

        fabricCanvas.observe('mouse:down', handleTouchEvent);

        $('#pencil').parent().children().unbind().click(function() {
            /**
             * Provides handling for shape's menu
             */
            if (currShape !== $(this).val()) {
                currShape = $(this).val();
                if ($(this).val() === 'pencil') {
                    fabricCanvas.isDrawingMode = true;
                    $('#stroke-menu').toggleClass('hidden', true);
                } else {
                    if (fabricCanvas.isDrawingMode) {
                        fabricCanvas.isDrawingMode = false;
                    }
                    $('#stroke-menu').toggleClass('hidden', false);
                }

                if ($(this).val() === 'polygon') {
                    $('#vertex').toggleClass('hidden', false);
                } else {
                    $('#vertex').toggleClass('hidden', true);
                }

                $(this).siblings().removeClass('selected');
                $(this).addClass('selected');
            }
        });

        $('#save-as').parent().children().unbind().click(function() {
            /**
             * Provides handling for top menu
             */
            // Hide all open options
            if (!($('#svg-files').hasClass('hidden'))) {
                svgFileName = '';
                $('#svg-file1').parent().children().removeClass('selected');
            }

            $('#file-name-option').toggleClass('hidden', true);
            $('#svg-files').toggleClass('hidden', true);

            if ($(this).hasClass('selected')) {
                $(this).parent().children().toggleClass('selected', false);
            } else {
                $(this).parent().children().toggleClass('selected', false);

                if ($(this).val() !== 'Clear canvas') {
                    $(this).toggleClass('selected', true);
                }

                switch ($(this).val()) {
                case 'Save As': {
                    /**
                     * Open input for filename with "Save" option
                     */
                    $('#save').removeClass('hidden');
                    $('#open').addClass('hidden');
                    $('#file-name-option').toggleClass('hidden', false);
                }
                    break;

                case 'Open': {
                    /**
                     * Open input for filename with "Open" option
                     */
                    $('#open').removeClass('hidden');
                    $('#save').addClass('hidden');
                    $('#file-name-option').toggleClass('hidden', false);
                }
                    break;

                case 'SVG files': {
                    /**
                     * Open menu for SVG files
                     */
                    $('#svg-files').toggleClass('hidden', false);
                }
                    break;

                case 'Clear canvas': {
                    /**
                     * Clear canvas
                     */
                    fabricCanvas.clear();
                }
                    break;

                case 'Close': {
                    exit();
                }
                    break;

                default:
                    break;
                }
            }
        });

        $('#save').unbind().click(function() {
            /**
             * Serializes canvas and writes it to file File is created if
             * doesn't exist
             */
            canvasSerialized = JSON.stringify(fabricCanvas);

            var filename = $('#file-name').val();

            if (!filename) {
                $('#alert-popup').toggleClass('alert-popup-maxi', false);
                $('#alert-popup').toggleClass('alert-popup-mini', true);
                $('#alert-popup').toggleClass('hidden', false);
                $('#alert-popup p').text('Input File name !');
                return;
            }

            try {
                gDocumentsDir.createFile(filename);
            } catch (exc) {
                $('#save-popup').toggleClass('hidden', false);
                return;
            }

            writeToFile($('#file-name').val());

            $('#file-name-option').toggleClass('hidden', true);
            $('#save-as').toggleClass('selected', false);
        });

        $('#replace-yes').unbind().click(function() {
            writeToFile($('#file-name').val());

            $('#file-name-option').toggleClass('hidden', true);
            $('#save-as').toggleClass('selected', false);

            $('#save-popup').toggleClass('hidden', true);
        });

        $('#replace-no').unbind().click(function() {
            $('#save-popup').toggleClass('hidden', true);
            return;
        });

        $('#alert-ok').unbind().click(function() {
            $('#alert-popup').toggleClass('hidden', true);
        });

        $('#open').unbind().click(function() {
            /**
             * Open and display content of selected file
             */
            if ($('#file-name').val().trim()) {
                displayFileContents();
                $('#file-name-option').toggleClass('hidden', true);
                $('#open-file').toggleClass('selected', false);
            } else {
                $('#alert-popup').toggleClass('alert-popup-maxi', false);
                $('#alert-popup').toggleClass('alert-popup-mini', true);
                $('#alert-popup').toggleClass('hidden', false);
                $('#alert-popup p').text('Input File name !');
            }
        });

        /**
         * Changes line width
         */

        $('#line-width-minus').unbind().click(function() {
            var value = $('#line-width').text();
            if (value > 1) {
                value--;
            }
            setStrokeWidth(value);
        });

        $('#line-width-plus').unbind().click(function() {
            var value = $('#line-width').text();
            if (value < 16) {
                value++;
            }
            setStrokeWidth(value);
        });

        /**
         * Changes number of polygon's vertex
         */

        $('#vertex-count-minus').unbind().click(function() {
            var vertexCount = $('#vertex-count').text();
            if (vertexCount > 4) {
                vertexCount--;
            }
            $('#vertex-count').text(vertexCount);
        });

        $('#vertex-count-plus').unbind().click(function() {
            var vertexCount = $('#vertex-count').text();
            if (vertexCount < 12) {
                vertexCount++;
            }
            $('#vertex-count').text(vertexCount);
        });

        /**
         * Sets stroke and fill options for figure
         */

        $('#stroke').unbind().click(function() {
            stroke = true;
            $('#fill').removeClass('selected');
            $('#stroke').addClass('selected');
        });

        $('#fill').unbind().click(function() {
            stroke = false;
            $('#stroke').removeClass('selected');
            $('#fill').addClass('selected');
        });

    }
    /**
     * Set some properties like: figure's stokeWidth, lineWidth
     *
     * @param value Width of the line/stroke
     */
    function setStrokeWidth(value) {
        $('#line-width').text(value);
        fabricCanvas.freeDrawingLineWidth = parseInt($('#line-width').text(),
                10);

        if (fabricCanvas.getActiveObject()) {
            if (fabricCanvas.getActiveObject().get('stroke')) {
                fabricCanvas.getActiveObject().set('strokeWidth',
                        parseInt($('#line-width').text(), 10) || 1);
            }
            fabricCanvas.renderAll();
        }
    }

    /**
     * Resolves & opens file in write mode
     *
     * @param filename Name of the file to open
     */
    function writeToFile(filename) {
        var file;

        try {
            file = gDocumentsDir.resolve(filename);
        } catch (exc) {
            tlib.logger.err('Could not resolve file: ' + exc.message);
            return;
        }

        try {
            file.openStream('w', writeToStream, onError);
        } catch (exc) {
            tlib.logger.err('Could not write to file: ' + exc.message);
        }
    }

    /**
     * Writes serialized canvas to the file
     *
     * @param fileStream FileStreamSuccessCallback
     */
    function writeToStream(fileStream) {
        try {
            fileStream.write(canvasSerialized);
            fileStream.close();
        } catch (exc) {
            tlib.logger.err(exc.message);
        }
    }

    /**
     * Resolves & opens file in read mode
     */
    function displayFileContents() {
        var file;

        try {
            file = gDocumentsDir.resolve($('#file-name').val());
        } catch (exc) {
            $('#alert-popup').toggleClass('alert-popup-maxi', true);
            $('#alert-popup').toggleClass('alert-popup-mini', false);
            $('#alert-popup').toggleClass('hidden', false);
            $('#alert-popup p').text('File does not exist or access denied.');
            return;
        }

        try {
            file.openStream('r', readFromStream, onError);
        } catch (exc) {
            tlib.logger.err('openStream() exception:' + exc.message);
        }
    }

    /**
     * Reads and displays file's content (serialized canvas) Loads content to
     * fabricCanvas
     *
     * @param fileStream FileStreamSuccessCallback
     */
    function readFromStream(fileStream) {
        try {
            var contents = fileStream.read(fileStream.bytesAvailable);
            fileStream.close();

            fabricCanvas.loadFromJSON(contents);

            for ( var i = 0; i < fabricCanvas.getObjects().length; i++) {
                if (drawMode) {
                    fabricCanvas.item(i).set('selectable', false);
                } else {
                    fabricCanvas.item(i).set('selectable', true);
                }
            }
        } catch (exc) {
            tlib.logger.err('File reading exception: ' + exc.message);
        }
    }

    /**
     * ErrorCallback
     */
    function onError(err) {
        tlib.logger.err("Error: " + err.message);
    }

    /**
     * On touch event: - adds selected figure/line to the canvas or - or adds
     * svg file on touched place
     *
     * @param e Event object
     */
    function handleTouchEvent(e) {
        e.e.preventDefault();

        var touches = fabricCanvas.getPointer(e.e);
        var x = touches.x;
        var y = touches.y;

        if (drawMode) {
            var fig = new Figures(x, y);

            switch (currShape) {
            case 'square':
                fabricCanvas.add(fig.square());
                break;

            case 'circle':
                fabricCanvas.add(fig.circle());
                break;

            case 'triangle':
                fabricCanvas.add(fig.triangle());
                break;

            case 'polygon':
                fabricCanvas.add(fig.polygon());
                break;

            default:
                break;
            }

        } else {
            if (svgFileName.trim()) {
                fabric.loadSVGFromURL('images/' + svgFileName + '.svg',
                        function(objects, options) {
                            var loadedObject = fabric.util.groupSVGElements(
                                    objects, options);

                            loadedObject.set({
                                left : x,
                                top : y,
                                angle : 0,
                                padding : 0
                            });
                            loadedObject.setCoords();

                            fabricCanvas.add(loadedObject);
                        });

                svgFileName = '';
                $('#svg-file1').parent().children().removeClass('selected');
            }
        }
        fabricCanvas.renderAll();
    }

    /**
     * Sets color of selected object or color you want to use to draw object
     *
     * @param color Selected element
     */
    function setColors(color) {
        if (!(color.prop("checked"))) {
            actualColor.css('border', '1px solid black');
            actualColor.prop("checked", false);
            color.css('border', '3px solid red');
            color.prop("checked", true);
            actualColor = color;

            fabricCanvas.freeDrawingColor = color.css('background-color');

            if (fabricCanvas.getActiveObject()
                    && fabricCanvas.getActiveObject().get('active')) {
                if (fabricCanvas.getActiveObject().get('stroke')) {
                    fabricCanvas.getActiveObject().set('stroke',
                            color.css('background-color'));
                } else {
                    fabricCanvas.getActiveObject().set('fill',
                            color.css('background-color'));
                }
            }

            fabricCanvas.renderAll();
        }
    }

    /**
     * Makes figures selectable
     */
    function editModeOn() {
        if (drawMode) {
            drawMode = false;
            fabricCanvas.isDrawingMode = false;
            $('#shape-menu').toggleClass('hidden', true);
            $('#stroke-menu').toggleClass('hidden', true);
            $('#vertex').toggleClass('hidden', true);
            $('#add-figure').removeClass('selected');
            $('#edit-figure').addClass('selected');

            for ( var i = 0; i < fabricCanvas.getObjects().length; i++) {
                fabricCanvas.item(i).selectable = true;
            }

            fabricCanvas.renderAll();
        }
    }

    return {
        /**
         * Provides initialization for the app
         */
        initialize : function() {
            init();
            // Resolve directory
            try {
                tizen.filesystem.resolve("documents", function(dir) {
                    gDocumentsDir = dir;
                }, onError, "rw");
            } catch (exc) {
                tlib.logger.err("tizen.filesystem.resolve(\"documents\") exc: "
                        + exc.message);
            }
        }

    };
}();