package org.tizen.gwt.bootstrap;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;

import org.tizen.gwt.ant.TizenIdUtil;

public class ProjectBootstrapper {

	private static final String SELF_JAR_PATH = ProjectBootstrapper.class.getProtectionDomain().getCodeSource().getLocation().getPath();
	private static final String ROOT_PATH = new File(SELF_JAR_PATH).getParent();
	
	private final OutputConsole mOutputConsole;

	public ProjectBootstrapper(OutputConsole outputConsole) {
		mOutputConsole = outputConsole;
	}
	
	public void bootstrapBuildFiles(GwtModuleInfo moduleInfo) throws IOException {
		
		final File buildProperties = getCopyTarget("build.properties");
		if(overrideExisting(buildProperties)) {
			copyTextFile("templates/build.properties", buildProperties, null);
		}

		final File buildXml = getCopyTarget("build.xml");
		if(overrideExisting(buildXml)) {
			copyTextFile("templates/build.xml", buildXml, new String[][] {
					{ "__GWT_ENTRYPOINT_NAME__", moduleInfo.getEntryPointName() },
					{ "__GWT_PACKAGE_NAME__", moduleInfo.getPackageName() },
					{ "__SELF_JAR__", SELF_JAR_PATH }
			});
		}
	}

	public void bootstrapProjectFiles(GwtModuleInfo moduleInfo) throws IOException {
		
		final File configXml = getCopyTarget("config.xml");
		if(overrideExisting(configXml)) {
			copyTextFile("templates/config.xml", configXml, new String[][] {
					{ "${ENTRY_POINT_NAME}", moduleInfo.getEntryPointName() },
					{ "${TIZEN_PACKAGE_ID}", TizenIdUtil.generateId(10) },
					{ "${W3C_WIDGET_ID}", moduleInfo.getW3CWidgetId() }
			});
		}
		
		final File webXml = getCopyTarget("war/WEB-INF/web.xml");
		if(!webXml.exists()) {
			copyTextFile("templates/web.xml", webXml, null);
		}
		
		final File entryPointJava = getCopyTarget(getEntryPointPath(moduleInfo));
		if(overrideExisting(entryPointJava)) {
			copyTextFile("templates/ENTRY_POINT.java.tpl", entryPointJava, new String[][] {
					{ "${PACKAGE}", moduleInfo.getClientPackageName() },
					{ "${ENTRY_POINT}", moduleInfo.getEntryPointName() }					
			});
		}

		final File moduleGwtXml = getCopyTarget(getModuleDescriptorPath(moduleInfo));
		if(overrideExisting(moduleGwtXml)) {
			copyTextFile("templates/MODULE_NAME.gwt.xml", moduleGwtXml, new String[][] {
					{ "${GWT_ENTRY_POINT_CLASS}", moduleInfo.getEntryPointFullName() },
					{ "${GWT_RENAME_TO}", moduleInfo.getModuleRenameTo() }					
			});
		}
		
		final File indexHtml = getCopyTarget("war/index.html");
		if(overrideExisting(indexHtml)) {
			copyTextFile("templates/index.html", indexHtml, new String[][] {
					{ "${GWT_RENAME_TO}", moduleInfo.getModuleRenameTo() },
					{ "${PAGE_TITLE}", moduleInfo.getEntryPointName() }
			});
		}

		final File iconPng = getCopyTarget("war/icon.png");
		if(overrideExisting(iconPng)) {
			copy("templates/icon.png", iconPng);
		}
	}
	
	private String getEntryPointPath(GwtModuleInfo moduleInfo) {
		return "src/" + moduleInfo.getClientPackageName().replaceAll(
				"\\.", "/") + "/" + moduleInfo.getEntryPointName() + ".java";
	}
	
	private String getModuleDescriptorPath(GwtModuleInfo moduleInfo) {
		return "src/" + moduleInfo.getPackageName().replaceAll(
				"\\.", "/") + "/" + moduleInfo.getEntryPointName() + ".gwt.xml";
	}
	
	private File getCopyTarget(String outputPath) {
		return new File(ROOT_PATH + "/" + outputPath);
	}
	
	private boolean overrideExisting(File file) {
		if(file.exists()) {
			return mOutputConsole.prompt("Should overwrite existing file \"" 
					+ file.getPath() +"\"?");
		}
		return true;
	}
	
	private void copyTextFile(String resName, File outputFile, String[][] replacements) throws IOException {
		BufferedReader in = null;
		FileWriter out = null;
		try {
			if(!outputFile.getParentFile().exists()) {
				outputFile.getParentFile().mkdirs();
			}
			
			in = new BufferedReader(new InputStreamReader(
					getClass().getClassLoader().getResourceAsStream(resName)));
			out = new FileWriter(outputFile);
			String line;
			while((line = in.readLine()) != null) {
				if(replacements != null) {
					for(String[] pair : replacements) {
						line = line.replace(pair[0], pair[1]);
					}
				}
				out.write(line + System.lineSeparator());
			}
			out.flush();
			mOutputConsole.logInfo("Created file: " + outputFile.getPath());
		}
		finally {
			try {
				in.close();
				out.close();
			}
			catch (Exception e) {}
		}
	}
	
	private void copy(String resName, File outputFile) throws IOException {
		InputStream in = null;
		FileOutputStream out = null;
		try {
			in = getClass().getClassLoader().getResourceAsStream(resName);
			if(!outputFile.getParentFile().exists()) {
				outputFile.getParentFile().mkdirs();
			}
			out = new FileOutputStream(outputFile);
			int b;
			while((b = in.read()) > -1) {
				out.write(b);
			}
			out.flush();
			mOutputConsole.logInfo("Created file: " + outputFile.getPath());
		} 
		finally {
			try {
				in.close();
				out.close();
			}
			catch (Exception c) {}
		}
	}
}