/*******************************************************************************
 *
 * @author Zaneta Szymanska <<ahref='mailto:z.szymanska@samsung.com'>z.szymanska@samsung.com</a>>
 *
 * **************************************************************************************
 *
 * Copyright (c) 2012 Samsung Electronics All Rights Reserved.
 *
 ******************************************************************************/
"use strict";
var gradients = function() {

    // default properties of gradient
    var defaultProperties = {
        fromColor : {
            red : 255,
            green : 255,
            blue : 255,
            alpha : 0.5
        },
        toColor : {
            red : 0,
            green : 0,
            blue : 0,
            alpha : 0.5
        },
        toPosition : {
            xEnable : false,
            yEnable : false,
            x : 'right',
            y : 'top'
        },
        angle : 0,
        shape : 'circle',
        size : 'def',
        type : 'linear'
    };

    // actual active color tab
    var activeColorTab = 'fromColor';

    // actual properties of gradient
    var properties = jQuery.extend(true, {}, defaultProperties);

    /**
     * Sets values of visible color's sliders and refreshes them
     *
     * @param color {Object} Contains rgba values for color's sliders
     */
    var setSliders = function(color) {
        $('#red').val(color.red).slider("refresh");
        $('#green').val(color.green).slider("refresh");
        $('#blue').val(color.blue).slider("refresh");
        $('#alpha').val(color.alpha).slider("refresh");
    };

    /**
     * Restores original image and sets default configuration
     */
    var setDefaultConfiguration = function() {
        $('#linear').trigger("click");
        $(".gtypes").checkboxradio('refresh');

        $('#direction').trigger("click");
        $("input[name='gradient-parameters']").checkboxradio('refresh');

        if (properties.toPosition.xEnable) {
            $('#if-x-enable').trigger("click").checkboxradio('refresh');
        }
        if (properties.toPosition.yEnable) {
            $('#if-y-enable').trigger("click").checkboxradio('refresh');
        }

        properties = jQuery.extend(true, {}, defaultProperties);

        $("#x-line-" + defaultProperties.toPosition.x).trigger("click");
        $('#y-line-' + defaultProperties.toPosition.y).trigger("click");

        setDirectionButtons(false, '#x-line-left', '#x-line-right');
        setDirectionButtons(false, '#y-line-top', '#y-line-bottom');

        $(".gr-direction").checkboxradio('refresh');

        $('#angle-slider').val(defaultProperties.angle).slider("refresh");

        $('#circle').trigger("click");
        $("input[name='shape-menu']").checkboxradio('refresh');

        $('#def').trigger("click");
        $("input[name='size-menu']").checkboxradio('refresh');

        $("#color-1").trigger("click");
        $(".colors").checkboxradio('refresh');

        setSliders(defaultProperties.fromColor);

        $('#angle-menu').hide();
        $('#radial-shape-menu').hide();
        $('#radial-size-menu').hide();

        $('#radial-shape').attr("disabled", true).checkboxradio("refresh");
        $('#radial-size').attr("disabled", true).checkboxradio("refresh");

    };

    /**
     * Sets gradient on the image
     */
    var setGradientOnImage = function() {
        var css = "";

        css += "-webkit-" + properties.type + "-gradient(";

        if (((properties.type === 'linear') || (properties.type === 'repeating-linear'))
                && properties.angle !== 0) {
            css += properties.angle + "deg, ";
        }

        if (properties.toPosition.xEnable || properties.toPosition.yEnable) {
            if (properties.toPosition.xEnable) {
                css += properties.toPosition.x;
            }
            if (properties.toPosition.yEnable) {
                css += " " + properties.toPosition.y;
            }
            css += ", ";
        }

        if ((properties.type === 'radial')
                || (properties.type === 'repeating-radial')) {
            css += properties.shape;

            if (properties.size !== 'def') {
                css += " " + properties.size + ", ";
            } else {
                css += ", ";
            }
        }

        css += "rgba("
                + properties.fromColor.red
                + ", "
                + properties.fromColor.green
                + ", "
                + properties.fromColor.blue
                + ", "
                + properties.fromColor.alpha
                + ")"
                + ((properties.type === 'repeating-linear')
                        || (properties.type === 'repeating-radial') ? " 20%"
                        : "")
                + ", rgba("
                + properties.toColor.red
                + ", "
                + properties.toColor.green
                + ", "
                + properties.toColor.blue
                + ", "
                + properties.toColor.alpha
                + ")"
                + ((properties.type === 'repeating-linear')
                        || (properties.type === 'repeating-radial') ? " 50%"
                        : "") + ")";

        css += ", url(./css/images/lena.jpg)";

        $('#image-with-gradient').css('background-image', css);
        $('#used-gradient').val(css);
    };

    /**
     * Sets UI buttons of directions (left-right, top-bottom) enabled/disabled
     *
     * @param enable {Boolean} If buttons of directions (left-right/top-bottom)
     *        should be enabled
     * @param firstB {String} Name of the first button which should be
     *        enabled/disabled
     * @param secondB {String} Name of the second button which should be
     *        enabled/disabled
     */
    var setDirectionButtons = function(enable, firstB, secondB) {
        if (enable) {
            $(firstB).attr("disabled", false);
            $(secondB).attr("disabled", false);
        } else {
            $(firstB).attr("disabled", true);
            $(secondB).attr("disabled", true);
        }

        $(firstB).checkboxradio("refresh");
        $(secondB).checkboxradio("refresh");
    };

    /**
     * Sets proper state for parameters' buttons (according to type of gradient
     * to set)
     *
     * @param direction {Boolean} If 'direction' parameter is enable for
     *        selected type of gradient
     * @param angle {Boolean} If 'angle' parameter is enable for selected type
     *        of gradient
     * @param shape {Boolean} If 'shape' parameter is enable for selected type
     *        of gradient
     * @param size {Boolean} If 'size' parameter is enable for selected type of
     *        gradient
     */
    var setDisabledParameters = function(gradient) {
        $('#direction').attr("disabled", gradient.direction).checkboxradio(
                "refresh");
        $('#linear-angle').attr("disabled", gradient.angle).checkboxradio(
                "refresh");
        $('#radial-shape').attr("disabled", gradient.shape).checkboxradio(
                "refresh");
        $('#radial-size').attr("disabled", gradient.size).checkboxradio(
                "refresh");

        $(".gradient-parameters").checkboxradio('refresh');
    };

    /**
     * Sets proper menu visible
     *
     * @param direction {Boolean} If 'direction-menu' is visible for selected
     *        parameter
     * @param angle {Boolean} If 'angle-menu' is visible for selected parameter
     * @param shape {Boolean} If 'radial-shape-menu' is visible for selected
     *        parameter
     * @param size {Boolean} If 'radial-size-menu' is visible for selected
     *        parameter
     */
    var setMenuVisible = function(gradient) {
        (gradient.direction) ? $('#direction-menu').show() : $(
                '#direction-menu').hide();
        (gradient.angle) ? $('#angle-menu').show() : $('#angle-menu').hide();
        (gradient.shape) ? $('#radial-shape-menu').show() : $(
                '#radial-shape-menu').hide();
        (gradient.size) ? $('#radial-size-menu').show()
                : $('#radial-size-menu').hide();
    };

    /**
     * Set menu of parameters for selected gradient's type
     */
    var setPropertiesMenu = function() {

        if (properties.type === 'no-gradient') {
            var css = "url(./css/images/lena.jpg)";
            $('#image-with-gradient').css('background-image', css);

            setDisabledParameters({
                direction : true,
                angle : true,
                shape : true,
                size : true
            });

            setMenuVisible({
                direction : false,
                angle : false,
                shape : false,
                size : false
            });

            $('#used-gradient').val(css);

        } else {

            if ((properties.type === 'linear')
                    || (properties.type === 'repeating-linear')) {

                $('#direction').trigger("click").checkboxradio("refresh");

                setDisabledParameters({
                    direction : false,
                    angle : false,
                    shape : true,
                    size : true
                });

            } else {
                if (properties.angle !== 0) {
                    properties.angle = 0;
                    $('#angle-slider').val(0).slider("refresh");
                }

                $('#direction').trigger("click").checkboxradio("refresh");

                setDisabledParameters({
                    direction : false,
                    angle : true,
                    shape : false,
                    size : true
                });

                if (properties.type === 'radial') {
                    $('#radial-size').attr("disabled", false).checkboxradio(
                            "refresh");
                }
            }

            setMenuVisible({
                direction : true,
                angle : false,
                shape : false,
                size : false
            });

            setGradientOnImage();
        }
    };

    /**
     * Allows on change the position of the gradient's line
     * (activates/deactivates axes: left-right, top-bottom)
     *
     * @param index {String} Name of axis to activate/deactivate
     */
    var setLinePos = function(index) {
        var firstB, secondB;

        if (properties.angle !== 0) {
            properties.angle = 0;
            $('#angle-slider').val(0).slider("refresh");
        }

        properties.toPosition[index] = !properties.toPosition[index];

        if (index === 'xEnable') {
            setDirectionButtons(properties.toPosition[index], '#x-line-left',
                    '#x-line-right');
        } else {
            setDirectionButtons(properties.toPosition[index], '#y-line-top',
                    '#y-line-bottom');
        }

        setGradientOnImage();
    };

    return {
        /**
         * Provides initialization for the app
         */
        initialize : function() {

            setDefaultConfiguration();

            $('#default').bind("click", function() {
                setDefaultConfiguration();
            });

            $('.colors').bind("click", function() {
                if (this.id.slice(6) === "1") {
                    setSliders(properties.fromColor);
                    activeColorTab = 'fromColor';
                } else {
                    setSliders(properties.toColor);
                    activeColorTab = 'toColor';
                }

                setGradientOnImage();
            });

            $('.color-range').on("slidestop", function() {
                properties[activeColorTab][this.id] = $(this).val();
                setGradientOnImage();
            });

            $('.gr-direction').bind("click", function() {
                var buttonId = this.id;

                if (buttonId.slice(0, 1) === 'x') {
                    properties.toPosition.x = buttonId.slice(7);
                } else {
                    properties.toPosition.y = buttonId.slice(7);
                }

                setGradientOnImage();
            });

            $('.gtypes').bind("click", function() {
                properties.type = this.id;
                setPropertiesMenu();
            });

            $("input[name='gradient-parameters']").bind("click", function() {
                var optionId = this.id;

                switch (optionId) {
                case "direction": {
                    setMenuVisible({
                        direction : true,
                        angle : false,
                        shape : false,
                        size : false
                    });
                }
                    break;

                case "linear-angle": {
                    setMenuVisible({
                        direction : false,
                        angle : true,
                        shape : false,
                        size : false
                    });
                }
                    break;

                case "radial-shape": {
                    setMenuVisible({
                        direction : false,
                        angle : false,
                        shape : true,
                        size : false
                    });
                }
                    break;

                case "radial-size": {
                    setMenuVisible({
                        direction : false,
                        angle : false,
                        shape : false,
                        size : true
                    });
                }
                    break;

                default:
                    break;
                }
            });

            $("input[name='shape-menu']").bind("click", function() {
                properties.shape = this.id;
                setGradientOnImage();
            });

            $("input[name='size-menu']").bind("click", function() {
                properties.size = this.id;
                setGradientOnImage();
            });

            $('#if-x-enable').bind("click", function() {
                setLinePos('xEnable');
            });

            $('#if-y-enable').bind("click", function() {
                setLinePos('yEnable');
            });

            $("#angle-slider")
                    .bind(
                            "slidestop",
                            function() {
                                var value = $(this).val();

                                if (value !== 0) {
                                    properties.angle = value;
                                }

                                if (properties.toPosition.xEnable
                                        || properties.toPosition.yEnable) {
                                    if (properties.toPosition.xEnable) {
                                        $('#if-x-enable').trigger("click")
                                                .checkboxradio("refresh");
                                    }
                                    if (properties.toPosition.yEnable) {
                                        $('#if-y-enable').trigger("click")
                                                .checkboxradio("refresh");
                                    }

                                    properties.toPosition.xEnable = properties.toPosition.yEnable = false;
                                }

                                setGradientOnImage();
                            });

            setGradientOnImage();
        }
    };
}();
