/*
 * Copyright (c) 2013 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "objloader.h"
#include "obj_parser.h"
#include <math.h>
#include <Elementary_GL_Helpers.h>
#include <image_util.h>

#define Z_POS_INC 0.01f

float *vertices;
float *texture_coord;
unsigned short *index_buffer;

_model_s diceModel = {0};
static const char fp[] = OBJ_FILENAME;

static void
set_perspective(Evas_Object *obj, float fovDegree, int w, int h, float zNear,  float zFar)
{
   float aspect;
   float fxdYMax, fxdYMin, fxdXMax, fxdXMin;
   int degree;

   ELEMENTARY_GLVIEW_USE(obj);

   if((w == 0) || (h == 0)) return;
   aspect = (w > h)? ((float)w / h): ((float)h / w);

   /* tan(double(degree) * 3.1415962 / 180.0 / 2.0); */
   static const float HALF_TAN_TABLE[91] =
   {
      0.00000f, 0.00873f, 0.01746f, 0.02619f, 0.03492f, 0.04366f, 0.05241f, 0.06116f, 0.06993f,
      0.07870f, 0.08749f, 0.09629f, 0.10510f, 0.11394f, 0.12278f, 0.13165f, 0.14054f, 0.14945f,
      0.15838f, 0.16734f, 0.17633f, 0.18534f, 0.19438f, 0.20345f, 0.21256f, 0.22169f, 0.23087f,
      0.24008f, 0.24933f, 0.25862f, 0.26795f, 0.27732f, 0.28675f, 0.29621f, 0.30573f, 0.31530f,
      0.32492f, 0.33460f, 0.34433f, 0.35412f, 0.36397f, 0.37389f, 0.38386f, 0.39391f, 0.40403f,
      0.41421f, 0.42448f, 0.43481f, 0.44523f, 0.45573f, 0.46631f, 0.47698f, 0.48773f, 0.49858f,
      0.50953f, 0.52057f, 0.53171f, 0.54296f, 0.55431f, 0.56577f, 0.57735f, 0.58905f, 0.60086f,
      0.61280f, 0.62487f, 0.63707f, 0.64941f, 0.66189f, 0.67451f, 0.68728f, 0.70021f, 0.71329f,
      0.72654f, 0.73996f, 0.75356f, 0.76733f, 0.78129f, 0.79544f, 0.80979f, 0.82434f, 0.83910f,
      0.85408f, 0.86929f, 0.88473f, 0.90041f, 0.91633f, 0.93252f, 0.94897f, 0.96569f, 0.98270f,
      1.00000f
   };

   degree = (int)(fovDegree + 0.5f);
   degree = (degree >=  0) ? degree :  0;
   degree = (degree <= 90) ? degree : 90;

   fxdYMax = zNear * HALF_TAN_TABLE[degree];
   fxdYMin = -fxdYMax;

   fxdXMax = fxdYMax * aspect;
   fxdXMin = -fxdXMax;

   glViewport(0, 0, w, h);
   glMatrixMode(GL_PROJECTION);
   glLoadIdentity();
   glFrustumf(fxdXMin, fxdXMax, fxdYMin, fxdYMax, zNear, zFar);
}

void
init_gles(Evas_Object *obj)
{
   int w, h;
   appdata_s *ad;

   ELEMENTARY_GLVIEW_USE(obj);
   ad = evas_object_data_get(obj, APPDATA_KEY);

  	static unsigned int size_decode = 0;
  	static unsigned char *img_source = NULL;
  	static int ret = 0;
  	static int width = 0, height = 0;
  	static const image_util_colorspace_e colorspace = IMAGE_UTIL_COLORSPACE_RGBA8888;

  	ret = image_util_decode_jpeg(TEXTURE_FILENAME, colorspace, &img_source, &width, &height, &size_decode);

   glGenTextures(1, ad->tex_ids);

   /* Create and map texture */
   glBindTexture(GL_TEXTURE_2D, ad->tex_ids[0]);

   glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, 64 * 3, 64 * 2, 0, GL_RGBA, GL_UNSIGNED_BYTE, img_source);

   glTexParameterx(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
   glTexParameterx(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
   glTexParameterx(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
   glTexParameterx(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);

   glShadeModel(GL_SMOOTH);

   glEnable(GL_CULL_FACE);
   glCullFace(GL_BACK);

   glEnable(GL_DEPTH_TEST);
   glDepthFunc(GL_LESS);

   elm_glview_size_get(obj, &w, &h);
   set_perspective(obj, 60.0f, w, h, 1.0f, 400.0f);

   getDataModelSizes(fp, &diceModel);

   vertices = calloc(diceModel.vertices * 2, sizeof(float));
   index_buffer = calloc(diceModel.vertices, sizeof(unsigned char));
   texture_coord = calloc(diceModel.texels * 2, sizeof(float));

   getDataModel(fp, vertices, index_buffer, texture_coord, diceModel);
}

void
destroy_gles(Evas_Object *obj)
{
   appdata_s *ad;

   ELEMENTARY_GLVIEW_USE(obj);
   ad = evas_object_data_get(obj, APPDATA_KEY);

   if (ad->tex_ids[0])
   {
      glDeleteTextures(1, &(ad->tex_ids[0]));
      ad->tex_ids[0] = 0;
   }
}

void
resize_gl(Evas_Object *obj)
{
   int w, h;

   elm_glview_size_get(obj, &w, &h);
   set_perspective(obj, 60.0f, w, h, 1.0f, 400.0f);
}

static void
draw_model(Evas_Object *obj)
{
   appdata_s *ad;
   static float zPos = -5.0f;
   static float zPosInc = Z_POS_INC;
   static int angle = 0;

   ELEMENTARY_GLVIEW_USE(obj);
   ad = evas_object_data_get(obj, APPDATA_KEY);

   glEnableClientState(GL_VERTEX_ARRAY);
   glVertexPointer(3, GL_FLOAT, 0, vertices);

   glEnableClientState(GL_TEXTURE_COORD_ARRAY);
   glTexCoordPointer(2, GL_FLOAT, 0, texture_coord);

   glEnable(GL_TEXTURE_2D);
   glBindTexture(GL_TEXTURE_2D, ad->tex_ids[0]);

   glMatrixMode(GL_MODELVIEW);

   zPos += zPosInc;

   if (zPos < -8.0f)
   {
      zPosInc = Z_POS_INC;
   }

   if (zPos > -5.0f)
      zPosInc = -Z_POS_INC;

   glLoadIdentity();
   glTranslatef(0, 1.2f, zPos);

   angle = (angle + 5) % (360 * 3);

   glRotatef((float)angle, 0, 1.0f, 0);
   glRotatef((float)angle, 1.0f, 0, 0);

   glDrawElements(GL_TRIANGLES, diceModel.vertices, GL_UNSIGNED_SHORT, &index_buffer[0]);

   glDisable(GL_TEXTURE_2D);
   glDisableClientState(GL_VERTEX_ARRAY);
   glDisableClientState(GL_TEXTURE_COORD_ARRAY);
}


void
draw_gl(Evas_Object *obj)
{
	ELEMENTARY_GLVIEW_USE(obj);

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glClearColor(0.0f, 0.0f, 0.0f, 0.0f);

	draw_model(obj);
}
