/*
 * Copyright (c) 2017 Samsung Electronics Co., Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

using System;
using Tizen.NUI;

namespace HelloTestExample
{
    internal class Example : NUIApplication
    {
        private TableView _contentContainer;
        private Stage _stage;
        private ProgressBar _progressBar;

        // The list of NUI views.
        // These are basic views for making application.
        private String[] _mViewList = { "PushButton", "Toggle", "CheckBox", "RadioButton", "TextLabel", "ImageView", "AnimateGif", "ProgressBar" };

        private void Initialize()
        {
            // Get Stage instance.
            // Stage is only one for each application.
            _stage = Stage.Instance;
            _stage.Key += OnStageKeyEvent;
            // Set Stage's background color using Color class constant.
            // Color class has constants for basic colors.
            _stage.BackgroundColor = Color.White;

            // Make Top Label
            TextLabel topLabel = new TextLabel();
            topLabel.Text = " NUI Views";

            // View can inherit parent's size via setting SizeModeFactor.
            // First value is factor for width.
            // Second value is factor for height.
            topLabel.WidthResizePolicy = ResizePolicyType.FillToParent;
            topLabel.HeightResizePolicy = ResizePolicyType.SizeRelativeToParent;
            topLabel.SizeModeFactor = new Vector3(0.0f, 0.1f, 0.0f);

            // Set AnchorPoint and ParentOrigin of TextLabel.
            topLabel.AnchorPoint = AnchorPoint.TopCenter;
            topLabel.ParentOrigin = ParentOrigin.TopCenter;
            topLabel.HorizontalAlignment = "BEGIN";
            topLabel.VerticalAlignment = "CENTER";
            topLabel.TextColor = Color.White;
            topLabel.BackgroundColor = new Color(43.0f / 255.0f, 145.0f / 255.0f, 175.0f / 255.0f, 1.0f);
            topLabel.PointSize = 16.0f;

            // Add TextLabel to Default Layer.
            // Stage has Default Layer to contain ordinary contents.
            _stage.GetDefaultLayer().Add(topLabel);

            // Make Grid container to contain views.
            // Use tableView because FlexContainer support focus navigation for two direction ( up/down or left/right )
            // First parameter is number of rows.
            // Second parameter is number of columns.
            _contentContainer = new TableView(6, 4);

            // Set width and height relative to parent.
            _contentContainer.WidthResizePolicy = ResizePolicyType.FillToParent;
            _contentContainer.HeightResizePolicy = ResizePolicyType.SizeRelativeToParent;
            _contentContainer.SizeModeFactor = new Vector3(0.0f, 0.9f, 0.0f);
            _contentContainer.AnchorPoint = AnchorPoint.BottomCenter;
            _contentContainer.ParentOrigin = ParentOrigin.BottomCenter;

            // Set height of each row relative to parent.
            // Developer can set each height of rows and each width of columns.
            _contentContainer.SetRelativeHeight(0, 0.07f);
            _contentContainer.SetRelativeHeight(1, 0.26f);
            _contentContainer.SetRelativeHeight(2, 0.07f);
            _contentContainer.SetRelativeHeight(3, 0.26f);
            _contentContainer.SetRelativeHeight(4, 0.07f);
            _contentContainer.SetRelativeHeight(5, 0.26f);

            // Make view can get focus.
            // View can get focus after Focusable sets as true.
            // If container view gets focus, it will pass focus to its child which can get focus.
            _contentContainer.Focusable = true;
            _stage.GetDefaultLayer().Add(_contentContainer);

            // Create all inner content.
            // Each item will consists of view name label and view.
            CreateContent();

            FocusManager.Instance.SetCurrentFocusView(View.DownCast<View>(_contentContainer.GetChildAt(new TableView.CellPosition(1, 0))));
            // Add callback to manage focus.
            // This callback is called before focus moves to other view.
            FocusManager.Instance.PreFocusChange += OnPreFocusChange;
        }

        // Callback for FocusManager.
        private View OnPreFocusChange(object source, FocusManager.PreFocusChangeEventArgs e)
        {
            // Set initial focused view.
            // If there is no focused view, then make first view in container be focused.
            // This logic can prevent focus lost.
            if (!e.ProposedView && !e.CurrentView)
            {
                // Get first child of container.
                // Downcast return value to "View", because the return value of GetChildAt() is Actor.
                // Assign return value to e.ProposedView.
                // FocusManager will give focus to e.ProposedView.
                e.ProposedView = View.DownCast<View>(_contentContainer.GetChildAt(new TableView.CellPosition(1, 0)));
            }
            return e.ProposedView;
        }

        // Create inner contents.
        // Make view name labels and views.
        private void CreateContent()
        {
            for (int i = 0; i < _mViewList.Length; i++)
            {
                // Make view name label.
                TextLabel viewLabel = new TextLabel(_mViewList[i]);
                viewLabel.Size = new Size(_stage.Size.Width * 0.22f, _stage.Size.Height * 0.080f, 0.0f);
                viewLabel.HorizontalAlignment = "BEGIN";
                viewLabel.VerticalAlignment = "BOTTOM";
                viewLabel.PointSize = 10.0f;
                _contentContainer.AddChild(viewLabel, new TableView.CellPosition(((uint)i / 4) * 2, (uint)i % 4));
                _contentContainer.SetCellAlignment(new TableView.CellPosition(((uint)i / 4) * 2, (uint)i % 4), HorizontalAlignmentType.Center, VerticalAlignmentType.Center);

                // Make view.
                // Set view height and width relative to parent.
                View currentView = CreateView(_mViewList[i]);
                currentView.Focusable = true;
                currentView.Size = new Size(_stage.Size.Width * 0.22f, _stage.Size.Height * 0.12f, 0.0f);
                _contentContainer.AddChild(currentView, new TableView.CellPosition(((uint)i / 4) * 2 + 1, (uint)i % 4));
                _contentContainer.SetCellAlignment(new TableView.CellPosition(((uint)i / 4) * 2 + 1, (uint)i % 4), HorizontalAlignmentType.Center, VerticalAlignmentType.Center);
            }
        }

        // Create views.
        // Create 11 basic views and explain how to use each view.
        private View CreateView(String viewName)
        {
            if (viewName.CompareTo("PushButton") == 0)
            {
                // Create NUI PushButton.
                // This is basic button in NUI.
                // Pressing enter will generate click event when PushButton has focus.
                PushButton pushButton = new PushButton();
                pushButton.LabelText = "Push Button";
                pushButton.UnselectedColor = new Color(43.0f / 255.0f, 145.0f / 255.0f, 175.0f / 255.0f, 1.0f);
                pushButton.SelectedColor = new Color(43.0f / 255.0f, 195.0f / 255.0f, 175.0f / 255.0f, 1.0f);

                // Add click event callback.
                // The source object in parameter is same as pushButton.
                pushButton.Clicked += OnPushButtonClicked;
                return pushButton;
            }
            else if (viewName.CompareTo("Toggle") == 0)
            {
                // Create NUI ToggleButton.
                // Developer can set many different toggle images to ToggleButton.
                // Pressing enter will change ToggleButton image when ToggleButton has focus.
                ToggleButton toggleButton = new ToggleButton();

                // Developer can set toggle images via PropertyArray.
                // In this example, set two different images to ToggleButton.
                toggleButton.StateVisuals = new PropertyArray().Add(new PropertyValue(Current.DirectoryInfo.Resource + "/images/star-highlight.png")).Add(new PropertyValue(Current.DirectoryInfo.Resource + "/images/star-dim.png"));
                return toggleButton;
            }
            else if (viewName.CompareTo("CheckBox") == 0)
            {
                // Create NUI CheckBoxButton.
                // Pressing enter will change CheckBoxButton state when CheckBoxButton has focus.
                CheckBoxButton checkBoxButton = new CheckBoxButton();

                // Set label for CheckBoxButton.
                // Label appears next to CheckBoxButton.
                checkBoxButton.LabelText = "Yes";
                return checkBoxButton;
            }
            else if (viewName.CompareTo("RadioButton") == 0)
            {
                // Create NUI RadioButton.
                // Pressing enter will change RadioButton state when RadioButton has focus.
                // RadioButtons are auto grouping if they have same parent.
                // Only one RadioButton can be selected in group.
                TableView tableView = new TableView(1, 2);

                RadioButton leftRadioButton = new RadioButton();
                // Select leftRadioButton
                leftRadioButton.Selected = true;
                tableView.AddChild(leftRadioButton, new TableView.CellPosition(0, 0));

                RadioButton rightRadioButton = new RadioButton();
                rightRadioButton = new RadioButton();
                tableView.AddChild(rightRadioButton, new TableView.CellPosition(0, 1));
                return tableView;
            }
            else if (viewName.CompareTo("TextLabel") == 0)
            {
                // Create NUI TextLabel.
                // Developer can display single or multi line text.
                // TextLabel can scroll inner text automatically.
                TextLabel textLabel = new TextLabel("This is NUI text label. TextLabel can scroll its inner text automatically!");
                textLabel.EnableAutoScroll = true;
                return textLabel;
            }
            else if (viewName.CompareTo("ImageView") == 0)
            {
                // Create NUI ImageView.
                // ImageView can handle many kinds of image spec.
                // JPEG, PNG, SVG(not animated), GIF and etc.
                ImageView imageView = new ImageView(Current.DirectoryInfo.Resource + "/images/defaultImage.jpg");
                return imageView;
            }
            else if (viewName.CompareTo("AnimateGif") == 0)
            {
                // Create NUI ImageView.
                // Developer can use animate gif.
                ImageView imageView = new ImageView(Current.DirectoryInfo.Resource + "/images/animateGif.gif");
                return imageView;
            }
            else if (viewName.CompareTo("ProgressBar") == 0)
            {
                // Create NUI ProgressBar.
                // ProgressBar has two different value.
                // One is for current progress, and another is for loading progress.
                _progressBar = new ProgressBar();

                // Set Timer to change progress value.
                // Tick will be generated at every 100 milliseconds.
                Timer timer = new Timer(100);
                timer.Tick += OnTick;
                timer.Start();
                return _progressBar;
            }
            else
            {
                return null;
            }
        }

        // Callback for PushButton.
        private bool OnPushButtonClicked(Object source, EventArgs e)
        {
            // Downcast Object to PushButton.
            PushButton button = source as PushButton;
            // Change PushButton's inner text to 'Clicked'.
            button.LabelText = "Clicked";

            return true;
        }

        // Callback for Timer.
        private bool OnTick(Object source, EventArgs e)
        {
            // Update progress value.
            float progress = (float)Math.Round(_progressBar.ProgressValue, 2);

            if (progress == 1.0f)
            {
                // Reset progress value when the value reaches 1.
                _progressBar.ProgressValue = 0.0f;
                _progressBar.SecondaryProgressValue = 0.01f;
            }
            else
            {
                // If progress value doesn't reach 1, plus 0.01f.
                _progressBar.ProgressValue = progress + 0.01f;
                _progressBar.SecondaryProgressValue = progress + 0.21f;
            }

            // If return ture, tick will be generated continuously.
            // If return false, tick will be generated once.
            return true;
        }

        // Callback for Stage Key Event
        private void OnStageKeyEvent(object source, Stage.KeyEventArgs e)
        {
            if (e.Key.State == Key.StateType.Up)
            {
                if (e.Key.KeyPressedName.CompareTo("Escape") == 0 || e.Key.KeyPressedName.CompareTo("XF86Back") == 0)
                {
                    Exit();
                }
            }
        }

        protected override void OnCreate()
        {
            base.OnCreate();
            Initialize();
        }

        [STAThread]
        private static void Main(string[] args)
        {
            Example example = new Example();
            example.Run(args);
        }
    }
}
