#include <tizen.h>
#include "myservicewidget.h"
#include <app_preference.h>

#define PREF_KEY_UNREAD_EVENTS "unread_events" // the preference key representing the number of shake events that happened since last opening of the launcher
#define PREF_KEY_TOTAL_EVENTS "total_events" // the preference key representing the total number of shake events that occurred since the installation of the package
#define MYSERVICELAUNCHER_APP_ID "org.example.myservicelauncher" // an ID of the UI application of our package
#define MAX_LABEL_LENGTH 256

typedef struct widget_instance_data {
	Evas_Object *win;
	Evas_Object *conform;
	Evas_Object *label_unread_events;
	Evas_Object *label_total_events;
	Evas_Object *button;
} widget_instance_data_s;

void button_clicked_cb(void *data, Evas_Object *obj, void *event_info)
{
	app_control_h app_control = NULL;
	if (app_control_create(&app_control)== APP_CONTROL_ERROR_NONE)
	{
		if ((app_control_set_app_id(app_control, MYSERVICELAUNCHER_APP_ID) == APP_CONTROL_ERROR_NONE)
			&& (app_control_send_launch_request(app_control, NULL, NULL) == APP_CONTROL_ERROR_NONE))
		{
			dlog_print(DLOG_INFO, LOG_TAG, "App launch request sent!");
		}
		else
		{
			dlog_print(DLOG_ERROR, LOG_TAG, "App launch request sending failed!");
		}
		if (app_control_destroy(app_control) == APP_CONTROL_ERROR_NONE)
		{
			dlog_print(DLOG_INFO, LOG_TAG, "App control destroyed.");
		}
	}
	else
	{
		dlog_print(DLOG_ERROR, LOG_TAG, "App control creation failed!");
	}
}

void initialize_preferences()
{
	bool unread_exists = false;
	bool read_exists = false;
	if ((preference_is_existing(PREF_KEY_UNREAD_EVENTS, &unread_exists) == PREFERENCE_ERROR_NONE)
		&&(preference_is_existing(PREF_KEY_TOTAL_EVENTS, &read_exists) == PREFERENCE_ERROR_NONE))
	{
		if (unread_exists == false)
		{
			if (preference_set_int(PREF_KEY_UNREAD_EVENTS, 0) == PREFERENCE_ERROR_NONE)
			{
				dlog_print(DLOG_INFO, LOG_TAG, "Unread events preference was not present, setting to default false value...");
			}
			else
			{
				dlog_print(DLOG_ERROR, LOG_TAG, "Error creating unread events preference!");
			}
		}
		if (read_exists == false)
		{
			if (preference_set_int(PREF_KEY_TOTAL_EVENTS, 0) == PREFERENCE_ERROR_NONE)
			{
				dlog_print(DLOG_INFO, LOG_TAG, "Total events preference was not present, setting to default false value...");
			}
			else
			{
				dlog_print(DLOG_ERROR, LOG_TAG, "Error creating total events preference!");
			}
		}

	}
	else
	{
		dlog_print(DLOG_INFO, LOG_TAG, "Error creating event preferences!");
	}
}

bool refresh_unread_events_label_cb(widget_context_h context, void *data)
{
	widget_instance_data_s *wid = NULL;
	widget_app_context_get_tag(context,(void**)&wid);

	int *no_events = (int*)data;

	char label_text[MAX_LABEL_LENGTH];
	snprintf(label_text, MAX_LABEL_LENGTH, "Unread events: %d", *no_events);
	elm_object_text_set(wid->label_unread_events, label_text);

	return true;
}

bool refresh_total_events_label_cb(widget_context_h context, void *data)
{
	widget_instance_data_s *wid = NULL;
	widget_app_context_get_tag(context,(void**)&wid);

	int *no_events = (int*)data;

	char label_text[MAX_LABEL_LENGTH];
	snprintf(label_text, MAX_LABEL_LENGTH, "Total events: %d", *no_events);
	elm_object_text_set(wid->label_total_events, label_text);

	return true;
}

void on_unread_events_changed_cb (const char *key, void *user_data)
{
	int value = 0;

	if (preference_get_int(key, &value) != PREFERENCE_ERROR_NONE)
	{
		dlog_print(DLOG_ERROR, LOG_TAG, "Error reading preference!");
		value = 0;
	}
	if (widget_app_foreach_context(refresh_unread_events_label_cb, &value) == WIDGET_ERROR_NONE)
	{
		dlog_print(DLOG_INFO, LOG_TAG, "Refreshing UI for all widget instances.");
	}
	else
	{
		dlog_print(DLOG_ERROR, LOG_TAG, "Refreshing UI failed!");
	}
}

void on_total_events_changed_cb (const char *key, void *user_data)
{
	int value = 0;

	if (preference_get_int(key, &value) != PREFERENCE_ERROR_NONE)
	{
		dlog_print(DLOG_ERROR, LOG_TAG, "Error reading preference!");
		value = 0;
	}
	if (widget_app_foreach_context(refresh_total_events_label_cb, &value) == WIDGET_ERROR_NONE)
	{
		dlog_print(DLOG_INFO, LOG_TAG, "Refreshing UI for all widget instances.");
	}
	else
	{
		dlog_print(DLOG_ERROR, LOG_TAG, "Refreshing UI failed!");
	}
}

static int
widget_instance_create(widget_context_h context, bundle *content, int w, int h, void *user_data)
{
	widget_instance_data_s *wid = (widget_instance_data_s*) malloc(sizeof(widget_instance_data_s));
	if (wid == NULL)
	{
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to allocate widget instance data!");
		return WIDGET_ERROR_FAULT;
	}

	int ret = WIDGET_ERROR_NONE;

	if (content != NULL)
	{
		/* Recover the previous status with the bundle object. */
	}

	/* Window */
	ret = widget_app_get_elm_win(context, &wid->win);
	if (ret != WIDGET_ERROR_NONE)
	{
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to get window. err = %d", ret);
		free(wid);
		return WIDGET_ERROR_FAULT;
	}

	evas_object_resize(wid->win, w, h);

	int unread_events_number = 0, total_events_number = 0;

	char label_text_unread[MAX_LABEL_LENGTH] = {0};
	char label_text_total[MAX_LABEL_LENGTH] = {0};

	if (preference_get_int(PREF_KEY_UNREAD_EVENTS, &unread_events_number) == PREFERENCE_ERROR_NONE)
	{
		dlog_print(DLOG_INFO, LOG_TAG, "Geting unread events preferences...");
		snprintf(label_text_unread, MAX_LABEL_LENGTH, "Unread events: %d", unread_events_number);
	}
	else
	{
		dlog_print(DLOG_ERROR, LOG_TAG, "Error reading unread events preferences!");
		snprintf(label_text_unread, MAX_LABEL_LENGTH, "Error reading unread events number!");
	}

	if (preference_get_int(PREF_KEY_TOTAL_EVENTS, &total_events_number) == PREFERENCE_ERROR_NONE)
	{
		dlog_print(DLOG_INFO, LOG_TAG, "Geting total events preferences...");
		snprintf(label_text_total, MAX_LABEL_LENGTH, "Total events: %d", total_events_number);
	}
	else
	{
		dlog_print(DLOG_ERROR, LOG_TAG, "Error reading total events preferences!");
		snprintf(label_text_total, MAX_LABEL_LENGTH, "Error reading total events number!");
	}

	Evas_Object *bg = elm_bg_add(wid->win);
	evas_object_size_hint_weight_set(bg, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_color_set(bg, 106, 205, 216, 32);
	elm_win_resize_object_add(wid->win, bg);
	evas_object_show(bg);

	wid->conform = elm_conformant_add(bg);
	evas_object_size_hint_weight_set(wid->conform, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_win_resize_object_add(wid->win, wid->conform);
	evas_object_show(wid->conform);

	Evas_Object *box = elm_box_add(wid->conform);
	elm_box_horizontal_set(box, EINA_FALSE);
	elm_box_align_set(box, 0.5, 0.5);
	elm_object_content_set(wid->conform, box);
	evas_object_show(box);

	/* Label - unread events*/
	wid->label_unread_events = elm_label_add(wid->conform);
	evas_object_size_hint_min_set(wid->label_unread_events, w / 1.5, h / 4);
	evas_object_size_hint_max_set(wid->label_unread_events, w, h / 4);
	elm_box_pack_end(box, wid->label_unread_events);
	evas_object_show(wid->label_unread_events);
	elm_object_text_set(wid->label_unread_events, label_text_unread);

	/* Label - total events*/
	wid->label_total_events = elm_label_add(wid->conform);
	evas_object_size_hint_min_set(wid->label_total_events, w / 1.5, h / 4);
	evas_object_size_hint_max_set(wid->label_total_events, w, h / 4);
	elm_box_pack_end(box, wid->label_total_events);
	evas_object_show(wid->label_total_events);
	elm_object_text_set(wid->label_total_events, label_text_total);

	/* Button */
	wid->button = elm_button_add(wid->conform);
	evas_object_smart_callback_add(wid->button, "clicked", button_clicked_cb, NULL);
	evas_object_size_hint_min_set(wid->button, w / 2, h / 4);
	evas_object_size_hint_max_set(wid->button, w / 2, h / 4);
	evas_object_size_hint_weight_set(wid->button, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_fill_set(wid->button, EVAS_HINT_FILL, EVAS_HINT_FILL);
	elm_object_text_set(wid->button, "Open Launcher");
	elm_box_pack_end(box, wid->button);
	evas_object_show(wid->button);
	
	/* Show window after base gui is set up */
	evas_object_show(wid->win);

	widget_app_context_set_tag(context, wid);
	return WIDGET_ERROR_NONE;
}

static int
widget_instance_destroy(widget_context_h context, widget_app_destroy_type_e reason, bundle *content, void *user_data)
{
	widget_instance_data_s *wid = NULL;
	widget_app_context_get_tag(context,(void**)&wid);

	if (wid->win)
		evas_object_del(wid->win);

	free(wid);

	return WIDGET_ERROR_NONE;
}

static int
widget_instance_pause(widget_context_h context, void *user_data)
{
	/* Take necessary actions when widget instance becomes invisible. */
	return WIDGET_ERROR_NONE;

}

static int
widget_instance_resume(widget_context_h context, void *user_data)
{
	/* Take necessary actions when widget instance becomes visible. */
	return WIDGET_ERROR_NONE;
}

static int
widget_instance_update(widget_context_h context, bundle *content,
                           int force, void *user_data)
{
	/* Take necessary actions when widget instance should be updated. */
	return WIDGET_ERROR_NONE;
}

static int
widget_instance_resize(widget_context_h context, int w, int h, void *user_data)
{
	/* Take necessary actions when the size of widget instance was changed. */
	return WIDGET_ERROR_NONE;
}

static void
widget_app_lang_changed(app_event_info_h event_info, void *user_data)
{
	/* APP_EVENT_LANGUAGE_CHANGED */
	char *locale = NULL;
	app_event_get_language(event_info, &locale);
	elm_language_set(locale);
	free(locale);
}

static void
widget_app_region_changed(app_event_info_h event_info, void *user_data)
{
	/* APP_EVENT_REGION_FORMAT_CHANGED */
}

static widget_class_h
widget_app_create(void *user_data)
{
	/* Hook to take necessary actions before main event loop starts.
	   Initialize UI resources.
	   Make a class for widget instance.
	*/
	app_event_handler_h handlers[5] = {NULL, };

	widget_app_add_event_handler(&handlers[APP_EVENT_LANGUAGE_CHANGED],
		APP_EVENT_LANGUAGE_CHANGED, widget_app_lang_changed, user_data);
	widget_app_add_event_handler(&handlers[APP_EVENT_REGION_FORMAT_CHANGED],
		APP_EVENT_REGION_FORMAT_CHANGED, widget_app_region_changed, user_data);

	widget_instance_lifecycle_callback_s ops = {
		.create = widget_instance_create,
		.destroy = widget_instance_destroy,
		.pause = widget_instance_pause,
		.resume = widget_instance_resume,
		.update = widget_instance_update,
		.resize = widget_instance_resize,
	};

	initialize_preferences();



	if ((preference_set_changed_cb(PREF_KEY_UNREAD_EVENTS, on_unread_events_changed_cb, NULL) == PREFERENCE_ERROR_NONE)
	&& (preference_set_changed_cb(PREF_KEY_TOTAL_EVENTS, on_total_events_changed_cb, NULL) == PREFERENCE_ERROR_NONE))
	{
		dlog_print(DLOG_INFO, LOG_TAG, "Preference change callbacks added!");
	}
	else
	{
		dlog_print(DLOG_ERROR, LOG_TAG, "Error occurred when setting preference change callbacks!");
	}

	return widget_app_class_create(ops, user_data);
}

static void
widget_app_terminate(void *user_data)
{
	/* Release all resources. */
}

int
main(int argc, char *argv[])
{
	widget_app_lifecycle_callback_s ops = {0,};
	int ret = WIDGET_ERROR_NONE;

	ops.create = widget_app_create;
	ops.terminate = widget_app_terminate;

	ret = widget_app_main(argc, argv, &ops, NULL);
	if (ret != WIDGET_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "widget_app_main() is failed. err = %d", ret);
	}

	return ret;
}


