/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

using System;
using System.Collections.Generic;
using Xamarin.Forms;

namespace MusicPlayerUI
{
    /// <summary>
    /// The main page of the MusicPlayer application.
    /// </summary>
    public class MainPage : TabbedPage
    {
        /// <summary>
        /// Initializes a new instance of the <see cref="MainPage"/> class.
        /// </summary>
        public MainPage() : base()
        {
            Title = "Music Player";
            IsVisible = true;

            // Create artists content page.
            ContentPage artistsPage = new ContentPage();
            artistsPage.Title = "Artists";
            artistsPage.Content = CreateArtistsPageContent();
            Children.Add(artistsPage);

            // Create albums content page.
            ContentPage albumsPage = new ContentPage();
            albumsPage.Title = "Albums";
            albumsPage.Content = CreateAlbumsPageContent();
            Children.Add(albumsPage);

            // Create songs content page.
            ContentPage songsPage = new ContentPage();
            songsPage.Title = "Songs";
            songsPage.Content = CreateSongsPageContent();
            Children.Add(songsPage);
        }

        /// <summary>
        /// Create a page for artists.
        /// </summary>
        /// <returns>Return StackLayout object.</returns>
        private StackLayout CreateArtistsPageContent()
        {
            // Add new ListView.
            ListView listViewArtists = CreateListView();
            AddListItems(listViewArtists, "Artists Title");

            // Create player panel.
            StackLayout panel = CreatePlayerPanel();

            StackLayout stackLayout = new StackLayout()
            {
                IsVisible = true,
                Children =
                {
                    listViewArtists,
                    panel,
                },
            };

            return stackLayout;
        }

        /// <summary>
        /// Create a page for albums.
        /// </summary>
        /// <returns>Return StackLayout object.</returns>
        private StackLayout CreateAlbumsPageContent()
        {
            // Add new ListView.
            ListView listViewAlbums = CreateListView();
            AddListItems(listViewAlbums, "Albums Title");

            // Create new StackLayout.
            StackLayout stackLayout = new StackLayout
            {
                Children =
                {
                    listViewAlbums,
                },
            };

            return stackLayout;
        }

        /// <summary>
        /// Create a page for songs.
        /// </summary>
        /// <returns>Return StackLayout object.</returns>
        private StackLayout CreateSongsPageContent()
        {
            // Add new ListView.
            ListView listViewSongs = CreateListView();
            AddListItems(listViewSongs, "Song Title");

            StackLayout stackLayout = new StackLayout
            {
                Children =
                {
                    listViewSongs,
                },
            };

            return stackLayout;
        }

        /// <summary>
        /// Create a player panel.
        /// </summary>
        /// <returns>Return StackLayout object.</returns>
        private StackLayout CreatePlayerPanel()
        {
            // Add new Image.
            Image imgIcon = new Image()
            {
                Source = "icon/music_player_icon.png",
            };

            StackLayout marginLayoutFirst = new StackLayout
            {
                WidthRequest = 20,
            };

            // Add new Label.
            Label label = new Label()
            {
                Text = "Artists - Songs",
                VerticalTextAlignment = TextAlignment.Center,
                FontSize = 36,
            };

            StackLayout marginLayoutSecond = new StackLayout
            {
                WidthRequest = 220,
            };

            // Add new image for player icon.
            Image playImg = new Image()
            {
                Source = "icon/music_player_play.png",
            };

            StackLayout stackLayout = new StackLayout()
            {
                Padding = new Thickness(20),
                IsVisible = true,

                // Set height.
                HeightRequest = 80,
                Orientation = StackOrientation.Horizontal,
                Children =
                {
                    imgIcon,
                    marginLayoutFirst,
                    label,
                    marginLayoutSecond,
                    playImg,
                },
            };

            return stackLayout;
        }

        /// <summary>
        /// This is class is defined for list view source data template.
        /// </summary>
        class ListData
        {
            /// <summary>
            /// Initializes a new instance of the <see cref="ListData"/> class.
            /// </summary>
            /// <param name="name">It is initialization parameter.</param>
            public ListData(string name)
            {
                this.Name = name;
            }

            public string Name { private set; get; }
        }

        /// <summary>
        /// Create a custom ListView.
        /// </summary>
        /// <returns>Return ListView Object.</returns>
        private ListView CreateListView()
        {
            return new ListView()
            {
                // Create and define the data template.
                ItemTemplate = new DataTemplate(() =>
                {
                    // Add new Label to show name.
                    Label nameLabel = new Label();

                    // Set text color.
                    nameLabel.TextColor = Color.Black;

                    // Set font size.
                    nameLabel.FontSize = 40;

                    // set label height.
                    nameLabel.HeightRequest = 60;
                    nameLabel.SetBinding(Label.TextProperty, "Name");

                    // Create ViewCell object.
                    return new ViewCell
                    {
                        // Add new StackLayout.
                        View = new StackLayout
                        {
                            // Set padding.
                            Padding = new Thickness(10),
                            Orientation = StackOrientation.Horizontal,
                            Children =
                            {
                                nameLabel,
                            },
                        },
                    };
                }),

                // Set row height.
                RowHeight = 80,
            };
        }

        /// <summary>
        /// Add list item for a <see cref="ListView"/> class instance.
        /// </summary>
        /// <param name="listView">The target into which item add.</param>
        /// <param name="text">The list data which to be set.</param>
        private void AddListItems(ListView listView, string text)
        {
            var dataList = new List<ListData>();
            for (int i = 0; i < 16; i++)
            {
                // Set text into list view.
                dataList.Add(new ListData(text));
            }

            // Set data for ListView ItemSource.
            listView.ItemsSource = dataList;
        }
    }
}
