/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "user_callbacks.h"
#include "main.h"

#include <sensor.h>

#define BUFLEN 200

sensor_listener_h listener;
Evas_Object *start, *stop, *event_label;

void on_sensor_event(sensor_h sensor, sensor_event_s *event, void *user_data)
{
    // Select a specific sensor with a sensor handle
    // This example uses sensor type, assuming there is only 1 sensor for each type
    sensor_type_e type;
    sensor_get_type(sensor, &type);

    char event_details[BUFLEN];
    switch (type) {
    case SENSOR_LIGHT:
        snprintf(event_details, BUFLEN, "Light: %.2f", event->values[0]);
        elm_object_text_set(event_label, event_details);
        dlog_print(DLOG_DEBUG, LOG_TAG, "Light event received");
        break;

    default:
        dlog_print(DLOG_ERROR, LOG_TAG, "Not an Light event");
    }
}


void _sensor_accuracy_changed_cb(sensor_h sensor, unsigned long long timestamp,
                                 sensor_data_accuracy_e accuracy, void *data)
{
    PRINT_MSG("Sensor accuracy change callback invoked");
    dlog_print(DLOG_DEBUG, LOG_TAG, "Sensor accuracy change callback invoked");
}


void _sensor_start_cb(void *data, Evas_Object *obj, void *event_info)
{
    void *user_data = NULL;

    // Retrieving a Sensor, Check Sensor Support
    sensor_type_e type = SENSOR_LIGHT;
    sensor_h sensor;

    bool supported;
    int error = sensor_is_supported(type, &supported);
    if (error != SENSOR_ERROR_NONE) {
        PRINT_MSG("sensor_is_supported error: %d", error);
        dlog_print(DLOG_ERROR, LOG_TAG, "sensor_is_supported error: %d", error);
        return;
    }

    dlog_print(DLOG_DEBUG, LOG_TAG, "Light is%s supported", supported ? "" : " not");
    PRINT_MSG("Light is%s supported", supported ? "" : " not");

    error = sensor_get_default_sensor(type, &sensor);
    if (error != SENSOR_ERROR_NONE) {
        PRINT_MSG("sensor_get_default_sensor error: %d", error);
        dlog_print(DLOG_ERROR, LOG_TAG, "sensor_get_default_sensor error: %d", error);
        return;
    }

    dlog_print(DLOG_DEBUG, LOG_TAG, "sensor_get_default_sensor");

    // Registering a Sensor Event
    error = sensor_create_listener(sensor, &listener);
    if (error != SENSOR_ERROR_NONE) {
        PRINT_MSG("sensor_create_listener error: %d", error);
        dlog_print(DLOG_ERROR, LOG_TAG, "sensor_create_listener error: %d", error);
        return;
    }

    dlog_print(DLOG_DEBUG, LOG_TAG, "sensor_create_listener");

    int min_interval = 0;
    error = sensor_get_min_interval(sensor, &min_interval);
    if (error != SENSOR_ERROR_NONE) {
        PRINT_MSG("sensor_get_min_interval error: %d", error);
        dlog_print(DLOG_ERROR, LOG_TAG, "sensor_get_min_interval error: %d", error);
        return;
    }

    PRINT_MSG("Minimum interval of the sensor: %d", min_interval);
    dlog_print(DLOG_DEBUG, LOG_TAG, "sensor_get_min_interval");

    error = sensor_listener_set_event_cb(listener, min_interval, on_sensor_event, user_data);
    if (error != SENSOR_ERROR_NONE) {
        PRINT_MSG("sensor_listener_set_event_cb error: %d", error);
        dlog_print(DLOG_ERROR, LOG_TAG, "sensor_listener_set_event_cb error: %d", error);
        return;
    }

    dlog_print(DLOG_DEBUG, LOG_TAG, "sensor_listener_set_event_cb");

    // Registering the Accuracy Changed Callback
    error = sensor_listener_set_accuracy_cb(listener, _sensor_accuracy_changed_cb, user_data);
    if (error != SENSOR_ERROR_NONE) {
        PRINT_MSG("sensor_listener_set_accuracy_cb error: %d", error);
        dlog_print(DLOG_ERROR, LOG_TAG, "sensor_listener_set_accuracy_cb error: %d", error);
        return;
    }

    dlog_print(DLOG_DEBUG, LOG_TAG, "sensor_listener_set_accuracy_cb");

    error = sensor_listener_set_interval(listener, 100);
    if (error != SENSOR_ERROR_NONE) {
        PRINT_MSG("sensor_listener_set_interval error: %d", error);
        dlog_print(DLOG_ERROR, LOG_TAG, "sensor_listener_set_interval error: %d", error);
        return;
    }

    dlog_print(DLOG_DEBUG, LOG_TAG, "sensor_listener_set_intervals");

    error = sensor_listener_set_option(listener, SENSOR_OPTION_ALWAYS_ON);
    if (error != SENSOR_ERROR_NONE) {
        PRINT_MSG("sensor_listener_set_option error: %d", error);
        dlog_print(DLOG_ERROR, LOG_TAG, "sensor_listener_set_option error: %d", error);
        return;
    }

    dlog_print(DLOG_DEBUG, LOG_TAG, "sensor_listener_set_option");

    error = sensor_listener_start(listener);
    if (error != SENSOR_ERROR_NONE) {
        PRINT_MSG("sensor_listener_start error: %d", error);
        dlog_print(DLOG_ERROR, LOG_TAG, "sensor_listener_start error: %d", error);
        return;
    }

    dlog_print(DLOG_DEBUG, LOG_TAG, "sensor_listener_start");

    sensor_event_s event;
    error = sensor_listener_read_data(listener, &event);
    if (error != SENSOR_ERROR_NONE) {
        PRINT_MSG("sensor_listener_read_data error: %d", error);
        dlog_print(DLOG_ERROR, LOG_TAG, "sensor_listener_read_data error: %d", error);
        return;
    }

    switch (type) {
    case SENSOR_LIGHT:
        PRINT_MSG("Light values: %.2f", event.values[0]);
        dlog_print(DLOG_DEBUG, LOG_TAG, "Light values");
        break;

    default:
        dlog_print(DLOG_ERROR, LOG_TAG, "Not an Light event");
    }

    dlog_print(DLOG_DEBUG, LOG_TAG, "sensor_listener_read_data");

    float min_range = 0.0;
    float max_range = 0.0;

    error = sensor_get_type(sensor, &type);
    if (error != SENSOR_ERROR_NONE) {
        PRINT_MSG("sensor_get_type error: %d", error);
        dlog_print(DLOG_ERROR, LOG_TAG, "sensor_get_type error: %d", error);
        return;
    }

    /**/

    dlog_print(DLOG_DEBUG, LOG_TAG, "sensor_get_type");

    error = sensor_get_min_range(sensor, &min_range);
    if (error != SENSOR_ERROR_NONE) {
        PRINT_MSG("sensor_get_min_range error: %d", error);
        dlog_print(DLOG_ERROR, LOG_TAG, "sensor_get_min_range error: %d", error);
        return;
    }

    PRINT_MSG("Minimum range of the sensor: %f", min_range);
    dlog_print(DLOG_DEBUG, LOG_TAG, "sensor_get_min_range");

    error = sensor_get_max_range(sensor, &max_range);
    if (error != SENSOR_ERROR_NONE) {
        PRINT_MSG("sensor_get_max_range error: %d", error);
        dlog_print(DLOG_ERROR, LOG_TAG, "sensor_get_max_range error: %d", error);
        return;
    }

    PRINT_MSG("Maximum range of the sensor: %f", max_range);
    dlog_print(DLOG_DEBUG, LOG_TAG, "sensor_get_max_range");

    elm_object_disabled_set(start, EINA_TRUE);
    elm_object_disabled_set(stop, EINA_FALSE);
}

void _sensor_stop_cb(void *data, Evas_Object *obj, void *event_info)
{
    int error = sensor_listener_unset_event_cb(listener);
    if (error != SENSOR_ERROR_NONE) {
        PRINT_MSG("sensor_listener_unset_event_cb error: %d", error);
        dlog_print(DLOG_ERROR, LOG_TAG, "sensor_listener_unset_event_cb error: %d", error);
    }

    error = sensor_listener_stop(listener);
    if (error != SENSOR_ERROR_NONE) {
        PRINT_MSG("sensor_listener_stop error: %d", error);
        dlog_print(DLOG_ERROR, LOG_TAG, "sensor_listener_stop error: %d", error);
    }

    error = sensor_destroy_listener(listener);
    if (error != SENSOR_ERROR_NONE) {
        PRINT_MSG("sensor_destroy_listener error: %d", error);
        dlog_print(DLOG_ERROR, LOG_TAG, "sensor_destroy_listener error: %d", error);
    }

    elm_object_disabled_set(start, EINA_FALSE);
    elm_object_disabled_set(stop, EINA_TRUE);
}

void create_buttons_in_main_window(appdata_s *ad)
{
    Evas_Object *display = _create_new_cd_display(ad, "Sensor", _pop_cb);

    start = _new_button(ad, display, "GET READING", _sensor_start_cb);
    stop = _new_button(ad, display, "STOP SENSOR", _sensor_stop_cb);

    event_label = elm_label_add(display);
    elm_object_text_set(event_label, "Sensor event data");
    elm_box_pack_end(display, event_label);
    evas_object_show(event_label);

    elm_object_disabled_set(stop, EINA_TRUE);
}
