#include "img-mods.h"

#include <camera.h>
#include <image_util.h>
#include <dlog.h>
#include <tizen.h>
#include <media_content.h>

#ifndef LOG_TAG
#define LOG_TAG "ImageMods"
#endif

// Universal function to clip int values to the range [0..255]
// and convert it to unsigned char as stored in image buffers.
unsigned char _clip_n_convert(int val)
{
	if (val > 255)
	{
		return 255;
	}
	if (val < 0)
	{
		return 0;
	}
	return (unsigned char)val;
}

// Function for modifying luminance data of the image
// to make it look like the old picture with overexposed highlights.
void _luma_mod_blow_highlights(unsigned char* data, uint64_t size)
{
	uint64_t i = 0;

	for (i = 0; i < size; i++)
	{
		int Yval = (int)(data[i]);
		Yval = (Yval*15)/10; // 150% of the original luma size
		data[i] = _clip_n_convert(Yval);

	}
}

// Function for modifying chrominance data of the image
// to give it the sepia tone.
void _chroma_mod_sepia(unsigned char* data, uint64_t size)
{
	uint64_t i = 0;

	for (i = 0; i < size; i++)
	{
		int Cval = (int)(data[i]);

		if (i%2 == 0) //Even byte reflects Cb chroma value
		{
			Cval = 114; //Setting the Cb value to look like sepia
		}
		else //Odd byte reflects Cr chroma value
		{
			Cval = 144; //Setting the Cr value to look like sepia
		}
		data[i] = (unsigned char)Cval;
	}
}

// Optional function for updating media database on the device
// instantly after taking a picture.
// It is not necessary for capturing the picture itself,
// but only to reflect the newly taken picture in the apps such as Gallery.
void _update_media_db(const char * filename)
{
	if (media_content_connect() == MEDIA_CONTENT_ERROR_NONE)
	{
		if (media_content_scan_file(filename) == MEDIA_CONTENT_ERROR_NONE)
		{
			dlog_print(DLOG_INFO, LOG_TAG, "Media database updated with the new file.");
		}
		media_content_disconnect();
	}
}

// Callback function to be executed on each preview frame captured
// for modifying the NV12 preview frame
void _camera_preview_callback(camera_preview_data_s *frame, void *user_data)
{
	// We take the luminance (Y) component
	// and modify it to look like the old overexposed image:
	_luma_mod_blow_highlights(frame->data.double_plane.y, frame->data.double_plane.y_size);

	// We take the chrominance component and modify it to look like sepia:
	_chroma_mod_sepia(frame->data.double_plane.uv, frame->data.double_plane.uv_size);
}

// Function preparing camera:
// First, changing capture format to NV12 to easily modify the image in the NV12 colorspace
// using the same filters as the preview frame.
// Second, setting custom preview callback to grab and modify the preview frames
int prepare_camera(camera_h camera)
{
	if ((camera_set_capture_format(camera, CAMERA_PIXEL_FORMAT_NV12) == CAMERA_ERROR_NONE)
		&& (camera_set_preview_cb(camera, _camera_preview_callback, NULL) == CAMERA_ERROR_NONE))
	{
		dlog_print(DLOG_INFO, LOG_TAG, "The camera has been prepared for an image modification.");
		return 0;
	}
	else
	{
		dlog_print(DLOG_ERROR, LOG_TAG, "Error occurred when preparing camera!");
		return -1;
	}
}

// Wrapper function for modifying the final picture captured
int modify_final_image(camera_image_data_s *image)
{
	if (image->format == CAMERA_PIXEL_FORMAT_NV12)
	{
		dlog_print(DLOG_INFO, LOG_TAG, "We've got NV12 data - data [%p], length [%d], width [%d], height [%d]",
			image->data, image->size, image->width, image->height);
		int luma_size = image->width * image->height;
		int chroma_size = luma_size/2;
		_luma_mod_blow_highlights(image->data, luma_size);
		_chroma_mod_sepia((image->data)+luma_size, chroma_size);
		dlog_print(DLOG_INFO, LOG_TAG, "The modifications have been applied.");
		return 0;
	}
	else
	{
		dlog_print(DLOG_ERROR, LOG_TAG, "Wrong image format!");
		return -1;
	}
}

int finalize_taking_picture(camera_image_data_s *image, const char *filename)
{
	if (image->format == CAMERA_PIXEL_FORMAT_NV12
		&& image_util_encode_jpeg(image->data, image->width, image->height, IMAGE_UTIL_COLORSPACE_NV12, 100, filename) == IMAGE_UTIL_ERROR_NONE)
	{
		dlog_print(DLOG_INFO, LOG_TAG, "Image saved successfully.");

		// Updating media db...
		// This line is optional, use it when you want to update media database
		// instantly after taking a picture.
		_update_media_db(filename);

		return 0;
	}
	else
	{
		dlog_print(DLOG_ERROR, LOG_TAG, "Error occurred when saving image!");
		return -1;
	}
}
