#include "videoplayer.h"
#include <player.h>

typedef struct appdata {
	Evas_Object *win;
	Evas_Object *conform;
	Evas_Object *label;
    player_h player;
    Evas_Object *video_rect;
} appdata_s;

const char* file_name[] = { "sampleH263.3gp", "color_short.mp4" };

static void
win_back_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = data;
	/* Let window go to hide state. */
	elm_win_iconified_set(ad->win, EINA_TRUE);
}

// Get full path of resource file
static inline const char *
get_resource_path(const char *file_path)
{
    static char absolute_path[PATH_MAX] = {'\0'};
    static char *res_path_buff = NULL;
    if(res_path_buff == NULL)
    {
        res_path_buff = app_get_resource_path();
    }

    snprintf(absolute_path, PATH_MAX, "%s%s", res_path_buff, file_path);
    return absolute_path;
}

// Create Image for screen
static Evas_Object *
video_rect_add(Evas_Object *parent)
{
    Evas *evas = evas_object_evas_get(parent);
    Evas_Object *image = evas_object_image_filled_add(evas);
    evas_object_size_hint_weight_set(image, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
    evas_object_size_hint_align_set(image, EVAS_HINT_FILL, EVAS_HINT_FILL);
    evas_object_show(image);
    return image;
}

// Get player state
static player_state_e
get_player_state(player_h player)
{
    player_state_e state;
    player_get_state(player, &state);
    return state;
}

// Play completed event function
static void
on_player_completed(void *data)
{
	player_h player = data;
    if(player)
        player_stop(player);
}

// Create player
static player_h
create_player()
{
    player_h player;

    player_create(&player);
    player_set_sound_type(player, SOUND_TYPE_MEDIA);
    player_set_volume(player, 1.0, 1.0);
    player_set_looping(player, false);
    player_set_completed_cb(player, on_player_completed, player);

    return player;
}

// Stop play
static void
stop_player(void *data, Evas_Object *obj, void *event_info)
{
    appdata_s *ad = data;

    if( get_player_state(ad->player) == PLAYER_STATE_PLAYING || get_player_state(ad->player) == PLAYER_STATE_PAUSED)
        player_stop(ad->player);
}

// Load video file to Player
static void
prepare_player(appdata_s* ad, int index)
{
	player_stop(ad->player);
    // Close file
    player_unprepare(ad->player);

    const char* file = file_name[index];
    // Get full path of resource file
    const char *res_path = get_resource_path(file);

    // Load file
    player_set_uri(ad->player, res_path);
    player_set_display(ad->player, PLAYER_DISPLAY_TYPE_EVAS, GET_DISPLAY(ad->video_rect));
    player_set_display_mode(ad->player, PLAYER_DISPLAY_MODE_FULL_SCREEN);
    // Prepare play
    int result = player_prepare(ad->player);
    dlog_print(DLOG_INFO, "tag", "File load : %d", result);
}

// Start play
static void
start_player(void *data, Evas_Object *obj, void *event_info)
{
    appdata_s *ad = data;

    if( get_player_state(ad->player) != PLAYER_STATE_PLAYING)
        player_start(ad->player);
}

// Pause play
static void
pause_player(void *data, Evas_Object *obj, void *event_info)
{
    appdata_s *ad = data;

    if( get_player_state(ad->player) == PLAYER_STATE_PLAYING )
        player_pause(ad->player);
}

static void
btn_load_file1(void *data, Evas_Object *obj, void *event_info)
{
    appdata_s *ad = data;
    // Load file to Player
    prepare_player(ad, 0);
}

static void
btn_load_file2(void *data, Evas_Object *obj, void *event_info)
{
    appdata_s *ad = data;
    // Load file to Player
    prepare_player(ad, 1);
}

static Evas_Object *
my_button_add(Evas_Object *parent, const char *text, Evas_Smart_Cb cb, void *cb_data)
{
	Evas_Object *btn;

	btn = elm_button_add(parent);
	elm_object_text_set(btn, text);
	evas_object_smart_callback_add(btn, "clicked", cb, cb_data);

	return btn;
}

static void
my_table_pack(Evas_Object *table, Evas_Object *child, int x, int y, int w, int h)
{
   evas_object_size_hint_align_set(child, EVAS_HINT_FILL, EVAS_HINT_FILL);
   evas_object_size_hint_weight_set(child, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
   elm_table_pack(table, child, x, y, w, h);
   evas_object_show(child);
}

static void
create_base_gui(appdata_s *ad)
{
	/* set up policy to exit when last window is closed */
	elm_policy_set(ELM_POLICY_QUIT, ELM_POLICY_QUIT_LAST_WINDOW_CLOSED);
	/* Window */
	ad->win = elm_win_util_standard_add(PACKAGE, PACKAGE);
	elm_win_autodel_set(ad->win, EINA_TRUE);

	int rots[4] = { 0, 90, 180, 270 };
	elm_win_wm_rotation_available_rotations_set(ad->win, (const int *)(&rots), 4);

	eext_object_event_callback_add(ad->win, EEXT_CALLBACK_BACK, win_back_cb, ad);

    /* Conformant */
    /*ad->conform = elm_conformant_add(ad->win);
    elm_win_indicator_mode_set(ad->win, ELM_WIN_INDICATOR_SHOW);
    elm_win_indicator_opacity_set(ad->win, ELM_WIN_INDICATOR_OPAQUE);
    evas_object_size_hint_weight_set(ad->conform, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
    elm_win_resize_object_add(ad->win, ad->conform);
    evas_object_show(ad->conform);*/

    /* Label*/
    /*ad->label = elm_label_add(ad->conform);
    elm_object_text_set(ad->label, "<align=center>Hello EFL</align>");
    evas_object_size_hint_weight_set(ad->label, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
    elm_object_content_set(ad->conform, ad->label);*/

    Evas_Object *bg, *btn;

    /* Background */
    bg = elm_bg_add(ad->win);
    elm_bg_color_set(bg, 0, 0, 0);
    elm_win_resize_object_add(ad->win, bg);
    evas_object_show(bg);

    {
        /* Box to put the table in so we can bottom-align the table
         * window will stretch all resize object content to win size */
        Evas_Object *box = elm_box_add(ad->win);
        evas_object_size_hint_weight_set(box, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
        elm_win_resize_object_add(ad->win, box);
        evas_object_show(box);

        /* Table */
        Evas_Object *table = elm_table_add(ad->win);
        /* Make table homogenous - every cell will be the same size */
        elm_table_homogeneous_set(table, EINA_TRUE);
        /* Set padding of 10 pixels multiplied by scale factor of UI */
        elm_table_padding_set(table, 10 * elm_config_scale_get(), 10 * elm_config_scale_get());
        /* Let the table child allocation area expand within in the box */
        evas_object_size_hint_weight_set(table, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
        /* Set table to fiill width but align to bottom of box */
        evas_object_size_hint_align_set(table, EVAS_HINT_FILL, EVAS_HINT_FILL);
        elm_box_pack_end(box, table);
        evas_object_show(table);

        {
            /* Play Button */
            btn = elm_button_add(ad->win);
            elm_object_text_set(btn, "Play");
            evas_object_smart_callback_add(btn, "clicked", start_player, (void*)ad);
            my_table_pack(table, btn, 0, 3, 1, 1);

            /* Pause Button */
            btn = elm_button_add(ad->win);
            elm_object_text_set(btn, "Pause");
            evas_object_smart_callback_add(btn, "clicked", pause_player, (void*)ad);
            my_table_pack(table, btn, 1, 3, 1, 1);

            /* Stop Button */
            btn = elm_button_add(ad->win);
            elm_object_text_set(btn, "Stop");
            evas_object_smart_callback_add(btn, "clicked", stop_player, (void*)ad);
            my_table_pack(table, btn, 2, 3, 1, 1);

            /* File Load-1 Button */
            btn = elm_button_add(ad->win);
            elm_object_text_set(btn, "File-1");
            evas_object_smart_callback_add(btn, "clicked", btn_load_file1, (void*)ad);
            my_table_pack(table, btn, 0, 4, 3, 1);

            /* File Load-2 Button */
            btn = elm_button_add(ad->win);
            elm_object_text_set(btn, "File-2");
            evas_object_smart_callback_add(btn, "clicked", btn_load_file2, (void*)ad);
            my_table_pack(table, btn, 0, 5, 3, 1);

            /* Container: standard table */
            Evas_Object *tbl = elm_table_add(ad->win);
            evas_object_size_hint_weight_set(tbl, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
            evas_object_size_hint_align_set(tbl, EVAS_HINT_FILL, EVAS_HINT_FILL);
            elm_object_content_set(ad->conform, tbl);
            evas_object_show(tbl);

            /* The video object */
            ad->video_rect = video_rect_add(tbl);
            my_table_pack(table, ad->video_rect, 0, 0, 3, 3);
        }
    }

    /* Show window after base gui is set up */
    evas_object_show(ad->win);

    // Create Player
    ad->player = create_player();

    // Load audio file to Player
    prepare_player(ad, 0);
}

static bool
app_create(void *data)
{
	/* Hook to take necessary actions before main event loop starts
		Initialize UI resources and application's data
		If this function returns true, the main loop of application starts
		If this function returns false, the application is terminated */
	appdata_s *ad = data;

	create_base_gui(ad);

	return true;
}

static void
app_control(app_control_h app_control, void *data)
{
	/* Handle the launch request. */
}

static void
app_pause(void *data)
{
	/* Take necessary actions when application becomes invisible. */
}

static void
app_resume(void *data)
{
	/* Take necessary actions when application becomes visible. */
}

static void
app_terminate(void *data)
{
	/* Release all resources. */
}

static void
ui_app_lang_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LANGUAGE_CHANGED*/
	char *locale = NULL;
	system_settings_get_value_string(SYSTEM_SETTINGS_KEY_LOCALE_LANGUAGE, &locale);
	elm_language_set(locale);
	free(locale);
	return;
}

int
main(int argc, char *argv[])
{
	appdata_s ad = {0,};
	int ret = 0;

	ui_app_lifecycle_callback_s event_callback = {0,};
	app_event_handler_h handlers[5] = {NULL, };

	event_callback.create = app_create;
	event_callback.terminate = app_terminate;
	event_callback.pause = app_pause;
	event_callback.resume = app_resume;
	event_callback.app_control = app_control;

	ui_app_add_event_handler(&handlers[APP_EVENT_LANGUAGE_CHANGED], APP_EVENT_LANGUAGE_CHANGED, ui_app_lang_changed, &ad);

	ret = ui_app_main(argc, argv, &event_callback, &ad);
	if (ret != APP_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "app_main() is failed. err = %d", ret);
	}

	return ret;
}
