#include "ttsex.h"
#include <tts.h>

typedef struct appdata {
    Evas_Object *win;
    Evas_Object *conform;
    Evas_Object *label;
    Evas_Object *entry;
    Evas_Object *button;
    tts_h tts;
} appdata_s;

static void
win_back_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = data;
	/* Let window go to hide state. */
	elm_win_iconified_set(ad->win, EINA_TRUE);
}

static void
state_changed_cb(tts_h tts, tts_state_e previous, tts_state_e current, void* user_data)
{
    appdata_s *ad = user_data;

    switch (current)
    {
    case TTS_STATE_PLAYING:
        elm_object_text_set(ad->button, "Stop");
        break;
    case TTS_STATE_READY:
    default:
        elm_object_text_set(ad->button, "Play");
        break;
    }
}

static void
utterance_completed_cb(tts_h tts, int utt_id, void *user_data)
{
    appdata_s *ad = user_data;

    dlog_print(DLOG_INFO, LOG_TAG, "Utterance completed: %d", utt_id);
    elm_object_text_set(ad->button, "Stop (idle)");
}

static void
utterance_started_cb(tts_h tts, int utt_id, void *user_data)
{
    appdata_s *ad = user_data;

    dlog_print(DLOG_INFO, LOG_TAG, "Utterance started: %d", utt_id);
    elm_object_text_set(ad->button, "Stop (speaking)");
}

static tts_h
create_tts_handle(appdata_s *ad)
{
    tts_h tts;
    int ret = tts_create(&tts);
    if (TTS_ERROR_NONE != ret)
    {
        dlog_print(DLOG_INFO, "tag", "%s err = %d", __func__, ret);
    }
    else
    {
        tts_set_utterance_started_cb(tts, utterance_started_cb, ad);
        tts_set_utterance_completed_cb(tts, utterance_completed_cb, ad);
        tts_set_state_changed_cb(tts, state_changed_cb, ad);
        tts_prepare(tts);
    }

    return tts;
}

static void
destroy_tts_handle(tts_h tts)
{
    int ret = tts_destroy(tts); // tts is the TTS handle
    if (TTS_ERROR_NONE != ret)
    {
        dlog_print(DLOG_INFO, "tag", "%s err = %d", __func__, ret);
    }
}

static void
add_text(tts_h tts, appdata_s *ad)
{
    const char* text = "Good morning"; // Text for read
    const char* language = "en_US"; // Language
    int voice_type = TTS_VOICE_TYPE_FEMALE; // Voice type
    int speed = TTS_SPEED_AUTO;
    int utt_id; // Utterance ID for the requested text
    text = elm_object_text_get(ad->entry);

    int ret = tts_add_text(tts, text, language, voice_type, speed, &utt_id);
    if (TTS_ERROR_NONE != ret)
    {
        dlog_print(DLOG_INFO, "tag", "%s err = %d", __func__, ret);
    }
}

static int
get_state(tts_h* tts)
{
    tts_state_e current_state;
    int ret;
    ret = tts_get_state(*tts, &current_state);

    if (TTS_ERROR_NONE != ret)
    {
        dlog_print(DLOG_INFO, "tag", "%s state = %d", __func__, ret);
        return -1;
    }
    else
    {
        dlog_print(DLOG_INFO, "tag", "%s state = %d", __func__, current_state);
        return (int) current_state;
    }
}

static void
btn_play_cb(void *data, Evas_Object *obj, void *event_info)
{
    appdata_s *ad = data;
    int state = get_state(&ad->tts);
    if ((tts_state_e) state == TTS_STATE_READY || (tts_state_e) state == TTS_STATE_PAUSED)
    {
        add_text(ad->tts, ad);
        int ret = tts_play(ad->tts);
        if (TTS_ERROR_NONE != ret)
        {
            dlog_print(DLOG_INFO, "tag", "%s err = %d", __func__, ret);
        }
    }
    else if ((tts_state_e) state == TTS_STATE_PLAYING)
    {
        int ret = tts_stop(ad->tts);
        if (TTS_ERROR_NONE != ret)
        {
            dlog_print(DLOG_INFO, "tag", "%s err = %d", __func__, ret);
        }
    }
}

static void
my_box_pack(Evas_Object *box, Evas_Object *child, double h_weight, double v_weight,
        double h_align, double v_align)
{
	/* create a frame we shall use as padding around the child widget */
	Evas_Object *frame = elm_frame_add(box);
	/* use the medium padding style. there is "pad_small", "pad_medium",
	 * "pad_large" and "pad_huge" available as styles in addition to the
	 * "default" frame style */
	elm_object_style_set(frame, "pad_medium");
	/* set the input weight/aling on the frame insted of the child */
	evas_object_size_hint_weight_set(frame, h_weight, v_weight);
	evas_object_size_hint_align_set(frame, h_align, v_align);
	{
		/* tell the child that is packed into the frame to be able to expand */
		evas_object_size_hint_weight_set(child, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
		/* fill the expanded area (above) as opposaed to center in it */
		evas_object_size_hint_align_set(child, EVAS_HINT_FILL, EVAS_HINT_FILL);
		/* actually put the child in the frame and show it */
		evas_object_show(child);
		elm_object_content_set(frame, child);
	}
	/* put the frame into the box instead of the child directly */
	elm_box_pack_end(box, frame);
	/* show the frame */
	evas_object_show(frame);
}

static void
create_base_gui(appdata_s *ad)
{
    /* set up policy to exit when last window is closed */
    elm_policy_set(ELM_POLICY_QUIT, ELM_POLICY_QUIT_LAST_WINDOW_CLOSED);
    /* Window */
    ad->win = elm_win_util_standard_add(PACKAGE, PACKAGE);
    elm_win_autodel_set(ad->win, EINA_TRUE);

    int rots[4] = { 0, 90, 180, 270 };
    elm_win_wm_rotation_available_rotations_set(ad->win, (const int *)(&rots), 4);

    eext_object_event_callback_add(ad->win, EEXT_CALLBACK_BACK, win_back_cb, ad);

    /* Conformant */
    ad->conform = elm_conformant_add(ad->win);
    elm_win_indicator_mode_set(ad->win, ELM_WIN_INDICATOR_SHOW);
    elm_win_indicator_opacity_set(ad->win, ELM_WIN_INDICATOR_OPAQUE);
    evas_object_size_hint_weight_set(ad->conform, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
    elm_win_resize_object_add(ad->win, ad->conform);
    evas_object_show(ad->conform);

    {
        Evas_Object *btn, *box;

        /* Container: standard box */
        box = elm_box_add(ad->win);
        elm_box_homogeneous_set(box, EINA_TRUE);
        elm_box_horizontal_set(box, EINA_FALSE);
        evas_object_size_hint_weight_set(box, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
        evas_object_size_hint_align_set(box, EVAS_HINT_FILL, EVAS_HINT_FILL);
        elm_object_content_set(ad->conform, box);
        evas_object_show(box);

        {
            /* Entry */
            ad->entry = elm_entry_add(box);
            elm_entry_single_line_set(ad->entry, EINA_FALSE);
            elm_entry_scrollable_set(ad->entry, EINA_TRUE);
            elm_object_text_set(ad->entry, "Hello world");
            my_box_pack(box, ad->entry, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND, EVAS_HINT_FILL, EVAS_HINT_FILL);

            /* Button-1 */
            btn = elm_button_add(box);
            elm_object_text_set(btn, "Play/Stop");
            evas_object_smart_callback_add(btn, "clicked", btn_play_cb, ad);
            my_box_pack(box, btn, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND, EVAS_HINT_FILL, 0.0);
            ad->button = btn;
        }
    }

    /* Show window after base gui is set up */
    evas_object_show(ad->win);

    ad->tts = create_tts_handle(ad);
}

static bool
app_create(void *data)
{
	/* Hook to take necessary actions before main event loop starts
		Initialize UI resources and application's data
		If this function returns true, the main loop of application starts
		If this function returns false, the application is terminated */
	appdata_s *ad = data;

	create_base_gui(ad);

	return true;
}

static void
app_control(app_control_h app_control, void *data)
{
	/* Handle the launch request. */
}

static void
app_pause(void *data)
{
	/* Take necessary actions when application becomes invisible. */
}

static void
app_resume(void *data)
{
	/* Take necessary actions when application becomes visible. */
}

static void
app_terminate(void *data)
{
    appdata_s *ad = data;
    destroy_tts_handle(ad->tts);
}

static void
ui_app_lang_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LANGUAGE_CHANGED*/
	char *locale = NULL;
	system_settings_get_value_string(SYSTEM_SETTINGS_KEY_LOCALE_LANGUAGE, &locale);
	elm_language_set(locale);
	free(locale);
	return;
}

int
main(int argc, char *argv[])
{
	appdata_s ad = {0,};
	int ret = 0;

	ui_app_lifecycle_callback_s event_callback = {0,};
	app_event_handler_h handlers[5] = {NULL, };

	event_callback.create = app_create;
	event_callback.terminate = app_terminate;
	event_callback.pause = app_pause;
	event_callback.resume = app_resume;
	event_callback.app_control = app_control;

	ui_app_add_event_handler(&handlers[APP_EVENT_LANGUAGE_CHANGED], APP_EVENT_LANGUAGE_CHANGED, ui_app_lang_changed, &ad);

	ret = ui_app_main(argc, argv, &event_callback, &ad);
	if (ret != APP_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "app_main() is failed. err = %d", ret);
	}

	return ret;
}
