#include "netconnection.h"
#include <net_connection.h>

typedef struct appdata {
	Evas_Object *win;
	Evas_Object *conform;
    Evas_Object *label1;
    Evas_Object *label2;
    Evas_Object *label3;
    connection_h connection;
} appdata_s;

static void
win_back_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = data;
	/* Let window go to hide state. */
	elm_win_iconified_set(ad->win, EINA_TRUE);
}

static int
net_state(appdata_s *ad)
{
    int error_code;

    error_code = connection_create(&ad->connection);
    if (error_code != CONNECTION_ERROR_NONE) {
        dlog_print(DLOG_ERROR, "tag", "connection error");
       return error_code;
    }

    connection_type_e net_state;
    error_code = connection_get_type(ad->connection, &net_state);
    switch( net_state )
    {
    case CONNECTION_TYPE_DISCONNECTED    : /**< Disconnected */
        elm_object_text_set(ad->label1, "Net state Disconnected");
        break;
    case CONNECTION_TYPE_WIFI : /**< Wi-Fi type */
        elm_object_text_set(ad->label1, "Net state Wifi");
        break;
    case CONNECTION_TYPE_CELLULAR : /**< Cellular type */
        elm_object_text_set(ad->label1, "Net state Cellular");
        break;
    case CONNECTION_TYPE_ETHERNET : /**< Ethernet type */
        elm_object_text_set(ad->label1, "Net state Ethernet");
        break;
    case CONNECTION_TYPE_BT : /**< Bluetooth type */
        elm_object_text_set(ad->label1, "Net state BT");
        break;
    }
    return error_code;
}

static void
cellular_state(appdata_s *ad)
{
    int error_code;
    connection_cellular_state_e state;
    error_code = connection_get_cellular_state(ad->connection, &state);

    switch( state )
    {
   case CONNECTION_CELLULAR_STATE_OUT_OF_SERVICE:
      elm_object_text_set(ad->label2, "Cell state Out of service");
      break;
   case CONNECTION_CELLULAR_STATE_FLIGHT_MODE:
      elm_object_text_set(ad->label2, "Cell state Flight mode");
      break;
   case CONNECTION_CELLULAR_STATE_ROAMING_OFF:
      elm_object_text_set(ad->label2, "Cell state Roaming off");
      break;
   case CONNECTION_CELLULAR_STATE_CALL_ONLY_AVAILABLE:
      elm_object_text_set(ad->label2, "Cell state Call only");
      break;
   case CONNECTION_CELLULAR_STATE_AVAILABLE:
      elm_object_text_set(ad->label2, "Cell state Available");
      break;
   case CONNECTION_CELLULAR_STATE_CONNECTED:
      elm_object_text_set(ad->label2, "Cell state Connected");
      break;
   default:
      elm_object_text_set(ad->label2, "Cell state Error");
      break;
    }
}

static void
wifi_state(appdata_s *ad)
{
    connection_wifi_state_e wifi_state;
    connection_get_wifi_state(ad->connection, &wifi_state);
    switch (wifi_state)
    {
       case CONNECTION_WIFI_STATE_DEACTIVATED:
          elm_object_text_set(ad->label3, "Wifi state Deactivated");
          break;
       case CONNECTION_WIFI_STATE_DISCONNECTED:
          elm_object_text_set(ad->label3, "Wifi state Disconnected");
          break;
       case CONNECTION_WIFI_STATE_CONNECTED:
          elm_object_text_set(ad->label3, "Wifi state Connected");
          break;
       default:
          dlog_print(DLOG_INFO, "tag", "Wifi error");
          elm_object_text_set(ad->label3, "Wifi state Error");
          break;
    }
}

static void
create_base_gui(appdata_s *ad)
{
	/* set up policy to exit when last window is closed */
	elm_policy_set(ELM_POLICY_QUIT, ELM_POLICY_QUIT_LAST_WINDOW_CLOSED);
	/* Window */
	ad->win = elm_win_util_standard_add(PACKAGE, PACKAGE);
	elm_win_autodel_set(ad->win, EINA_TRUE);

	int rots[4] = { 0, 90, 180, 270 };
	elm_win_wm_rotation_available_rotations_set(ad->win, (const int *)(&rots), 4);

	eext_object_event_callback_add(ad->win, EEXT_CALLBACK_BACK, win_back_cb, ad);

    /* Conformant */
    ad->conform = elm_conformant_add(ad->win);
    elm_win_indicator_mode_set(ad->win, ELM_WIN_INDICATOR_SHOW);
    elm_win_indicator_opacity_set(ad->win, ELM_WIN_INDICATOR_OPAQUE);
    evas_object_size_hint_weight_set(ad->conform, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
    elm_win_resize_object_add(ad->win, ad->conform);
    evas_object_show(ad->conform);

    /* Frame for outer padding */
    Evas_Object *frame = elm_frame_add(ad->win);
    elm_object_style_set(frame, "pad_huge");
    elm_object_content_set(ad->conform, frame);
    evas_object_show(frame);

    /* Vertical box */
    Evas_Object *vbox = elm_box_add(ad->win);
    elm_box_padding_set(vbox, ELM_SCALE_SIZE(10), ELM_SCALE_SIZE(10));
    elm_object_content_set(frame, vbox);
    evas_object_show(vbox);

    {
        /* Label-1 */
        ad->label1 = elm_label_add(ad->conform);
        elm_object_text_set(ad->label1, "Net state");
        evas_object_size_hint_weight_set(ad->label1, EVAS_HINT_EXPAND, 0);
        elm_box_pack_end(vbox, ad->label1);
        evas_object_show(ad->label1);

        /* Label-2 */
        ad->label2 = elm_label_add(ad->conform);
        elm_object_text_set(ad->label2, "Cell state");
        evas_object_size_hint_weight_set(ad->label2, EVAS_HINT_EXPAND, 0);
        elm_box_pack_end(vbox, ad->label2);
        evas_object_show(ad->label2);

        /* Label-3 */
        ad->label3 = elm_label_add(ad->conform);
        elm_object_text_set(ad->label3, "Wifi state");
        evas_object_size_hint_weight_set(ad->label3, EVAS_HINT_EXPAND, 0);
        elm_box_pack_end(vbox, ad->label3);
        evas_object_show(ad->label3);
    }

    /* Show window after base gui is set up */
    evas_object_show(ad->win);

    int error_code = net_state(ad);
    if (error_code == CONNECTION_ERROR_NONE) {
        cellular_state(ad);
        wifi_state(ad);
    }
}

static bool
app_create(void *data)
{
	/* Hook to take necessary actions before main event loop starts
		Initialize UI resources and application's data
		If this function returns true, the main loop of application starts
		If this function returns false, the application is terminated */
	appdata_s *ad = data;

	create_base_gui(ad);

	return true;
}

static void
app_control(app_control_h app_control, void *data)
{
	/* Handle the launch request. */
}

static void
app_pause(void *data)
{
	/* Take necessary actions when application becomes invisible. */
}

static void
app_resume(void *data)
{
	/* Take necessary actions when application becomes visible. */
}

static void
app_terminate(void *data)
{
    appdata_s *ad = data;
    connection_destroy(ad->connection);
}

static void
ui_app_lang_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LANGUAGE_CHANGED*/
	char *locale = NULL;
	system_settings_get_value_string(SYSTEM_SETTINGS_KEY_LOCALE_LANGUAGE, &locale);
	elm_language_set(locale);
	free(locale);
	return;
}

int
main(int argc, char *argv[])
{
	appdata_s ad = {0,};
	int ret = 0;

	ui_app_lifecycle_callback_s event_callback = {0,};
	app_event_handler_h handlers[5] = {NULL, };

	event_callback.create = app_create;
	event_callback.terminate = app_terminate;
	event_callback.pause = app_pause;
	event_callback.resume = app_resume;
	event_callback.app_control = app_control;

	ui_app_add_event_handler(&handlers[APP_EVENT_LANGUAGE_CHANGED], APP_EVENT_LANGUAGE_CHANGED, ui_app_lang_changed, &ad);

	ret = ui_app_main(argc, argv, &event_callback, &ad);
	if (ret != APP_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "app_main() is failed. err = %d", ret);
	}

	return ret;
}
