#include "canvasimage.h"

typedef struct appdata {
	Evas_Object *win;
	Evas_Object *conform;
	Evas_Object *label;
} appdata_s;

static void
win_back_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = data;
	/* Let window go to hide state. */
	elm_win_iconified_set(ad->win, EINA_TRUE);
}

static void
app_get_resource(const char *res_file_in, char *res_path_out, int res_path_max)
{
    char *res_path = app_get_resource_path();
    if (res_path) {
        snprintf(res_path_out, res_path_max, "%s%s", res_path, res_file_in);
        free(res_path);
    }
}

// Create IMAGE object. Source image's rate is no change
static Evas_Object *
create_image(Evas *canvas, Evas_Object *grid, const char *img_path, int x, int y, int w, int h)
{
   int source_w, source_h, new_x, new_y, new_w, new_h;
   float rate_h, rate_v, rate;

   // Create IMAGE object
   Evas_Object *img = evas_object_image_add(canvas);
   // Set source image file
   evas_object_image_file_set(img, img_path, NULL);
   // Get source image size
   evas_object_image_size_get(img, &source_w, &source_h);
   // Load failed - zero sized image
   if ((source_w == 0) || (source_h == 0))
     {
        evas_object_del(img);
        return NULL;
     }

   // Calculage Zoom rate
   rate_h = (float)w / (float)source_w;
   rate_v = (float)h / (float)source_h;
   rate = (rate_h < rate_v) ? rate_h : rate_v;

   // Calculate output image size
   new_w = source_w * rate;
   new_h = source_h * rate;
   evas_object_image_fill_set(img, 0, 0, new_w, new_h);

   // Calculate output Image position
   new_x = x + (w - new_w) / 2;
   new_y = y + (h - new_h) / 2;
   elm_grid_pack(grid, img, new_x, new_y, new_w, new_h);

   evas_object_show(img);
   return img;
}

static void
create_base_gui(appdata_s *ad)
{
    /* set up policy to exit when last window is closed */
    elm_policy_set(ELM_POLICY_QUIT, ELM_POLICY_QUIT_LAST_WINDOW_CLOSED);
	/* Window */
	ad->win = elm_win_util_standard_add(PACKAGE, PACKAGE);
	elm_win_autodel_set(ad->win, EINA_TRUE);

	int rots[4] = { 0, 90, 180, 270 };
	elm_win_wm_rotation_available_rotations_set(ad->win, (const int *)(&rots), 4);

	eext_object_event_callback_add(ad->win, EEXT_CALLBACK_BACK, win_back_cb, ad);

    /* Conformant */
    ad->conform = elm_conformant_add(ad->win);
    elm_win_indicator_mode_set(ad->win, ELM_WIN_INDICATOR_SHOW);
    elm_win_indicator_opacity_set(ad->win, ELM_WIN_INDICATOR_OPAQUE);
    evas_object_size_hint_weight_set(ad->conform, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
    elm_win_resize_object_add(ad->win, ad->conform);
    evas_object_show(ad->conform);

    /* Label*/
    /*ad->label = elm_label_add(ad->conform);
    elm_object_text_set(ad->label, "<align=center>Hello EFL</align>");
    evas_object_size_hint_weight_set(ad->label, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
    elm_object_content_set(ad->conform, ad->label);*/

    { /* child object - indent to how relationship */
        /* A grid to stretch content within grid size */
        Evas_Object *grid = elm_grid_add(ad->win);
        elm_grid_size_set(grid, 480, 800);
        evas_object_size_hint_weight_set(grid, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
        elm_object_content_set(ad->conform, grid);
        evas_object_show(grid);

        {
            /* Canvas */
            Evas* canvas = evas_object_evas_get(ad->win);

            char img_path[PATH_MAX] = "";
            app_get_resource("tizen_logo.png", img_path, PATH_MAX);

            /* Image-1 */
            Evas_Object *img = evas_object_image_filled_add(canvas);
            evas_object_image_file_set(img, img_path, NULL);
            elm_grid_pack(grid, img, 40, 10, 400, 280);
            evas_object_show(img);

            /* Image-2 */
            int w, h;
            img = evas_object_image_add(canvas);
            evas_object_image_file_set(img, img_path, NULL);
            evas_object_image_size_get(img, &w, &h);
            evas_object_image_fill_set(img, 110, 37, w, h);
            elm_grid_pack(grid, img, 40, 310, 400, 280);
            evas_object_show(img);

            /* Image-3 */
            create_image(canvas, grid, img_path, 40, 610, 400, 180);
        }
    }

    /* Show window after base gui is set up */
    evas_object_show(ad->win);
}

static bool
app_create(void *data)
{
	/* Hook to take necessary actions before main event loop starts
		Initialize UI resources and application's data
		If this function returns true, the main loop of application starts
		If this function returns false, the application is terminated */
	appdata_s *ad = data;

	create_base_gui(ad);

	return true;
}

static void
app_control(app_control_h app_control, void *data)
{
	/* Handle the launch request. */
}

static void
app_pause(void *data)
{
	/* Take necessary actions when application becomes invisible. */
}

static void
app_resume(void *data)
{
	/* Take necessary actions when application becomes visible. */
}

static void
app_terminate(void *data)
{
	/* Release all resources. */
}

static void
ui_app_lang_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LANGUAGE_CHANGED*/
	char *locale = NULL;
	system_settings_get_value_string(SYSTEM_SETTINGS_KEY_LOCALE_LANGUAGE, &locale);
	elm_language_set(locale);
	free(locale);
	return;
}

int
main(int argc, char *argv[])
{
	appdata_s ad = {0,};
	int ret = 0;

	ui_app_lifecycle_callback_s event_callback = {0,};
	app_event_handler_h handlers[5] = {NULL, };

	event_callback.create = app_create;
	event_callback.terminate = app_terminate;
	event_callback.pause = app_pause;
	event_callback.resume = app_resume;
	event_callback.app_control = app_control;

	ui_app_add_event_handler(&handlers[APP_EVENT_LANGUAGE_CHANGED], APP_EVENT_LANGUAGE_CHANGED, ui_app_lang_changed, &ad);

	ret = ui_app_main(argc, argv, &event_callback, &ad);
	if (ret != APP_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "app_main() is failed. err = %d", ret);
	}

	return ret;
}
