#include "recorderex.h"
#include <recorder.h>
#include <player.h>

typedef struct appdata {
	Evas_Object *win;
	Evas_Object *conform;
	Evas_Object *label;

    Evas_Object *btn_rec, *btn_recstop, *btn_play, *btn_playstop;

    player_h player;
    recorder_h recorder;
    recorder_audio_codec_e *codec_list;
    int codec_list_len;
    char file_path[PATH_MAX];
    recorder_audio_codec_e codec;
    recorder_file_format_e file_format;
    FILE *preproc_file;
} appdata_s;

typedef struct
{
    recorder_audio_codec_e *codec_list;
    int len;
} supported_encoder_data;

static void
win_back_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = data;
	/* Let window go to hide state. */
	elm_win_iconified_set(ad->win, EINA_TRUE);
}

// Check is recording
static bool
_recorder_is_recording(appdata_s *ad)
{
    recorder_state_e state = RECORDER_STATE_NONE;
    recorder_get_state(ad->recorder, &state);
    return state == RECORDER_STATE_RECORDING;
}

// Stop recording
static void
record_stop(void *data, Evas_Object *obj, void *event_info)
{
    appdata_s *ad = data;
    if (ad->recorder)
    {
        recorder_commit(ad->recorder);
        // Check is recording
        if (!_recorder_is_recording(ad))
        {
            recorder_unprepare(ad->recorder);
        }
        elm_object_disabled_set(ad->btn_play, EINA_FALSE);
        elm_object_disabled_set(ad->btn_rec, EINA_FALSE);
        elm_object_disabled_set(ad->btn_playstop, EINA_TRUE);
        elm_object_disabled_set(ad->btn_recstop, EINA_TRUE);
    }
}

// Maximum recording time event callback function
static void
_on_recording_limit_reached_cb(recorder_recording_limit_type_e type, void *user_data)
{
    appdata_s *ad = user_data;
    if(type == RECORDER_RECORDING_LIMIT_TIME)
        // Stop recording
        record_stop(ad, NULL, NULL);
}

// Create recorder
static void
_recorder_create(appdata_s *ad)
{
    if(recorder_create_audiorecorder(&ad->recorder) == RECORDER_ERROR_NONE)
    {
        // Set maximum recording time event callback function
        recorder_set_recording_limit_reached_cb(ad->recorder, _on_recording_limit_reached_cb, ad);
        recorder_attr_set_audio_channel(ad->recorder, 1);
        recorder_attr_set_audio_device(ad->recorder, RECORDER_AUDIO_DEVICE_MIC);
        recorder_attr_set_time_limit(ad->recorder, 20);
    }
}

static bool
_recorder_supported_audio_encoder_cb(recorder_audio_codec_e codec, void *user_data)
{
    bool result = false;
    supported_encoder_data *data = user_data;

    if(data && codec != RECORDER_AUDIO_CODEC_DISABLE)
    {
        data->codec_list = realloc(data->codec_list, sizeof(supported_encoder_data) * (data->len + 1));
        data->codec_list[data->len] = codec;
        ++(data->len);
        result = true;
    }

    return result;
}

recorder_audio_codec_e*
audio_recorder_get_supported_encoder(recorder_h recorder, int *list_length)
{
    supported_encoder_data data = {0};
    data.codec_list = NULL;
    data.len = 0;

    int res = recorder_foreach_supported_audio_encoder(recorder, _recorder_supported_audio_encoder_cb, &data);

    if(res && list_length)
    {
        *list_length = data.len;
    }

    return data.codec_list;
}

const char*
get_file_format_by_codec(appdata_s* ad)
{
    switch(ad->codec)
    {
    case RECORDER_AUDIO_CODEC_AMR:
    	ad->file_format = RECORDER_FILE_FORMAT_AMR;
    	return "AMR";
        break;
    case RECORDER_AUDIO_CODEC_AAC:
    	ad->file_format = RECORDER_FILE_FORMAT_MP4;
       	return "MP4";
        break;
    case RECORDER_AUDIO_CODEC_VORBIS:
    	ad->file_format = RECORDER_FILE_FORMAT_OGG;
       	return "OGG";
        break;
    }

    ad->file_format = RECORDER_FILE_FORMAT_WAV;
    return "WAV";
}

static void
_codec_set(appdata_s *ad)
{
    char file_name[NAME_MAX] = {'\0'};
    const char *file_ext = get_file_format_by_codec(ad);

    char *data_path = app_get_data_path();
    snprintf(file_name, NAME_MAX, "record.%s", file_ext);
    snprintf(ad->file_path, PATH_MAX, "%s%s", data_path, file_name);
    free(data_path);
}

//=============================================================================================

/*static void
 * _on_recorder_audio_stream_cb(void* stream, int size, audio_sample_type_e format, int channel, unsigned int timestamp, void *user_data)
{
    appdata_s *ad = user_data;
    if(ad && ad->preproc_file && !feof(ad->preproc_file))
    {
        char buf[512];
        int read_bytes = 0;

        while(read_bytes != size)
        {
            int request_bytes = MIN(512, size - read_bytes);
            int read = fread(buf, sizeof(char), request_bytes, ad->preproc_file);

            if(!read)
                break;

            memcpy(stream + read_bytes, buf, read);
            read_bytes += read;
        }
    }
}*/

// Apply settings to recorder
static void
_recorder_apply_settings(appdata_s *ad)
{
    if(ad->recorder)
    {
        // Set record file name
        recorder_set_filename(ad->recorder, ad->file_path);
        // Set record file format
        recorder_set_file_format(ad->recorder, ad->file_format);
        // Set record codec
        recorder_set_audio_encoder(ad->recorder, ad->codec);
    }
}

// Start recording
static void
record_start(void *data, Evas_Object *obj, void *event_info)
{
    appdata_s *ad = data;
    if (ad->recorder)
    {
        // Apply settings to recorder
        _recorder_apply_settings(ad);
        recorder_prepare(ad->recorder);
        recorder_start(ad->recorder);
        elm_object_disabled_set(ad->btn_recstop, EINA_FALSE);
        elm_object_disabled_set(ad->btn_rec, EINA_TRUE);
        elm_object_disabled_set(ad->btn_play, EINA_TRUE);
        elm_object_disabled_set(ad->btn_playstop, EINA_TRUE);
    }
}

//========================================================================

// Get player state
static player_state_e
get_player_state(player_h player)
{
    player_state_e state;
    player_get_state(player, &state);
    return state;
}

// Stop play
static void
stop_player(void *data, Evas_Object *obj, void *event_info)
{
    appdata_s *ad = data;

    if( get_player_state(ad->player) == PLAYER_STATE_PLAYING || get_player_state(ad->player) == PLAYER_STATE_PAUSED)
        player_stop(ad->player);

    elm_object_disabled_set(ad->btn_play, EINA_FALSE);
    elm_object_disabled_set(ad->btn_playstop, EINA_TRUE);
    elm_object_disabled_set(ad->btn_rec, EINA_FALSE);
    elm_object_disabled_set(ad->btn_recstop, EINA_TRUE);
}

// Load file to player
static void
prepare_player(appdata_s* ad)
{
    stop_player(ad, NULL, NULL);
    player_unprepare(ad->player);
    player_set_uri(ad->player, ad->file_path);
    player_prepare(ad->player);
}

// Start play
static void
start_player(void *data, Evas_Object *obj, void *event_info)
{
    appdata_s *ad = data;
    prepare_player(ad);

    if (get_player_state(ad->player) != PLAYER_STATE_PLAYING)
    {
        player_start(ad->player);

        elm_object_disabled_set(ad->btn_rec, EINA_TRUE);
        elm_object_disabled_set(ad->btn_recstop, EINA_TRUE);
        elm_object_disabled_set(ad->btn_play, EINA_TRUE);
        elm_object_disabled_set(ad->btn_playstop, EINA_FALSE);
    }
}

// Create player
static player_h
create_player()
{
    player_h player;

    player_create(&player);
    player_set_completed_cb(player, NULL, player);

    return player;
}

//========================================================================

static void
create_base_gui(appdata_s *ad)
{
    /* set up policy to exit when last window is closed */
    elm_policy_set(ELM_POLICY_QUIT, ELM_POLICY_QUIT_LAST_WINDOW_CLOSED);
	/* Window */
	ad->win = elm_win_util_standard_add(PACKAGE, PACKAGE);
	elm_win_autodel_set(ad->win, EINA_TRUE);

	int rots[4] = { 0, 90, 180, 270 };
	elm_win_wm_rotation_available_rotations_set(ad->win, (const int *)(&rots), 4);

	eext_object_event_callback_add(ad->win, EEXT_CALLBACK_BACK, win_back_cb, ad);

    /* Conformant */
    ad->conform = elm_conformant_add(ad->win);
    elm_win_indicator_mode_set(ad->win, ELM_WIN_INDICATOR_SHOW);
    elm_win_indicator_opacity_set(ad->win, ELM_WIN_INDICATOR_OPAQUE);
    evas_object_size_hint_weight_set(ad->conform, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
    elm_win_resize_object_add(ad->win, ad->conform);
    evas_object_show(ad->conform);

    {
        Evas_Object *btn, *frame, *tbl;

        /* Frame for some outer padding */
        frame = elm_frame_add(ad->conform);
        elm_object_style_set(frame, "pad_medium");
        elm_object_content_set(ad->conform, frame);
        evas_object_show(frame);

        /* Table to pack our elements */
        tbl = elm_table_add(frame);
        elm_table_padding_set(tbl, 5 * elm_scale_get(), 5 * elm_scale_get());
        elm_object_content_set(frame, tbl);
        evas_object_show(tbl);

        {
            /* Just a label */
            ad->label = elm_label_add(tbl);
            elm_object_text_set(ad->label, "Audio recorder");
            evas_object_size_hint_align_set(ad->label, 0.5, 0.5);
            evas_object_size_hint_weight_set(ad->label, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
            elm_table_pack(tbl, ad->label, 0, 0, 2, 1);
            evas_object_show(ad->label);

            /* Record Start Button */
            btn = elm_button_add(tbl);
            elm_object_text_set(btn, "Recording Start");
            evas_object_smart_callback_add(btn, "clicked", record_start, (void*)ad);
            evas_object_size_hint_weight_set(btn, EVAS_HINT_EXPAND, 0);
            evas_object_size_hint_align_set(btn, EVAS_HINT_FILL, 0.5);
            elm_table_pack(tbl, btn, 0, 1, 1, 1);
            evas_object_show(btn);
            ad->btn_rec = btn;

            /* Record Stop Button */
            btn = elm_button_add(tbl);
            elm_object_disabled_set(btn, EINA_TRUE);
            elm_object_text_set(btn, "Recording Stop");
            evas_object_smart_callback_add(btn, "clicked", record_stop, (void*)ad);
            evas_object_size_hint_weight_set(btn, EVAS_HINT_EXPAND, 0);
            evas_object_size_hint_align_set(btn, EVAS_HINT_FILL, 0.5);
            elm_table_pack(tbl, btn, 1, 1, 1, 1);
            evas_object_show(btn);
            ad->btn_recstop = btn;

            /* Play Start Button */
            btn = elm_button_add(tbl);
            elm_object_disabled_set(btn, EINA_TRUE);
            elm_object_text_set(btn, "Play Start");
            evas_object_smart_callback_add(btn, "clicked", start_player, (void*)ad);
            evas_object_size_hint_weight_set(btn, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
            evas_object_size_hint_align_set(btn, EVAS_HINT_FILL, 0.0);
            elm_table_pack(tbl, btn, 0, 2, 1, 1);
            evas_object_show(btn);
            ad->btn_play = btn;

            /* Play Stop Button */
            btn = elm_button_add(tbl);
            elm_object_disabled_set(btn, EINA_TRUE);
            elm_object_text_set(btn, "Play Stop");
            evas_object_smart_callback_add(btn, "clicked", stop_player, (void*)ad);
            evas_object_size_hint_weight_set(btn, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
            evas_object_size_hint_align_set(btn, EVAS_HINT_FILL, 0.0);
            elm_table_pack(tbl, btn, 1, 2, 1, 1);
            evas_object_show(btn);
            ad->btn_playstop = btn;
        }
    }

    // create player
    ad->player = create_player();

    /* Show window after base gui is set up */
    evas_object_show(ad->win);

    // Create recorder
    _recorder_create(ad);
    ad->codec_list = audio_recorder_get_supported_encoder(ad->recorder, &ad->codec_list_len);
    ad->codec = ad->codec_list_len ? ad->codec_list[0] : RECORDER_AUDIO_CODEC_PCM;

    _codec_set(ad);
}

static bool
app_create(void *data)
{
	/* Hook to take necessary actions before main event loop starts
		Initialize UI resources and application's data
		If this function returns true, the main loop of application starts
		If this function returns false, the application is terminated */
	appdata_s *ad = data;

	create_base_gui(ad);

	return true;
}

static void
app_control(app_control_h app_control, void *data)
{
	/* Handle the launch request. */
}

static void
app_pause(void *data)
{
	/* Take necessary actions when application becomes invisible. */
}

static void
app_resume(void *data)
{
	/* Take necessary actions when application becomes visible. */
}

static void
app_terminate(void *data)
{
	/* Release all resources. */
}

static void
ui_app_lang_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LANGUAGE_CHANGED*/
	char *locale = NULL;
	system_settings_get_value_string(SYSTEM_SETTINGS_KEY_LOCALE_LANGUAGE, &locale);
	elm_language_set(locale);
	free(locale);
	return;
}

int
main(int argc, char *argv[])
{
	appdata_s ad = {0,};
	int ret = 0;

	ui_app_lifecycle_callback_s event_callback = {0,};
	app_event_handler_h handlers[5] = {NULL, };

	event_callback.create = app_create;
	event_callback.terminate = app_terminate;
	event_callback.pause = app_pause;
	event_callback.resume = app_resume;
	event_callback.app_control = app_control;

	ui_app_add_event_handler(&handlers[APP_EVENT_LANGUAGE_CHANGED], APP_EVENT_LANGUAGE_CHANGED, ui_app_lang_changed, &ad);

	ret = ui_app_main(argc, argv, &event_callback, &ad);
	if (ret != APP_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "app_main() is failed. err = %d", ret);
	}

	return ret;
}
