/*
 * MapView.h
 *
 *  Created on: Apr 29, 2015
 *      Author: 김시찬 Samsung Electronics co
 */

#ifndef MAPVIEW_H_
#define MAPVIEW_H_

#include <app.h>
#include <Evas_GL.h>
#include <Evas_GL_GLES2_Helpers.h>
#include <Elementary.h>
#include <system_settings.h>
#include <efl_extension.h>
#include <dlog.h>
#include <curl/curl.h>
#include <math.h>
#include <dlog.h>
#include <locations.h>

#define TMP_DIR "/tmp/map_temp"
#define START_URL "http://maps.googleapis.com/maps/api/staticmap?language=korean&zoom=99&center=unknown&size=480x800&key=AIzaSyC_3XrTOaUat4nf5XOLMBkNJKb23VgAmXQ"
#define PLACE_API_URL "https://maps.googleapis.com/maps/api/place/textsearch/xml?query=where&sensor=true&key=AIzaSyC_3XrTOaUat4nf5XOLMBkNJKb23VgAmXQ"

#define MAX_TMP_FILENAME_LEN 256
#define MAX_URL_LEN 2048
#define MAP_TILE_X 3
#define MAP_TILE_Y 3

#define MAX_ZOOM_SCALE 19
#define MIN_ZOOM_SCALE 0
#define START_ZOOM_LEVEL 13

#define X_RESOLUTION 480
#define Y_RESOLUTION 800

#define ALL_TILE_USABLE_X 1000
#define ALL_TILE_USABLE_Y 1000
#define GEO_INFO_STR_NUM 20
#define MAP_DLOAD_Q_NUM 500
#define PLACE_SEARCHED_NUM 100
#define PLACE_INFO_MAX (100*1000)

typedef struct mapdata {
	float            xangle;
	float            yangle;
	Eina_Bool        mouse_down : 1;
	Eina_Bool        mouse_move : 1;
	Eina_Bool        mouse_move_update : 1;
	Eina_Bool        wait_for_update : 1;
	Ecore_Thread*    thread;
} mapdata_s;

typedef struct all_tile_info {
	double lati_of_center;
	double longti_of_center;
	char download_ok;
	char download_request;
	char file_name[MAX_TMP_FILENAME_LEN];
}all_tile_info_s;

typedef struct index_of_all_tile {
	int x;
	int y;
}index_of_all_tile_s;

typedef struct certer_tile_info {
	int x_info;
	int y_info;
}certer_tile_info_s;

typedef struct map_dload_queue {
	char url[MAX_URL_LEN];
	char filename[MAX_TMP_FILENAME_LEN];
	index_of_all_tile_s index;
}map_dload_queue_s;

typedef struct place_search_list {
	double str_lati;
	double str_longti;
} place_search_list_s;

typedef struct tile_info {
	int curr_x;
	int curr_y;
	char file_name[MAX_TMP_FILENAME_LEN];
}tile_info_s;

typedef struct MemoryStruct {
	char *memory;
	size_t size;
} MemoryStruct_s;

enum
{
	URL_CHANGED,
	URL_NOT_CHANGED,
}new_url_result;

enum
{
	ZOOM_CHANGED,
	CENTER_NAME_CHANGED,
	CENTER_GEOMETRY_CHANGED,
	ADD_NEW_MARKER,
	ADD_NEW_MULTI_MARKER,
	DELITE_MARKER,
	PLACE_INFO,
	MAX_REASON
}new_url_reason;

const tile_info_s tile_info_init_information[MAP_TILE_X][MAP_TILE_Y] = {
		{{-X_RESOLUTION,Y_RESOLUTION}, {-X_RESOLUTION,0}, {-X_RESOLUTION,-Y_RESOLUTION}},
		{{0,Y_RESOLUTION}, {0,0}, {0,-Y_RESOLUTION}},
		{{X_RESOLUTION,Y_RESOLUTION}, {X_RESOLUTION,0}, {X_RESOLUTION,-Y_RESOLUTION}}
		};
tile_info_s tile_info[MAP_TILE_X][MAP_TILE_Y];

//appdata_s *ad_g;
mapdata_s *m_md;
Evas *m_canvas;
Evas_Object* main_page[MAP_TILE_X][MAP_TILE_Y];
Evas_Object *m_btn1;
Evas_Object *m_btn2;

int map_dload_q_idx = 0;
char do_not_update_map = 0;
map_dload_queue_s map_dload_q[MAP_DLOAD_Q_NUM];
all_tile_info_s all_tiles[ALL_TILE_USABLE_X][ALL_TILE_USABLE_Y];
char place_api_url[MAX_URL_LEN] = PLACE_API_URL;
char curr_url[MAX_URL_LEN] = START_URL;
char search_str[MAX_REASON][20]={"zoom=", "center=", "center=", "&", "&", "&markers=", "query="};
int place_searched_num;
place_search_list_s place_search_list[PLACE_SEARCHED_NUM];
location_manager_h  l_manager;
int location_initialized = 0;
double map_start_lati_of_center = 37.259606;
double map_start_longti_of_center = 127.045828;
double x_moved = 0;
double y_moved = 0;
Ecore_Timer * get_geometry_timer = NULL;
static char temp_place_info[PLACE_INFO_MAX];

Eina_Lock      set_info_mutex;
extern int map_dload_q_idx;
int current_zoom_level = START_ZOOM_LEVEL;
certer_tile_info_s current_center;
double curr_user_lati;
double curr_user_longti;
char m_icon_path[100];

void update_main_page(void);
Eina_Bool get_changed_location(void *data);

double next_lati_value(double curr_lati, int diff, int curr_zoom)
{
	double lat;

	int y1 = floor((1.0 - log(tan(curr_lati * ELM_PI / 180.0) +
	                           (1.0 / cos(curr_lati * ELM_PI / 180.0)))
	                 / ELM_PI) / 2.0 * (Y_RESOLUTION*pow(2, curr_zoom)));

	double n = ELM_PI - (2.0 * ELM_PI * (y1+2.5*diff) / (Y_RESOLUTION*pow(2, curr_zoom)));
	lat = 180.0 / ELM_PI *atan(0.5 * (exp(n) - exp(-n)));

	return lat;
}

double next_longti_value(double curr_longti, int diff, int curr_zoom)
{
	double lon;

	int x1 = floor((curr_longti + 180.0) / 360.0 * (X_RESOLUTION*pow(2, curr_zoom)));

	lon = ((x1+1.875*diff) / ((double)X_RESOLUTION*pow(2, curr_zoom)) * 360.0) - 180;

	return lon;
}

double latitude_of_polar(int polar)// polar=1 : Arctic, polar=-1 : Antarctic
{
    double n, y, lat;

    if(polar==1)
    	y = 0;
    else
    	y = Y_RESOLUTION;
    n = ELM_PI - (2.0 * ELM_PI * y / Y_RESOLUTION);
    lat = 180.0 / ELM_PI *atan(0.5 * (exp(n) - exp(-n)));
    return lat;
}

void delete_all_marker_from_url(char* url)
{
	char tmp_url_1[MAX_URL_LEN]={0,}, tmp_url_2[MAX_URL_LEN]={0,};
	char* next_pos = NULL;
	char* pos_and_oper = NULL;

	next_pos = strstr(url, "&markers=");

	while(next_pos!=NULL)
	{
		pos_and_oper = strstr((next_pos+1), "&");
		memset(tmp_url_1, 0, MAX_URL_LEN);
		memcpy(tmp_url_1, url, (next_pos-url));
		memset(tmp_url_2, 0, MAX_URL_LEN);
		memcpy(tmp_url_2, pos_and_oper, strlen(url)-(pos_and_oper-url));
		memset(url, 0, MAX_URL_LEN);
		snprintf(url, MAX_URL_LEN, "%s%s", tmp_url_1, tmp_url_2);
		next_pos = strstr(url, "&markers=");
	}
}

int make_new_url(int reason, int zoom, void* data, double x_changed, double y_changed)
{
	char prev[MAX_URL_LEN]={0,}, new_url[MAX_URL_LEN]={0,}, tmp_str_changed[MAX_URL_LEN]={0,}, tmp_str_changed_1[MAX_URL_LEN]={0,}, tmp_str_end[MAX_URL_LEN]={0,};
	char *url_handle;
	char str_lati[GEO_INFO_STR_NUM]={0,}, str_longi[GEO_INFO_STR_NUM]={0,};
	char* pos = NULL;
	char* pos_and_oper = NULL;
	char* pos_comma = NULL;
	double   latitude, longitude;

	if(reason == PLACE_INFO)
	{
		url_handle = place_api_url;
	}
	else
	{
		url_handle = curr_url;
	}
	memcpy(prev, url_handle, strlen(url_handle));

	pos = strstr(prev, search_str[reason]);

	if(pos == NULL)
	{
		return URL_NOT_CHANGED;
	}

	pos_and_oper = strstr(pos, "&"); // to search first '&' from after "zoom=..."

	if(pos_and_oper == NULL)
	{
		return URL_NOT_CHANGED;
	}

	memcpy(tmp_str_end, pos_and_oper, strlen(prev) - ((int)pos_and_oper-(int)prev));

	switch (reason)
	{
		case ZOOM_CHANGED:
		{
			if(zoom>0)
			{
				if(current_zoom_level<MAX_ZOOM_SCALE) current_zoom_level++;
				else return URL_NOT_CHANGED;
			}
			else if(zoom<0)
			{
				if(current_zoom_level>MIN_ZOOM_SCALE) current_zoom_level--;
				else return URL_NOT_CHANGED;
			}
			snprintf(tmp_str_changed, sizeof(tmp_str_changed), "zoom=%d", current_zoom_level);
		}
		break;

		case CENTER_NAME_CHANGED:
		{
			snprintf(tmp_str_changed, sizeof(tmp_str_changed), "center=%s", (char*)data);
		}
		break;

		case CENTER_GEOMETRY_CHANGED:
		{
			pos_comma = strstr(pos, ","); // to search first ',' from after "center=..."

			memcpy(str_lati, (char*)((int)pos+strlen("center=")), (int)pos_comma - ((int)pos+strlen("center=")));
			memcpy(str_longi, (char*)((int)pos_comma+1), ((int)pos_and_oper-((int)pos_comma+1)) );
			latitude = atof(str_lati);
			longitude = atof(str_longi);

			if(latitude>90) latitude=90;
			if(latitude<-90) latitude=-90;

			if(longitude>180) longitude=(-1)*(360-longitude);
			if(longitude<-180) longitude=(longitude+360);

			snprintf(tmp_str_changed, sizeof(tmp_str_changed), "center=%f,%f", latitude, longitude);
		}
		break;

		case ADD_NEW_MARKER:
		{
			char tmp_char = '%';
			snprintf(tmp_str_changed, sizeof(tmp_str_changed), "&markers=color:blue%c7Clabel:U%c7C%f,%f", tmp_char, tmp_char, y_changed, x_changed);
		}
		break;

		case ADD_NEW_MULTI_MARKER:
		{
			char tmp_char = '%';
			int i = place_searched_num;

			while(i--)
			{
				snprintf(tmp_str_changed_1, sizeof(tmp_str_changed_1), "%s&markers=color:red%c7Clabel:S%c7C%f,%f", tmp_str_changed, tmp_char, tmp_char, place_search_list[i].str_lati, place_search_list[i].str_longti);
				snprintf(tmp_str_changed, sizeof(tmp_str_changed), "%s", tmp_str_changed_1);
			}
		}
		break;

		case DELITE_MARKER:
		{
			//do not process tmp_str_changed because we just need to set tmp_str_changed=""
			delete_all_marker_from_url(tmp_str_end);// remove all "&markers.." from tmp_str_end
		}
		break;

		case PLACE_INFO:
		{
			snprintf(tmp_str_changed, sizeof(tmp_str_changed), "query=%s", (char*)data);
		}
		break;

		default :
			break;
	}

	memcpy(&new_url[0], prev, ((int)pos-(int)prev));
	memcpy(&new_url[((int)pos-(int)prev)], tmp_str_changed, strlen(tmp_str_changed));
	memcpy(&new_url[((int)pos-(int)prev)+strlen(tmp_str_changed)], tmp_str_end, strlen(tmp_str_end));

	memset(url_handle, 0, MAX_URL_LEN);
	memcpy(url_handle, new_url, strlen(new_url));

	return URL_CHANGED;
}

void request_map_download(int x_info, int y_info, double latitude, double longitude)
{
	char buf[100];

	if(all_tiles[x_info][y_info].download_request)
		return;

	all_tiles[x_info][y_info].download_request = 1;

	memset(all_tiles[x_info][y_info].file_name, 0, MAX_TMP_FILENAME_LEN);

	if(longitude>200 || longitude<-200 || latitude<latitude_of_polar(-1) || latitude>latitude_of_polar(1))
	{
		snprintf(all_tiles[x_info][y_info].file_name, sizeof(all_tiles[x_info][y_info].file_name), "%s/white.PNG", m_icon_path);
		return;
	}

	all_tiles[x_info][y_info].lati_of_center = latitude;
	all_tiles[x_info][y_info].longti_of_center = longitude;
	snprintf(all_tiles[x_info][y_info].file_name, sizeof(all_tiles[x_info][y_info].file_name), "%s/%d_%f,%f.PNG", TMP_DIR, current_zoom_level, latitude, longitude);

	memset(buf, 0, 100);
	snprintf(buf, sizeof(buf), "%f,%f", latitude, longitude);

	make_new_url(CENTER_NAME_CHANGED, 0, buf, 0, 0);

	eina_lock_take(&set_info_mutex);
	memset(&map_dload_q[map_dload_q_idx].url[0], 0, MAX_URL_LEN);
	memcpy(&map_dload_q[map_dload_q_idx].url[0], curr_url, MAX_URL_LEN);
	memset(&map_dload_q[map_dload_q_idx].filename[0], 0, MAX_TMP_FILENAME_LEN);
	memcpy(&map_dload_q[map_dload_q_idx].filename[0], all_tiles[x_info][y_info].file_name, MAX_TMP_FILENAME_LEN);
	map_dload_q[map_dload_q_idx].index.x = x_info;
	map_dload_q[map_dload_q_idx].index.y = y_info;
	map_dload_q_idx++;
	eina_lock_release(&set_info_mutex);
}

int save_map_temp_file(char* url, char* file_name)
{
	CURL *curl_handle;
	FILE *currfile;
	int ret_val=0;

	curl_global_init(CURL_GLOBAL_ALL);

	/* init the curl session */
	curl_handle = curl_easy_init();

	/* set URL to get */
	curl_easy_setopt(curl_handle, CURLOPT_URL, url);

	/* no progress meter please */
	curl_easy_setopt(curl_handle, CURLOPT_FOLLOWLOCATION, 1L);

	/* open the files */
	currfile = fopen(file_name,"w");
	if (currfile == NULL) {
		curl_easy_cleanup(curl_handle);
		return -1;
	}

	/* we want the headers to this file handle */
	curl_easy_setopt(curl_handle,   CURLOPT_WRITEDATA, currfile);

	/*
	 *    * Notice here that if you want the actual data sent anywhere else but
	 *       * stdout, you should consider using the CURLOPT_WRITEDATA option.  */

	/* get it! */
	ret_val = curl_easy_perform(curl_handle);

	/* close the header file */
	fclose(currfile);

	/* cleanup curl stuff */
	curl_easy_cleanup(curl_handle);

	return ret_val;
}

int map_download(char* tmp_url, char* tmp_filename)
{
	if(save_map_temp_file(tmp_url, tmp_filename) != 0)
	{
		return 0;//fail
	}

	return 1;//success
}

void set_map_main_tile_info(int current_center_x, int current_center_y)
{
	for(int i=0; i<MAP_TILE_X ; i++)
	{
		for(int j=0; j<MAP_TILE_Y ; j++)
		{
			memset(tile_info[i][j].file_name, 0, MAX_TMP_FILENAME_LEN);

			if(all_tiles[current_center_x-(1-i)][current_center_y-(1-j)].download_ok)
			{
				memcpy(tile_info[i][j].file_name, all_tiles[current_center_x-(1-i)][current_center_y-(1-j)].file_name,
					strlen(all_tiles[current_center_x-(1-i)][current_center_y-(1-j)].file_name));
			}
			else
			{
				snprintf(tile_info[i][j].file_name, sizeof(tile_info[i][j].file_name), "%s/NULL.PNG", m_icon_path);
			}
		}
	}
	current_center.x_info = current_center_x;
	current_center.y_info = current_center_y;
}

void mouse_move_cb(void *data, Evas *e , Evas_Object *obj , void *event_info)
{
	Evas_Event_Mouse_Move *ev;
	ev = (Evas_Event_Mouse_Move *)event_info;
	//appdata_s *ad = data;
	mapdata_s *md = data;

	if(md->mouse_down == EINA_TRUE)
	{
		if((ev->cur.canvas.x != ev->prev.canvas.x) || (ev->cur.canvas.y != ev->prev.canvas.y))
		{
			ecore_timer_del(get_geometry_timer);
		}

		// x point check
		if(ev->cur.canvas.x > ev->prev.canvas.x) //x++
		{
			if(tile_info[1][1].curr_x > 0)
			{
				for(int j=0 ; j<MAP_TILE_Y ; j++)
				{
					if(! all_tiles[current_center.x_info-2][current_center.y_info+(1-j)].download_request)
					{
						double lati_of_center = all_tiles[current_center.x_info][current_center.y_info].lati_of_center;
						double longti_of_center = all_tiles[current_center.x_info][current_center.y_info].longti_of_center;
						double longti_for_end_check;

						lati_of_center = next_lati_value(lati_of_center, Y_RESOLUTION*(j-1), current_zoom_level);
						longti_of_center = next_longti_value(longti_of_center, X_RESOLUTION*(-1), current_zoom_level);
						longti_of_center = next_longti_value(longti_of_center, X_RESOLUTION*(-1), current_zoom_level);

						request_map_download(current_center.x_info-2, current_center.y_info+(1-j), lati_of_center, longti_of_center);
					}
				}
			}

			for(int i=0; i<MAP_TILE_X ; i++)
			{
				for(int j=0; j<MAP_TILE_Y ; j++)
				{
					tile_info[i][j].curr_x = tile_info[i][j].curr_x+(ev->cur.canvas.x - ev->prev.canvas.x);
					evas_object_move(main_page[i][j], tile_info[i][j].curr_x, tile_info[i][j].curr_y);
				}
			}
			x_moved = x_moved + (ev->cur.canvas.x - ev->prev.canvas.x);

			if(tile_info[1][1].curr_x > X_RESOLUTION)
			{
				set_map_main_tile_info(current_center.x_info-1, current_center.y_info);
				update_main_page();

				for(int i=0; i<MAP_TILE_X ; i++)
				{
					for(int j=0; j<MAP_TILE_Y ; j++)
					{
						evas_object_move(main_page[i][j], tile_info[i][j].curr_x-X_RESOLUTION, tile_info[i][j].curr_y);
						tile_info[i][j].curr_x = tile_info[i][j].curr_x-X_RESOLUTION;
					}
				}
			}
		}
		else if(ev->cur.canvas.x < ev->prev.canvas.x) //x--
		{
			if(tile_info[1][1].curr_x < 0)
			{
				for(int j=0 ; j<MAP_TILE_Y ; j++)
				{
					if(! all_tiles[current_center.x_info+2][current_center.y_info+(1-j)].download_request)
					{
						double lati_of_center = all_tiles[current_center.x_info][current_center.y_info].lati_of_center;
						double longti_of_center = all_tiles[current_center.x_info][current_center.y_info].longti_of_center;
						double longti_for_end_check;

						lati_of_center = next_lati_value(lati_of_center, Y_RESOLUTION*(j-1), current_zoom_level);
						longti_of_center = next_longti_value(longti_of_center, X_RESOLUTION, current_zoom_level);
						longti_of_center = next_longti_value(longti_of_center, X_RESOLUTION, current_zoom_level);

						request_map_download(current_center.x_info+2, current_center.y_info+(1-j), lati_of_center, longti_of_center);
					}
				}
			}

			for(int i=0; i<MAP_TILE_X ; i++)
			{
				for(int j=0; j<MAP_TILE_Y ; j++)
				{
					tile_info[i][j].curr_x = tile_info[i][j].curr_x+(ev->cur.canvas.x - ev->prev.canvas.x);
					evas_object_move(main_page[i][j], tile_info[i][j].curr_x, tile_info[i][j].curr_y);
				}
			}
			x_moved = x_moved + (ev->cur.canvas.x - ev->prev.canvas.x);

			if(tile_info[1][1].curr_x < -(X_RESOLUTION-80))
			{
				set_map_main_tile_info(current_center.x_info+1, current_center.y_info);
				update_main_page();

				for(int i=0; i<MAP_TILE_X ; i++)
				{
					for(int j=0; j<MAP_TILE_Y ; j++)
					{
						evas_object_move(main_page[i][j], tile_info[i][j].curr_x+X_RESOLUTION, tile_info[i][j].curr_y);
						tile_info[i][j].curr_x = tile_info[i][j].curr_x+X_RESOLUTION;
					}
				}
			}
		}

		// y point check
		if(ev->cur.canvas.y > ev->prev.canvas.y) // y++
		{
			if(tile_info[1][1].curr_y > 0)
			{
				for(int i=0 ; i<MAP_TILE_Y ; i++)
				{
					if(! all_tiles[current_center.x_info-(1-i)][current_center.y_info+2].download_request)
					{
						double lati_of_center = all_tiles[current_center.x_info][current_center.y_info].lati_of_center;
						double longti_of_center = all_tiles[current_center.x_info][current_center.y_info].longti_of_center;

						lati_of_center = next_lati_value(lati_of_center, Y_RESOLUTION*(-1), current_zoom_level);
						lati_of_center = next_lati_value(lati_of_center, Y_RESOLUTION*(-1), current_zoom_level);
						longti_of_center = next_longti_value(longti_of_center, X_RESOLUTION*(i-1), current_zoom_level);

						request_map_download(current_center.x_info-(1-i), current_center.y_info+2, lati_of_center, longti_of_center);
					}
				}
			}

			for(int i=0; i<MAP_TILE_X ; i++)
			{
				for(int j=0; j<MAP_TILE_Y ; j++)
				{
					tile_info[i][j].curr_y = tile_info[i][j].curr_y+(ev->cur.canvas.y - ev->prev.canvas.y);
					evas_object_move(main_page[i][j], tile_info[i][j].curr_x, tile_info[i][j].curr_y);
				}
			}
			y_moved = y_moved + (ev->cur.canvas.y - ev->prev.canvas.y);

			if(tile_info[1][1].curr_y > Y_RESOLUTION)
			{
				set_map_main_tile_info(current_center.x_info, current_center.y_info+1);
				update_main_page();

				for(int i=0; i<MAP_TILE_X ; i++)
				{
					for(int j=0; j<MAP_TILE_Y ; j++)
					{
						evas_object_move(main_page[i][j], tile_info[i][j].curr_x, tile_info[i][j].curr_y-Y_RESOLUTION);
						tile_info[i][j].curr_y = tile_info[i][j].curr_y-Y_RESOLUTION;
					}
				}
			}
		}
		else if(ev->cur.canvas.y < ev->prev.canvas.y) // y--
		{
			if(tile_info[1][1].curr_y < 0)
			{
				for(int i=0 ; i<MAP_TILE_Y ; i++)
				{
					if(! all_tiles[current_center.x_info-(1-i)][current_center.y_info-2].download_request)
					{
						double lati_of_center = all_tiles[current_center.x_info][current_center.y_info].lati_of_center;
						double longti_of_center = all_tiles[current_center.x_info][current_center.y_info].longti_of_center;

						lati_of_center = next_lati_value(lati_of_center, Y_RESOLUTION, current_zoom_level);
						lati_of_center = next_lati_value(lati_of_center, Y_RESOLUTION, current_zoom_level);
						longti_of_center = next_longti_value(longti_of_center, X_RESOLUTION*(i-1), current_zoom_level);

						request_map_download(current_center.x_info-(1-i), current_center.y_info-2, lati_of_center, longti_of_center);
					}
				}
			}

			for(int i=0; i<MAP_TILE_X ; i++)
			{
				for(int j=0; j<MAP_TILE_Y ; j++)
				{
					tile_info[i][j].curr_y = tile_info[i][j].curr_y+(ev->cur.canvas.y - ev->prev.canvas.y);
					evas_object_move(main_page[i][j], tile_info[i][j].curr_x, tile_info[i][j].curr_y);
				}
			}
			y_moved = y_moved + (ev->cur.canvas.y - ev->prev.canvas.y);

			if(tile_info[1][1].curr_y < -(Y_RESOLUTION-200))
			{
				set_map_main_tile_info(current_center.x_info, current_center.y_info-1);
				update_main_page();

				for(int i=0; i<MAP_TILE_X ; i++)
				{
					for(int j=0; j<MAP_TILE_Y ; j++)
					{
						evas_object_move(main_page[i][j], tile_info[i][j].curr_x, tile_info[i][j].curr_y+Y_RESOLUTION);
						tile_info[i][j].curr_y = tile_info[i][j].curr_y+Y_RESOLUTION;
					}
				}
			}
		}
	}
}

void mouse_up_cb(void *data, Evas *e , Evas_Object *obj , void *event_info)
{
	Evas_Event_Mouse_Move *ev;
	ev = (Evas_Event_Mouse_Move *)event_info;

	//appdata_s *ad = data;
	mapdata_s *md = data;
	md->mouse_down = EINA_FALSE;
}

void mouse_down_cb(void *data, Evas *e , Evas_Object *obj , void *event_info)
{
	Evas_Event_Mouse_Move *ev;
	ev = (Evas_Event_Mouse_Move *)event_info;

	//appdata_s *ad = data;
	mapdata_s *md = data;
	md->mouse_down = EINA_TRUE;
}

void update_main_page(void)
{
	for(int i=0; i<MAP_TILE_X ; i++)
	{
		for(int j=0; j<MAP_TILE_Y ; j++)
		{
			if(main_page[i][j] != NULL)
			{
				evas_object_del(main_page[i][j]);
				main_page[i][j] = NULL;
			}

			main_page[i][j] = evas_object_image_filled_add(m_canvas);

			evas_object_image_file_set(main_page[i][j], tile_info[i][j].file_name, NULL);
			evas_object_move(main_page[i][j], tile_info[i][j].curr_x, tile_info[i][j].curr_y);
			evas_object_resize(main_page[i][j], X_RESOLUTION, Y_RESOLUTION);
			evas_object_show(main_page[i][j]);
			evas_object_raise (m_btn1);
			evas_object_raise (m_btn2);

			evas_object_event_callback_add(main_page[i][j], EVAS_CALLBACK_MOUSE_MOVE, mouse_move_cb, m_md);
			evas_object_event_callback_add(main_page[i][j], EVAS_CALLBACK_MOUSE_DOWN, mouse_down_cb, m_md);
			evas_object_event_callback_add(main_page[i][j], EVAS_CALLBACK_MOUSE_UP, mouse_up_cb, m_md);
		}
	}
}

void map_dload_thread(void *data, Ecore_Thread *thread)
{
	int dload_success=0;

   while (1)
   {
	   while(map_dload_q_idx && !do_not_update_map)
	   {
		   char dload_url[MAX_URL_LEN] = {0,};
		   char dload_filename[MAX_TMP_FILENAME_LEN] = {0,};
		   index_of_all_tile_s index;

		   eina_lock_take(&set_info_mutex);
		   map_dload_q_idx--;
		   memcpy(dload_url, &map_dload_q[map_dload_q_idx].url[0], MAX_URL_LEN);
		   memcpy(dload_filename, &map_dload_q[map_dload_q_idx].filename[0], MAX_TMP_FILENAME_LEN);
		   index.x = map_dload_q[map_dload_q_idx].index.x;
		   index.y = map_dload_q[map_dload_q_idx].index.y;
		   eina_lock_release(&set_info_mutex);

		   dload_success = 0;
		   do
		   {
			   dload_success = map_download(dload_url, dload_filename);
		   } while(dload_success==0);

		   all_tiles[index.x][index.y].download_ok = 1;
		   set_map_main_tile_info(current_center.x_info, current_center.y_info);
		   update_main_page();

		   //ecore_thread_feedback(thread, &index);
	   }
   }
}

void  thread_feedback(void *data, Ecore_Thread *thread, void *msg_data)
{
	//TODO
}

void  thread_end(void *data, Ecore_Thread *thread)
{
	//TODO
}

void  thread_cancel(void *data, Ecore_Thread *thread)
{
	//TODO
}

void loc_state_changed_cb(location_service_state_e state, void *user_data)
{
	if(state == LOCATIONS_SERVICE_ENABLED)
		location_initialized = 1;
}

int make_start_url(double lati, double longti)
{
	char buf[GEO_INFO_STR_NUM];
	map_start_lati_of_center = lati;
	map_start_longti_of_center = longti;

	if(make_new_url(ZOOM_CHANGED, 0, NULL, 0, 0) != URL_CHANGED)
		return URL_NOT_CHANGED;

	memset(buf, 0, GEO_INFO_STR_NUM);
	snprintf(buf, sizeof(buf), "%f,%f", map_start_lati_of_center, map_start_longti_of_center);
	if(make_new_url(CENTER_NAME_CHANGED, 0, buf, 0, 0) != URL_CHANGED)
		return URL_NOT_CHANGED;

	return URL_CHANGED;
}

void arrange_start_main_page(void)
{
	map_dload_q_idx = 0;
	do_not_update_map = 1;

	x_moved = 0;
	y_moved = 0;

	current_center.x_info = ALL_TILE_USABLE_X/2;
	current_center.y_info = ALL_TILE_USABLE_Y/2;

	for(int i=0; i<ALL_TILE_USABLE_X ; i++)
	{
		for(int j=0; j<ALL_TILE_USABLE_Y ; j++)
		{
			all_tiles[i][j].download_ok = 0;
			all_tiles[i][j].download_request = 0;
			memset(all_tiles[i][j].file_name, 0, MAX_TMP_FILENAME_LEN);
		}
	}

	for(int i=0; i<MAP_TILE_X ; i++)
	{
		for(int j=0; j<MAP_TILE_Y ; j++)
		{
			tile_info[i][j].curr_x = tile_info_init_information[i][j].curr_x;
			tile_info[i][j].curr_y = tile_info_init_information[i][j].curr_y;

			if(i==1 && j==1) continue;
			request_map_download(current_center.x_info-(1-i), current_center.y_info-(1-j),
					next_lati_value(map_start_lati_of_center, Y_RESOLUTION*(1-j), current_zoom_level),
					next_longti_value(map_start_longti_of_center, X_RESOLUTION*(i-1), current_zoom_level));
		}
	}
	//download ceter map first for visual effect
	request_map_download(current_center.x_info, current_center.y_info, map_start_lati_of_center, map_start_longti_of_center);
	do_not_update_map = 0;

	set_map_main_tile_info(current_center.x_info, current_center.y_info);
}

void map_zoom_in()
{
	if(make_new_url(ZOOM_CHANGED, 1, NULL, 0, 0) == URL_CHANGED)
	{
		map_start_lati_of_center = next_lati_value(map_start_lati_of_center, y_moved*(-1), current_zoom_level-1);
		map_start_longti_of_center = next_longti_value(map_start_longti_of_center, -x_moved, current_zoom_level-1);

		arrange_start_main_page();
	}
}

void map_zoom_out()
{
	if(make_new_url(ZOOM_CHANGED, -1, NULL, 0, 0) == URL_CHANGED)
	{
		map_start_lati_of_center = next_lati_value(map_start_lati_of_center, y_moved*(-1), current_zoom_level+1);
		map_start_longti_of_center = next_longti_value(map_start_longti_of_center, -x_moved, current_zoom_level+1);

		arrange_start_main_page();
	}
}

static size_t
WriteMemoryCallback(void *contents, size_t size, size_t nmemb, void *userp)
{
	size_t realsize = size * nmemb;
	struct MemoryStruct *mem = (struct MemoryStruct *)userp;

	mem->memory = realloc(mem->memory, mem->size + realsize + 1);
	if(mem->memory == NULL) {
		/* out of memory! */
		printf("not enough memory (realloc returned NULL)\n");
		return 0;
	}

	memcpy(&(mem->memory[mem->size]), contents, realsize);
	mem->size += realsize;
	mem->memory[mem->size] = 0;

	return realsize;
}

int get_geometry_info_of_place(char* url, double* start_latitude, double* start_longitude)
{
	CURL *curl_handle;
	MemoryStruct_s chunk;
	int ret_val=0;
	char* search_index = NULL;
	char* pos = NULL;
	char* pos_lati_start = NULL;
	char* pos_lati_end = NULL;
	char* pos_longti_start = NULL;
	char* pos_longti_end = NULL;
	char tmp_str_lati[GEO_INFO_STR_NUM]={0,}, tmp_str_longti[GEO_INFO_STR_NUM]={0,};
	double tmp_lati_sum=0, tmp_longti_sum=0;

	chunk.memory = malloc(1);  /* will be grown as needed by the realloc above */
	chunk.size = 0;    /* no data at this point */

	curl_global_init(CURL_GLOBAL_ALL);

	/* init the curl session */
	curl_handle = curl_easy_init();

	/* set URL to get */
	curl_easy_setopt(curl_handle, CURLOPT_URL, url);
	curl_easy_setopt(curl_handle, CURLOPT_SSL_VERIFYPEER, 0L);

	/* no progress meter please */
	//curl_easy_setopt(curl_handle, CURLOPT_FOLLOWLOCATION, 1L);

	/* send all data to this function  */
	curl_easy_setopt(curl_handle, CURLOPT_WRITEFUNCTION, WriteMemoryCallback);

	/* we want the headers to this file handle */
	curl_easy_setopt(curl_handle,   CURLOPT_WRITEDATA, (void *)&chunk);

	/* some servers don't like requests that are made without a user-agent
	     field, so we provide one */
	curl_easy_setopt(curl_handle, CURLOPT_USERAGENT, "libcurl-agent/1.0");

	/*
	 *    * Notice here that if you want the actual data sent anywhere else but
	 *       * stdout, you should consider using the CURLOPT_WRITEDATA option.  */

	/* get it! */
	ret_val = curl_easy_perform(curl_handle);

	memset(temp_place_info, 0, PLACE_INFO_MAX);
	memcpy(temp_place_info, chunk.memory, chunk.size);

	/* cleanup curl stuff */
	curl_easy_cleanup(curl_handle);

	if(chunk.memory)
	    free(chunk.memory);

	  /* we're done with libcurl, so clean it up */
	curl_global_cleanup();

	place_searched_num = 0;
	search_index = temp_place_info;
	while(1)
	{
		pos = strstr(search_index, "<location>");
		if(pos != NULL)// detected one more
		{
			pos_lati_start = strstr((char*)pos, "<lat>");
			pos_lati_end = strstr((char*)pos, "</lat>");
			pos_longti_start = strstr((char*)pos, "<lng>");
			pos_longti_end = strstr((char*)pos, "</lng>");

			memset(tmp_str_lati, 0, GEO_INFO_STR_NUM);
			memcpy(tmp_str_lati, (char*)(pos_lati_start+5), pos_lati_end-(pos_lati_start+5));
			place_search_list[place_searched_num].str_lati = atof(tmp_str_lati);

			memset(tmp_str_longti, 0, GEO_INFO_STR_NUM);
			memcpy(tmp_str_longti, (char*)(pos_longti_start+5), pos_longti_end-(pos_longti_start+5));
			place_search_list[place_searched_num].str_longti = atof(tmp_str_longti);

			place_searched_num++;
			search_index = pos+1;//just for next search
		}
		else
		{
			break; //couldn't find anymore.
		}
	}

	for(int i=0 ; i<place_searched_num ; i++)
	{
		tmp_lati_sum = tmp_lati_sum + place_search_list[i].str_lati;
		tmp_longti_sum = tmp_longti_sum + place_search_list[i].str_longti;
	}

	*start_latitude = place_search_list[0].str_lati;
	*start_longitude = place_search_list[0].str_longti;

	return ret_val;
}

//handle key event for search
void key_down_cb(void *data, Evas *evas, Evas_Object *obj, void *event_info)
{
	const int MAX_CUR = 20;
	static char buf[50];
	static int cur = 0;
	double latitude, longitude;
	static char geometry_buf[50]={0,};

	Evas_Event_Key_Down *ev = event_info;
	Evas_Object *input = data;
	char tmp[50];

	if (cur == 0) snprintf(buf, sizeof(buf), "\0");

	if (!strcmp(ev->keyname, "Return"))
	{
		evas_object_text_text_set(input, "");
		cur = 0;
		for(int i =0 ; i<strlen(buf) ; i++)
		{
			if(buf[i]==' ') buf[i]=','; // google api can't recognize ' '
		}
		if(make_new_url(PLACE_INFO, 0, buf, 0, 0) == URL_CHANGED)//make url for query of place info.
		{
			if(!get_geometry_info_of_place(place_api_url, &latitude, &longitude)) // get place info data using query url and parse it.
			{
				memset(geometry_buf, 0, 50);
				snprintf(geometry_buf, sizeof(geometry_buf), "%f,%f", latitude, longitude);
				make_new_url(DELITE_MARKER, 0, NULL, 0, 0); // delete all previous markers
				if((make_new_url(CENTER_NAME_CHANGED, 0, geometry_buf, 0, 0) == URL_CHANGED)
				&& (make_new_url(ADD_NEW_MULTI_MARKER, 0, NULL, 0, 0) == URL_CHANGED))
				{
					map_start_lati_of_center = latitude;
					map_start_longti_of_center = longitude;
					arrange_start_main_page();
					update_main_page();
				}
			}
		}
		return;
	}

	if (!strcmp(ev->keyname, "BackSpace"))
	{
		snprintf(tmp, strlen(buf), "%s", buf);
		evas_object_text_text_set(input, tmp);
		strcpy(buf, tmp);
		cur--;
		return;
	}

	if (cur >= MAX_CUR) return;

	if(!strcmp(ev->keyname, "space"))
	{
		snprintf(tmp, sizeof(tmp), "%s%s", buf, " "); // replace ' ' to ' '
		evas_object_text_text_set(input, tmp);
		cur++;
		strcpy(buf, tmp);
	}
	else if(!strcmp(ev->keyname, "comma"))
	{
		snprintf(tmp, sizeof(tmp), "%s%s", buf, ",");
		evas_object_text_text_set(input, tmp);
		cur++;
		strcpy(buf, tmp);
	}
	else if((!strcmp(ev->keyname, "Caps_Lock"))
			|| !strcmp(ev->keyname, "Shift_L")
			|| !strcmp(ev->keyname, "Num_Lock")
			|| !strcmp(ev->keyname, "Left")
			|| !strcmp(ev->keyname, "Right")
			|| !strcmp(ev->keyname, "Up")
			|| !strcmp(ev->keyname, "Down"))
	{
		// to be ignored...
	}
	else
	{
		snprintf(tmp, sizeof(tmp), "%s%s", buf, ev->keyname);
		evas_object_text_text_set(input, tmp);
		cur++;
		strcpy(buf, tmp);
	}
}

void go_to_current_geometry(void)
{
	double altitude, climb, direction, speed;
	double horizontal, vertical;location_accuracy_level_e level;time_t timestamp;
	char buf[100];

	if(!location_initialized)
		return;

	location_manager_get_last_location(l_manager, &altitude, &curr_user_lati, &curr_user_longti,
												 &climb, &direction, &speed, &level, &horizontal, &vertical, &timestamp);

	memset(buf, 0, 100);
	snprintf(buf, sizeof(buf), "%f,%f", curr_user_lati, curr_user_longti);
	if(make_new_url(CENTER_NAME_CHANGED, 0, buf, 0, 0) != URL_CHANGED)
		return;
	make_new_url(DELITE_MARKER, 0, NULL, 0, 0);
	if(make_new_url(ADD_NEW_MARKER, 0, buf, curr_user_longti, curr_user_lati) != URL_CHANGED)
		return;

	map_start_lati_of_center = curr_user_lati;
	map_start_longti_of_center = curr_user_longti;

	arrange_start_main_page();

	get_geometry_timer = ecore_timer_add(1, get_changed_location, NULL);
}

Eina_Bool get_changed_location(void *data)
{
	double altitude, latitude, longtitude, climb, direction, speed;
	double horizontal, vertical;location_accuracy_level_e level;time_t timestamp;

	if(!location_initialized)
		return ECORE_CALLBACK_RENEW;

	location_manager_get_last_location(l_manager, &altitude, &latitude, &longtitude,
		                                         &climb, &direction, &speed, &level, &horizontal, &vertical, &timestamp);

	if((latitude != curr_user_lati) || (longtitude != curr_user_longti))
	{
		ecore_timer_del(get_geometry_timer);
		go_to_current_geometry();
		return ECORE_CALLBACK_CANCEL;
	}
	else
	{
		return ECORE_CALLBACK_RENEW;
	}
}

//go to current user's point
void current_btn_cb(void *data, Evas *e , Evas_Object *obj , void *event_info)
{
	go_to_current_geometry();
}
static void
btn_clicked_cb(void *data, Evas_Object *obj, void *event_info)
{
	int btn_num = (int)data;
	//dlog_print(DLOG_ERROR, "tag", "clicked event on Button:%d", btn_num);

	switch( btn_num ) {
	case 1 :
		map_zoom_in();
		break;
	case 2 :
		map_zoom_out();
		break;
	}
}

void create_map(Evas_Object *win, double lati, double longti)
{
	m_md = (mapdata_s*)malloc( sizeof( mapdata_s) );

	/* Button-1 */
	m_btn1 = elm_button_add(win);
	elm_object_text_set(m_btn1, "+");
	evas_object_move(m_btn1, 20, 20);
	evas_object_resize(m_btn1, 50, 50);
	evas_object_smart_callback_add(m_btn1, "clicked", btn_clicked_cb, (void *)1);
	evas_object_show(m_btn1);

	/* Button-2 */
	m_btn2 = elm_button_add(win);
	elm_object_text_set(m_btn2, "-");
	evas_object_move(m_btn2, 90, 20);
	evas_object_resize(m_btn2, 50, 50);
	evas_object_smart_callback_add(m_btn2, "clicked", btn_clicked_cb, (void *)2);
	evas_object_show(m_btn2);

	/* Canvas */
	m_canvas = evas_object_evas_get(win);

	char *res_path = app_get_resource_path();
	if (res_path) {
		snprintf(m_icon_path, PATH_MAX, "%s%s", res_path, "images");
		free(res_path);
	}

    /* Thread */
    ecore_thread_feedback_run(map_dload_thread, thread_feedback, thread_end, thread_cancel, NULL, EINA_TRUE);
    eina_lock_new(&set_info_mutex);

    /* LocationManager */
    location_manager_create(LOCATIONS_METHOD_GPS, &l_manager);
    location_manager_set_service_state_changed_cb(l_manager, loc_state_changed_cb, NULL) ;
    location_manager_start(l_manager);

    mkdir(TMP_DIR, 0755);

    if( make_start_url(lati, longti) != URL_CHANGED ) //let's make start url to load
    	return;

    arrange_start_main_page();
}


#endif /* MAPVIEW_H_ */
