﻿/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

using System;
using System.Collections.Generic;
using Xamarin.Forms;

namespace GalleryUI
{
    /// <summary>
    /// The main page of the gallery application.
    /// </summary>
    public class MainPage : ContentPage
    {
        // The portrait size of thumbnail.
        private int PORTRAIT_SIZE = (int)(0.084 * App.screenHeight);

        /// <summary>
        /// Initializes a new instance of the <see cref="MainPage"/> class.
        /// </summary>
        /// <param name="isLandscape">Is landscape or not.</param>
        public MainPage(bool isLandscape) : base()
        {
            // Create a relative layout.
            RelativeLayout relativeLayout = new RelativeLayout()
            {
                IsVisible = true,
            };

            // Create header label.
            Label headerLabel = CreateHeaderLabel();

            // Add header label into main page layout.
            relativeLayout.Children.Add(
                    headerLabel,
                    Constraint.Constant(0),
                    Constraint.Constant(0),
                    Constraint.RelativeToParent((parent) =>
                    {
                        return parent.Width;
                    }),
                    Constraint.Constant(0.078 * App.screenHeight));

            Grid grid = null;

            if (isLandscape)
            {
                // Create a Grid object to show the thumbnail for image for landscape.
                grid = CreateLandscapeGrid();
            }
            else
            {
                // Create a Grid object to show the thumbnail for image for portrait.
                grid = CreatePortraitGrid();
            }

            // Add grid view into layout.
            relativeLayout.Children.Add(
                grid,
                Constraint.Constant(0),
                Constraint.Constant(0.078 * App.screenHeight),
                Constraint.RelativeToParent((parent) =>
                {
                    return parent.Width;
                }),
                Constraint.RelativeToParent((parent) =>
                {
                    return parent.Height - (0.078 * App.screenHeight);
                }));

            // Build the page.
            Content = relativeLayout;
        }

        /// <summary>
        /// Create grid object for portrait.
        /// </summary>
        /// <returns>The grid for portrait.</returns>
        private Grid CreatePortraitGrid()
        {
            // Create a Grid object to show the thumbnail for image.
            Grid grid = new Grid()
            {
                Padding = 20,
                VerticalOptions = LayoutOptions.FillAndExpand,
                RowDefinitions =
                {
                    // Two rows.
                    new RowDefinition { Height = new GridLength(PORTRAIT_SIZE, GridUnitType.Absolute) },
                    new RowDefinition { Height = new GridLength(PORTRAIT_SIZE, GridUnitType.Absolute) },
                },
                ColumnDefinitions =
                {
                    // Six columns.
                    new ColumnDefinition { Width = new GridLength(PORTRAIT_SIZE, GridUnitType.Absolute) },
                    new ColumnDefinition { Width = new GridLength(PORTRAIT_SIZE, GridUnitType.Absolute) },
                    new ColumnDefinition { Width = new GridLength(PORTRAIT_SIZE, GridUnitType.Absolute) },
                    new ColumnDefinition { Width = new GridLength(PORTRAIT_SIZE, GridUnitType.Absolute) },
                    new ColumnDefinition { Width = new GridLength(PORTRAIT_SIZE, GridUnitType.Absolute) },
                    new ColumnDefinition { Width = new GridLength(PORTRAIT_SIZE, GridUnitType.Absolute) },
                },
            };

            // Put image into grid view one by one.
            grid.Children.Add(new Image
            {
                Source = "0.jpg",
                Aspect = Aspect.Fill,
            }, 0, 1, 0, 1);

            grid.Children.Add(new Image
            {
                Source = "1.jpg",
                Aspect = Aspect.Fill,
            }, 1, 2, 0, 1);

            grid.Children.Add(new Image
            {
                Source = "2.jpg",
                Aspect = Aspect.Fill,
            }, 2, 3, 0, 1);

            grid.Children.Add(new Image
            {
                Source = "3.jpg",
                Aspect = Aspect.Fill,
            }, 3, 4, 0, 1);

            grid.Children.Add(new Image
            {
                Source = "4.jpg",
                Aspect = Aspect.Fill,
            }, 4, 5, 0, 1);

            grid.Children.Add(new Image
            {
                Source = "5.jpg",
                Aspect = Aspect.Fill,
            }, 5, 6, 0, 1);

            grid.Children.Add(new Image
            {
                Source = "6.jpg",
                Aspect = Aspect.Fill,
            }, 0, 1, 1, 2);

            grid.Children.Add(new Image
            {
                Source = "7.jpg",
                Aspect = Aspect.Fill,
            }, 1, 2, 1, 2);

            grid.Children.Add(new Image
            {
                Source = "8.jpg",
                Aspect = Aspect.Fill,
            }, 2, 3, 1, 2);

            grid.Children.Add(new Image
            {
                Source = "9.jpg",
                Aspect = Aspect.Fill,
            }, 3, 4, 1, 2);

            return grid;
        }

        /// <summary>
        /// Create grid object for landscape.
        /// </summary>
        /// <returns>The grid for landscape.</returns>
        private Grid CreateLandscapeGrid()
        {
            // Create a Grid object to show the thumbnail for image.
            Grid grid = new Grid()
            {
                Padding = 20,
                VerticalOptions = LayoutOptions.FillAndExpand,
                RowDefinitions =
                {
                    // One rows.
                    new RowDefinition { Height = new GridLength(PORTRAIT_SIZE, GridUnitType.Absolute) },
                },
                ColumnDefinitions =
                {
                    // Ten columns.
                    new ColumnDefinition { Width = new GridLength(PORTRAIT_SIZE, GridUnitType.Absolute) },
                    new ColumnDefinition { Width = new GridLength(PORTRAIT_SIZE, GridUnitType.Absolute) },
                    new ColumnDefinition { Width = new GridLength(PORTRAIT_SIZE, GridUnitType.Absolute) },
                    new ColumnDefinition { Width = new GridLength(PORTRAIT_SIZE, GridUnitType.Absolute) },
                    new ColumnDefinition { Width = new GridLength(PORTRAIT_SIZE, GridUnitType.Absolute) },
                    new ColumnDefinition { Width = new GridLength(PORTRAIT_SIZE, GridUnitType.Absolute) },
                    new ColumnDefinition { Width = new GridLength(PORTRAIT_SIZE, GridUnitType.Absolute) },
                    new ColumnDefinition { Width = new GridLength(PORTRAIT_SIZE, GridUnitType.Absolute) },
                    new ColumnDefinition { Width = new GridLength(PORTRAIT_SIZE, GridUnitType.Absolute) },
                    new ColumnDefinition { Width = new GridLength(PORTRAIT_SIZE, GridUnitType.Absolute) },
                },
            };

            // Put image into grid view one by one.
            grid.Children.Add(new Image
            {
                Source = "0.jpg",
                Aspect = Aspect.Fill,
            }, 0, 1, 0, 1);

            grid.Children.Add(new Image
            {
                Source = "1.jpg",
                Aspect = Aspect.Fill,
            }, 1, 2, 0, 1);

            grid.Children.Add(new Image
            {
                Source = "2.jpg",
                Aspect = Aspect.Fill,
            }, 2, 3, 0, 1);

            grid.Children.Add(new Image
            {
                Source = "3.jpg",
                Aspect = Aspect.Fill,
            }, 3, 4, 0, 1);

            grid.Children.Add(new Image
            {
                Source = "4.jpg",
                Aspect = Aspect.Fill,
            }, 4, 5, 0, 1);

            grid.Children.Add(new Image
            {
                Source = "5.jpg",
                Aspect = Aspect.Fill,
            }, 5, 6, 0, 1);

            grid.Children.Add(new Image
            {
                Source = "6.jpg",
                Aspect = Aspect.Fill,
            }, 6, 7, 0, 1);

            grid.Children.Add(new Image
            {
                Source = "7.jpg",
                Aspect = Aspect.Fill,
            }, 7, 8, 0, 1);

            grid.Children.Add(new Image
            {
                Source = "8.jpg",
                Aspect = Aspect.Fill,
            }, 8, 9, 0, 1);

            grid.Children.Add(new Image
            {
                Source = "9.jpg",
                Aspect = Aspect.Fill,
            }, 9, 10, 0, 1);

            return grid;
        }

        /// <summary>
        /// Create header label.
        /// </summary>
        /// <returns>Header label.</returns>
        private Label CreateHeaderLabel()
        {
            return new Label()
            {
                Text = "Gallery",
                FontSize = 28,
                TextColor = Color.White,
                FontAttributes = FontAttributes.Bold,
                HorizontalOptions = LayoutOptions.Center,
                VerticalOptions = LayoutOptions.Center,
            };
        }
    }
}
