#include "flashlight.h"
#include <device/led.h>

static int screen_width;
static int screen_height;
static int button_height = 60;
static bool turned_on = false;

static void _button_clicked(void *data, Evas_Object *obj, void *event_info) {
	if (!turned_on) {
		change_state(true, data);
		set_notification(data);
	} else {
		change_state(false, data);
		delete_notification(data);
	}
}

void change_state(bool on, appdata_s *ad) {
	if (on) {
		int max_brightness;
		device_flash_get_max_brightness(&max_brightness);
		device_flash_set_brightness(max_brightness);
		elm_object_text_set(ad->button, "<font_size=30>OFF</font_size>");
		turned_on = true;
	} else {
		device_flash_set_brightness(0);
		elm_object_text_set(ad->button, "<font_size=30>ON</font_size>");
		turned_on = false;
	}
}

void set_notification(appdata_s *ad) {
	ad->notification = notification_create(NOTIFICATION_TYPE_NOTI);
	notification_set_text(ad->notification, NOTIFICATION_TEXT_TYPE_TITLE, "Flashlight",
	NULL, NOTIFICATION_VARIABLE_TYPE_NONE);

	notification_set_text(ad->notification, NOTIFICATION_TEXT_TYPE_CONTENT, "Tap to turn off",
	NULL, NOTIFICATION_VARIABLE_TYPE_NONE);

	notification_set_display_applist(ad->notification, NOTIFICATION_DISPLAY_APP_ALL);

	notification_set_size(ad->notification, 0.5);

	notification_set_layout(ad->notification, NOTIFICATION_LY_ONGOING_EVENT);

	app_control_h app_control = NULL;
	int noti_err = NOTIFICATION_ERROR_NONE;

	app_control_create(&app_control);
	app_control_set_app_id(app_control, "org.tizen.flashlight");

	noti_err = notification_set_launch_option(ad->notification, NOTIFICATION_LAUNCH_OPTION_APP_CONTROL,
			(void *) app_control);
	if (noti_err != NOTIFICATION_ERROR_NONE) {
		notification_free(ad->notification);
		return;
	}

	app_control_destroy(app_control);

	notification_post(ad->notification);
}

void delete_notification(appdata_s *ad) {
	notification_delete(ad->notification);
	notification_free(ad->notification);
}

static void win_delete_request_cb(void *data, Evas_Object *obj, void *event_info) {
	elm_exit();
}

static void win_back_cb(void *data, Evas_Object *obj, void *event_info) {
	appdata_s *ad = data;
	/* Let window go to hide state. */
	elm_win_lower(ad->win);
}

static void create_base_gui(appdata_s *ad) {
	/* Window */
	ad->win = elm_win_util_standard_add(PACKAGE, PACKAGE);
	elm_win_autodel_set(ad->win, EINA_TRUE);

	evas_object_smart_callback_add(ad->win, "delete,request", win_delete_request_cb, NULL);
	eext_object_event_callback_add(ad->win, EEXT_CALLBACK_BACK, win_back_cb, ad);

	/* Get window dimensions */
	elm_win_screen_size_get(ad->win, NULL, NULL, &screen_width, &screen_height);

	/* Create on/off button */
	ad->button = elm_button_add(ad->win);
	evas_object_smart_callback_add(ad->button, "clicked", _button_clicked, ad);
	elm_object_text_set(ad->button, "<font_size=30>ON</font_size>");
	evas_object_resize(ad->button, 0.6 * screen_width, button_height);
	evas_object_size_hint_weight_set(ad->button, EVAS_HINT_EXPAND,
	EVAS_HINT_EXPAND);
	evas_object_move(ad->button, 0.2 * screen_width, (screen_height - button_height) / 2);
	evas_object_show(ad->button);
	elm_object_content_set(ad->win, ad->button);

	/*Disable rotation*/
	elm_win_rotation_set(ad->win, 0);

	/* Show window after base gui is set up */
	evas_object_show(ad->win);
}

static bool app_create(void *data) {
	/* Hook to take necessary actions before main event loop starts
	 Initialize UI resources and application's data
	 If this function returns true, the main loop of application starts
	 If this function returns false, the application is terminated */
	appdata_s *ad = data;

	create_base_gui(ad);

	return true;
}

static void app_language_changed(void *data) {
	/* Set the current language */
	char *locale = NULL;
	runtime_info_get_value_string(RUNTIME_INFO_KEY_LANGUAGE, &locale);
	elm_language_set(locale);
	free(locale);
}

static void app_control(app_control_h app_control, void *data) {
	change_state(false, data);
}

static void app_pause(void *data) {
	/* Take necessary actions when application becomes invisible. */
}

static void app_resume(void *data) {
	/* Take necessary actions when application becomes visible. */
}

static void app_terminate(void *data) {
	appdata_s *ad = data;
	device_flash_set_brightness(0);
	delete_notification(data);
}

int main(int argc, char *argv[]) {
	appdata_s ad = { 0, };
	app_event_callback_s event_callback = { 0, };
	int ret = 0;

	event_callback.create = app_create;
	event_callback.terminate = app_terminate;
	event_callback.pause = app_pause;
	event_callback.resume = app_resume;
	event_callback.app_control = app_control;
	event_callback.language_changed = app_language_changed;

	event_callback.low_memory = NULL;
	event_callback.low_battery = NULL;
	event_callback.device_orientation = NULL;
	event_callback.region_format_changed = NULL;

	ret = app_main(argc, argv, &event_callback, &ad);
	if (ret != APP_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "app_main() is failed. err = %d", ret);
	}

	return ret;
}
