/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

using System;
using System.Collections.Generic;
using Xamarin.Forms;

namespace FileManager
{
    /// <summary>
    /// The page to show the directory list.
    /// </summary>
    public class PhoneDirectoryPage : ContentPage
    {
        private Label _directoryNameLabel;
        private CustomListView _directoryListView;
        private const string PANEL_IMAGE = "file_manager_panel_background.png";
        private string _parentDir;

        /// <summary>
        /// Initializes a new instance of the <see cref="PhoneDirectoryPage"/> class.
        /// </summary>
        /// <param name="parentDir">The parent directory name for current directory.</param>
        public PhoneDirectoryPage(string parentDir) : base()
        {
            Title = "Phone";
            IsVisible = true;
            _parentDir = parentDir;

            // Create phone directory page content.
            Content = CreatePhoneDirectoryPageContent();
            InitPhoneDirectoryPageContent(parentDir);
        }

        /// <summary>
        /// Create the page content for a directory.
        /// </summary>
        /// <returns>The RelativeLayout object.</returns>
        private RelativeLayout CreatePhoneDirectoryPageContent()
        {
            // Create a relative layout for the directory.
            RelativeLayout relativeLayout = new RelativeLayout()
            {
                IsVisible = true
            };

            // Create header label.
            Label headerLabel = CreateHeaderLabel();

            // Add header label into main page layout.
            relativeLayout.Children.Add(
                headerLabel,
                Constraint.Constant(0),
                Constraint.Constant(0),
                Constraint.RelativeToParent((parent) =>
                {
                    return parent.Width;
                }),
                Constraint.Constant(100));

            // Create a layout for directory name.
            RelativeLayout directoryNameLayout = CreateDirectoryNameLayout();
            relativeLayout.Children.Add(
                directoryNameLayout,
                Constraint.Constant(0),
                Constraint.Constant(100),
                Constraint.RelativeToParent((parent) =>
                {
                    return parent.Width;
                }),
                Constraint.Constant(80));

            // Create a layout for exit panel.
            RelativeLayout exitPanelLayout = CreateExitPanelLayout();
            relativeLayout.Children.Add(
                exitPanelLayout,
                Constraint.Constant(0),
                Constraint.RelativeToParent((parent) =>
                {
                    return parent.Bounds.Bottom - 100;
                }),
                Constraint.RelativeToParent((parent) =>
                {
                    return parent.Width;
                }),
                Constraint.Constant(100));

            // Create list view.
            _directoryListView = CreateDirectoryList();
            _directoryListView.ItemTapped += async (s, e) =>
            {
                if (!((Data)e.Item).IsFile)
                {
                    await Navigation.PushModalAsync(new PhoneDirectoryPage(_parentDir + "/" + ((Data)e.Item).Name));
                }
            };

            relativeLayout.Children.Add(
                _directoryListView,
                Constraint.Constant(0),
                Constraint.RelativeToView(directoryNameLayout, (parent, sibling) =>
                {
                    return sibling.Bounds.Y + sibling.Height;
                }),
                Constraint.RelativeToParent((parent) =>
                {
                    return parent.Width;
                }),
                Constraint.RelativeToView(directoryNameLayout, (parent, sibling) =>
                {
                    return parent.Height - 100 - 100 - sibling.Height;
                }));

            return relativeLayout;
        }

        /// <summary>
        /// Create a layout for directory name.
        /// </summary>
        /// <returns>The RelativeLayout object.</returns>
        private RelativeLayout CreateDirectoryNameLayout()
        {
            // Add new RelativeLayout.
            RelativeLayout relativeLayout = new RelativeLayout()
            {
                IsVisible = true
            };

            // Add new Label.
            _directoryNameLabel = new Label()
            {
                VerticalTextAlignment = TextAlignment.Center,
                FontSize = 40,
            };

            // Put name label into layout.
            relativeLayout.Children.Add(
                _directoryNameLabel,
                Constraint.Constant(0),
                Constraint.Constant(0),
                Constraint.RelativeToParent((parent) =>
                {
                    return parent.Width;
                }),
                Constraint.RelativeToParent((parent) =>
                {
                    return parent.Height;
                }));

            return relativeLayout;
        }

        /// <summary>
        /// Create header label.
        /// </summary>
        /// <returns>Header label.</returns>
        private Label CreateHeaderLabel()
        {
            return new Label()
            {
                Text = "Phone",
                FontSize = 50,
                TextColor = Color.White,
                FontAttributes = FontAttributes.Bold,
                HorizontalOptions = LayoutOptions.Center,
                VerticalOptions = LayoutOptions.Center,
            };
        }

        /// <summary>
        /// Create a layout for exit panel.
        /// </summary>
        /// <returns>The RelativeLayout object.</returns>
        private RelativeLayout CreateExitPanelLayout()
        {
            // Add new RelativeLayout.
            RelativeLayout relativeLayout = new RelativeLayout()
            {
                IsVisible = true,
            };

            // Add new Image.
            Image backgroundImg = new Image()
            {
                Source = PANEL_IMAGE,
                Aspect = Aspect.Fill
            };

            // Put background image into layout.
            relativeLayout.Children.Add(
                backgroundImg,
                Constraint.Constant(0),
                Constraint.Constant(0),
                Constraint.RelativeToParent((parent) =>
                {
                    return parent.Width;
                }),
                Constraint.RelativeToParent((parent) =>
                {
                    return parent.Height;
                }));

            // Add exit button.
            Button exitButton = new Button()
            {
                VerticalOptions = LayoutOptions.FillAndExpand,
                HorizontalOptions = LayoutOptions.FillAndExpand,
                Text = "EXIT",
                FontSize = 40,
            };

            exitButton.Clicked += (s, e) =>
            {
                DependencyService.Get<IInterface>().ExitApp();
            };

            // Put label object into layout.
            relativeLayout.Children.Add(
                exitButton,
                Constraint.Constant(0),
                Constraint.Constant(0),
                Constraint.RelativeToParent((parent) =>
                {
                    return parent.Width;
                }),
                Constraint.RelativeToParent((parent) =>
                {
                    return parent.Height;
                }));

            return relativeLayout;
        }

        /// <summary>
        /// Create a ListView for directory.
        /// </summary>
        /// <returns>The CustomListView object.</returns>
        private CustomListView CreateDirectoryList()
        {
            return new CustomListView();
        }

        /// <summary>
        /// Initializes name and list view data for parent directory.
        /// </summary>
        /// <param name="parentDir">The parent directory of current.</param>
        private void InitPhoneDirectoryPageContent(string parentDir)
        {
            // Set directory name.
            string rootPath = DependencyService.Get<IInterface>().GetLocalStorageRootPath();
            _directoryNameLabel.Text = parentDir.Equals(rootPath) ? "Phone/" : parentDir;
            List<Data> directoryList = DependencyService.Get<IInterface>().GetDirectoriesForPath(parentDir);
            List<Data> fileList = DependencyService.Get<IInterface>().GetFilesForPath(parentDir);

            List<Data> list = new List<Data>();

            if (directoryList != null)
            {
                list.AddRange(directoryList);
            }

            if (fileList != null)
            {
                list.AddRange(fileList);
            }

            _directoryListView.AddListItems(list);
        }
    }
}
