/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

using System;
using System.Collections.Generic;
using Xamarin.Forms;

namespace FileManager
{
    /// <summary>
    /// The main page of the FileManager application.
    /// </summary>
    public class MainPage : ContentPage
    {
        private CustomListView _directoryListView;
        private const string PANEL_IMAGE = "file_manager_panel_background.png";
        private const string FOLDER_ICON = "file_manager_folder.png";

        /// <summary>
        /// Initializes a new instance of the <see cref="MainPage"/> class.
        /// </summary>
        public MainPage() : base()
        {
            Title = "Storages";
            IsVisible = true;

            // Create main page.
            Content = CreateMainPageContent();

            // Initializes data.
            InitMainPageContent();
        }

        /// <summary>
        /// Create main page content.
        /// </summary>
        /// <returns>The RelativeLayout object.</returns>
        private RelativeLayout CreateMainPageContent()
        {
            // Create layout for main page.
            RelativeLayout relativeLayout = new RelativeLayout()
            {
                IsVisible = true
            };

            // Create header label.
            Label headerLabel = CreateHeaderLabel();

            // Add header label into main page layout.
            relativeLayout.Children.Add(
                headerLabel,
                Constraint.Constant(0),
                Constraint.Constant(0),
                Constraint.RelativeToParent((parent) =>
                {
                    return parent.Width;
                }),
                Constraint.Constant(100));

            RelativeLayout exitPanelLayout = CreateExitPanelLayout();

            // Add exit panel into main page layout.
            relativeLayout.Children.Add(
                exitPanelLayout,
                Constraint.Constant(0),
                Constraint.RelativeToParent((parent) =>
                {
                    return parent.Bounds.Bottom - 100;
                }),
                Constraint.RelativeToParent((parent) =>
                {
                    return parent.Width;
                }),
                Constraint.Constant(100));

            // Create directory list view.
            _directoryListView = CreateDirectoryList();
            _directoryListView.ItemTapped += async (s, e) =>
            {
                string rootPath = DependencyService.Get<IInterface>().GetLocalStorageRootPath();
                await Navigation.PushModalAsync(new PhoneDirectoryPage(rootPath));
            };

            // Add directory list view into main page layout.
            relativeLayout.Children.Add(
                _directoryListView,
                Constraint.Constant(0),
                Constraint.Constant(100),
                Constraint.RelativeToParent((parent) =>
                {
                    return parent.Width;
                }),
                Constraint.RelativeToView(exitPanelLayout, (parent, sibling) =>
                {
                    return parent.Height - sibling.Height;
                }));

            return relativeLayout;
        }

        /// <summary>
        /// Create header label.
        /// </summary>
        /// <returns>Header label.</returns>
        private Label CreateHeaderLabel()
        {
            return new Label()
            {
                Text = "Storages",
                FontSize = 50,
                TextColor = Color.White,
                FontAttributes = FontAttributes.Bold,
                HorizontalOptions = LayoutOptions.Center,
                VerticalOptions = LayoutOptions.Center,
            };
        }

        /// <summary>
        /// Create a panel layout for exit option.
        /// </summary>
        /// <returns>The RelativeLayout object.</returns>
        private RelativeLayout CreateExitPanelLayout()
        {
            // Create exit panel layout.
            RelativeLayout relativeLayout = new RelativeLayout()
            {
                IsVisible = true,
            };

            // Create background image.
            Image backgroundImg = new Image()
            {
                Source = PANEL_IMAGE,
                Aspect = Aspect.Fill
            };

            // Add background image into panel layout.
            relativeLayout.Children.Add(
                backgroundImg,
                Constraint.Constant(0),
                Constraint.Constant(0),
                Constraint.RelativeToParent((parent) =>
                {
                    return parent.Width;
                }),
                Constraint.RelativeToParent((parent) =>
                {
                    return parent.Height;
                }));

            // Add exit button.
            Button exitButton = new Button()
            {
                VerticalOptions = LayoutOptions.FillAndExpand,
                HorizontalOptions = LayoutOptions.FillAndExpand,
                Text = "EXIT",
                FontSize = 40,
            };

            exitButton.Clicked += (s, e) =>
            {
                DependencyService.Get<IInterface>().ExitApp();
            };

            // Add label into panel layout.
            relativeLayout.Children.Add(
                exitButton,
                Constraint.Constant(0),
                Constraint.Constant(0),
                Constraint.RelativeToParent((parent) =>
                {
                    return parent.Width;
                }),
                Constraint.RelativeToParent((parent) =>
                {
                    return parent.Height;
                }));

            return relativeLayout;
        }

        /// <summary>
        /// Create directory list view.
        /// </summary>
        /// <returns>The CustomListView object.</returns>
        private CustomListView CreateDirectoryList()
        {
            return new CustomListView();
        }

        /// <summary>
        /// Initializes current directory name and list view data.
        /// </summary>
        private void InitMainPageContent()
        {
            Data data = new Data();
            data.Name = "Phone";
            data.Icon = FOLDER_ICON;
            List<Data> directoryList = new List<Data>();
            directoryList.Add(data);
            _directoryListView.AddListItems(directoryList);
        }
    }
}
