﻿/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

using System;
using System.Collections.Generic;
using System.IO;
using FileManager.Tizen;
using Tizen.System;

[assembly: Xamarin.Forms.Dependency(typeof(InterfaceImplementation))]

namespace FileManager.Tizen
{
    /// <summary>
    /// The implementation of the <see cref="IDirectoryInformation"/> interface.
    /// </summary>
    class InterfaceImplementation : IInterface
    {
        private const string FOLDER_ICON = "file_manager_folder.png";
        private const string FILE_ICON = "myfile_icon_unknown.png";

        /// <summary>
        /// Get all of the directory information for specified path.
        /// </summary>
        /// <param name="path">The <see cref="string"> path which is the target path to get directory information.</param>
        /// <returns>Directories to get.</returns>
        public List<Data> GetDirectoriesForPath(string path)
        {
            if (!Directory.Exists(path))
            {
                return null;
            }

            // Create a list of directory name.
            List<Data> dirList = new List<Data>();

            // Create DirectoryInfo class to get directory information.
            System.IO.DirectoryInfo dir = new System.IO.DirectoryInfo(path);
            foreach (System.IO.DirectoryInfo d in dir.GetDirectories())
            {
                Data data = new Data();
                data.Name = d.Name;
                data.Icon = FOLDER_ICON;
                data.IsFile = false;
                dirList.Add(data);
            }

            return dirList;
        }

        /// <summary>
        /// Get all of the files for specified path.
        /// </summary>
        /// <param name="path">The <see cref="string"> path which is the target path to get files.</param>
        /// <returns>Files to get.</returns>
        public List<Data> GetFilesForPath(string path)
        {
            if (!Directory.Exists(path))
            {
                return null;
            }

            // Create a list of file name.
            List<Data> fileList = new List<Data>();

            // Create DirectoryInfo class to get directory information.
            System.IO.DirectoryInfo dir = new System.IO.DirectoryInfo(path);
            foreach (System.IO.FileInfo fileInfo in dir.GetFiles())
            {
                Data data = new Data();
                data.Name = fileInfo.Name;
                data.Icon = FILE_ICON;
                data.IsFile = true;
                fileList.Add(data);
            }

            return fileList;
        }

        /// <summary>
        /// Exit current application.
        /// </summary>
        public void ExitApp()
        {
            Program.instance.Exit();
        }

        /// <summary>
        /// Get the local storage root path.
        /// </summary>
        /// <returns>Local storage root path.</returns>
        public string GetLocalStorageRootPath()
        {
            var storageList = StorageManager.Storages as List<Storage>;
            string rootPath = "";
            foreach (var storage in storageList)
            {
                rootPath = storage.RootDirectory;
            }

            return rootPath;
        }
    }
}
