/*******************************************************************************
 * Module responsible for all application's logic
 *
 * @author Zaneta Szymanska <<ahref='mailto:z.szymanska@samsung.com'>z.szymanska@samsung.com</a>>
 *
 * **************************************************************************************
 *
 * Copyright (c) 2012 Samsung Electronics All Rights Reserved.
 *
 ******************************************************************************/
"use strict";
var addressBookMgr = function() {
    /**
     * Address Book used by sample application
     */
    var gAddressbook,
    /**
     * Filter selected by user: 0 (no Filter) .. 7(all filters)
     */
    filter = 0,
    /**
     * Displayed page: 0 (means main page) 1 (messsage & add receivers page).
     * Needed to recognize which listview should be refreshed
     */
    page = 0,
    /**
     * Array of message receivers (SMS)
     */
    receivers = [],
    /**
     * Temporary array of message receivers, needed before adding contacts to
     * receivers[] array
     */
    tmpReceivers = [];

    var log = function(msg) {
        tlib.logger.info(msg);
    };
    /**
     * Function called when an error occured
     *
     * @param {Object} err Generic error interface
     */
    var onError = function(err) {
        alert("Error(" + err.code + "): " + err.message);
    };
    /**
     * Closes application if possible
     */
    var exit = function() {
        if (typeof tizen !== "undefined" && tizen.application) {
            if (confirm("Exit?")) {
                var app = tizen.application.getCurrentApplication();
                app.exit();
                log("Application closed");
            }
        } else {
            alert("Not supported");
        }
    };
    /**
     * Gets phone number for selected receiver
     *
     * @param {Number} p Index of receiver whose phone number we want to add
     * @returns {String} phoneNr Phone number of selected receiver or empty
     *          string "", if the contact doesn't have phone number
     */
    var getPhoneNumber = function(p) {
        var phoneNr = "";

        var contact = gAddressbook.get(receivers[p]);
        /**
         * Actually contact has only one phone number In the future following
         * code should be improved
         */
        if (typeof (contact.phoneNumbers[0]) !== 'undefined') {
            if (contact.phoneNumbers[0]['number'][0] !== '+') {
                phoneNr += "+48";
            }

            phoneNr += (contact.phoneNumbers[0]['number']);
        }

        return phoneNr;
    };
    /**
     * Set message page after selecting message receiver (or receivers) by user
     */
    var setMsgPage = function() {
        var i, number = "";

        if (!page) {
            page = 1;
        } else {
            receivers.length = 0;
            for (i = 0; i < tmpReceivers.length; i++) {
                receivers[i] = tmpReceivers[i];
            }
        }

        for (i = 0; i < receivers.length; i++) {
            if (getPhoneNumber(i)) {
                number += getPhoneNumber(i);
                number += ", ";
            } else {
                // If the contact doesn't have proper phone number
                // contact.phoneNumbers[0] is undefined
                tlib.view.showPopup("phoneNumbers[0] for receiver (id "
                        + receivers[i] + ") is undefined!");
            }
        }

        $("#formPhone").attr("value", number);
        $("#formBody").attr("value", "");
        $("#logDiv").empty();
    };
    /**
     * Provide launching application "phone" with selected phone number
     */
    var callTo = function() {
        var number = "tel:";

        if (getPhoneNumber(0)) {
            number += getPhoneNumber(0);

            var service = new tizen.ApplicationControl(
                    'http://tizen.org/appcontrol/operation/call', number, null);

            try {
                tizen.application.launchAppControl(service, null, function() {
                    log("launchAppControl");
                }, onError, null);
            } catch (exc) {
                log("launchService exc: " + exc.message);
            }
        } else {
            // If the contact doesn't have proper phone number
            // contact.phoneNumbers[0] is undefined
            tlib.view.showPopup("contact.phoneNumbers[0] is undefined!");
        }
    };
    /**
     * Check if contact is one of the message's receiver
     *
     * @param {Number} index Index on the list of contacts
     */
    var isChecked = function(index) {
        for ( var i = 0; i < tmpReceivers.length; i++) {
            if (index == receivers[i]) {
                return true;
            }
        }
        return false;
    };

    /**
     * Fill the list of contacts in application
     *
     * @param {ArrayList} contacts Successfully retrieved list of contacts
     */
    var onContactFindSuccess = function(contacts) {
        var value, str = '', lastletter = '';

        // Add the list divider
        for ( var i = 0; i < contacts.length; i++) {

            if (contacts[i].name.firstName) {
                if (lastletter !== contacts[i].name.firstName[0]) {
                    lastletter = contacts[i].name.firstName[0];
                    str += '<li data-role="list-divider" data-item-type="contact">'
                            + lastletter + '</li>';
                }
            } else {
                if (lastletter !== '#') {
                    lastletter = '#';
                    str += '<li data-role="list-divider" data-item-type="contact"> # </li>';
                }
            }

            // Add contact
            if (page === 0) {
                str += '<li class="ui-li-has-multiline" data-item-type="contact" data-id="'
                        + contacts[i].id
                        + '">'
                        + ((contacts[i].name === null) ? ""
                                : (value = contacts[i].name.firstName) ? value
                                        : "")
                        + ' '
                        + ((contacts[i].name === null) ? ""
                                : (value = contacts[i].name.lastName) ? value
                                        : "")
                        + '    '
                        + '<span class="ui-li-text-sub">'
                        + ((value = contacts[i].phoneNumbers[0]) ? value.number
                                : "") + '</span></li>';
            } else {
                str += '<li class="ui-li-has-multiline" data-item-type="contact" data-id="'
                        + contacts[i].id
                        + '"><input type="checkbox" value="'
                        + contacts[i].id
                        + '" name="'
                        + contacts[i].id
                        + '" data-id="'
                        + contacts[i].id
                        + '" id="'
                        + contacts[i].id + '" class="custom"';

                if (isChecked(contacts[i].id)) {
                    str += ' checked="checked"';
                }

                str += '>'
                        + '<label for="'
                        + contacts[i].id
                        + '" class="checklistLabel">'
                        + ((contacts[i].name === null) ? ""
                                : (value = contacts[i].name.firstName) ? value
                                        : "")
                        + ' '
                        + ((contacts[i].name === null) ? ""
                                : (value = contacts[i].name.lastName) ? value
                                        : "")
                        + ' '
                        + ((value = contacts[i].phoneNumbers[0]) ? value.number
                                : "") + '</label></li>';
            }
        }

        if (page === 0) {
            $("#list>li[data-item-type=contact]").remove();

            var elem = $(str).unbind('swipeleft').bind('swipeleft',
                    function(e) {
                        /**
                         * On "swipeleft" event send message to selected person
                         */
                        receivers.length = [];
                        receivers[0] = $(this).data("id");

                        setMsgPage();
                        $.mobile.changePage("#msgpage");
                    });

            elem.unbind('swiperight').bind('swiperight', function(e) {
                /**
                 * On "swiperight" event call to selected person
                 */
                receivers.length = [];
                receivers[0] = $(this).data("id");

                callTo();
            });

            $("#list").append(elem).trigger("create").listview("refresh");
        } else {
            $("#listAddRec>li[data-item-type=contact]").remove();
            $("#listAddRec").append(str).trigger("create").listview("refresh");
        }
    };
    /**
     * Read all contacts from the contact book
     */
    var fetchAllContacts = function() {
        var sortingMode = new tizen.SortMode('name.firstName', 'ASC');
        var phoneFilter = new tizen.AttributeFilter('phoneNumbers.number',
                'CONTAINS', '');
        gAddressbook.find(onContactFindSuccess, onError, phoneFilter,
                sortingMode);
    };

    var setFilterMode = function(id, attrName) {
        var filterMode = null;
        var tmp = null;

        var firstNameFilter = new tizen.AttributeFilter('name.firstName',
                'CONTAINS', attrName);
        var lastNameFilter = new tizen.AttributeFilter('name.lastName',
                'CONTAINS', attrName);
        var phoneFilter = new tizen.AttributeFilter('phoneNumbers.number',
                'CONTAINS', attrName);
        var includePhone = new tizen.AttributeFilter('phoneNumbers.number',
                'CONTAINS', '');

        switch (id) {
        case 0:
            filterMode = includePhone;
            break;

        case 1:
            filterMode = new tizen.CompositeFilter("INTERSECTION", [
                    firstNameFilter, includePhone ]);
            break;

        case 2:
            filterMode = new tizen.CompositeFilter("INTERSECTION", [
                    lastNameFilter, includePhone ]);
            break;

        case 3: {
            tmp = new tizen.CompositeFilter("UNION", [ firstNameFilter,
                    lastNameFilter ]);
            filterMode = new tizen.CompositeFilter("INTERSECTION", [ tmp,
                    includePhone ]);
        }
            break;

        case 4:
            filterMode = phoneFilter;
            break;

        case 5: {
            tmp = new tizen.CompositeFilter("UNION", [ firstNameFilter,
                    phoneFilter ]);
            filterMode = new tizen.CompositeFilter("INTERSECTION", [ tmp,
                    includePhone ]);
        }
            break;

        case 6: {
            tmp = new tizen.CompositeFilter("UNION", [ lastNameFilter,
                    phoneFilter ]);
            filterMode = new tizen.CompositeFilter("INTERSECTION", [ tmp,
                    includePhone ]);
        }
            break;

        case 7: {
            tmp = new tizen.CompositeFilter("UNION", [ firstNameFilter,
                    lastNameFilter, phoneFilter ]);
            filterMode = new tizen.CompositeFilter("INTERSECTION", [ tmp,
                    includePhone ]);
        }
            break;

        default:
            break;
        }

        return filterMode;
    };

    /**
     * Read filtered (by various user given queries) contacts from the contact
     * book It is important that all contacts must have phone number
     */
    var fetchContacts = function() {
        if (gAddressbook) {
            var filterMode = null;

            if (page === 0) {
                filterMode = setFilterMode(filter, $('#search').val());
            } else {
                filterMode = setFilterMode(1, $("#search-receiver").val());
            }

            var sortingMode = new tizen.SortMode('name.firstName', 'ASC');

            if (filterMode !== null) {
                gAddressbook.find(onContactFindSuccess, onError, filterMode,
                        sortingMode);
            }
        }
    };
    /**
     * Set the response to change filters or send query
     */
    var listResponse = function() {
        if (($("#search").val().trim() === '')) {
            fetchAllContacts();
        } else {
            if (filter === 0) {
                fetchAllContacts();
            } else {
                fetchContacts();
            }
        }
    };
    /**
     * Set global variable page 0 for main 1 for message
     */
    var changePage = function() {
        if (page === 0) {
            page++;
        } else {
            page--;
            tmpReceivers.length = [];
        }
    };
    /**
     * Reset receiver's list at addRec page
     */
    var resetReceiversList = function() {
        $("#search-receiver").val('');
        fetchContacts();
        tmpReceivers.length = [];

        for ( var i = 0; i < receivers.length; i++) {
            tmpReceivers.push(receivers[i]);
        }
    };
    /**
     * Config addRec page before showing
     */
    var addRecPage = function() {
        resetReceiversList();
        $.mobile.changePage("#addRec");

        $("#search-receiver").on("keyup", function(event) {
            fetchContacts();
        });
    };

    /**
     * Check if number of message's receivers is less than 11
     */
    var checkReceiversCount = function() {
        if (tmpReceivers.length > 10) {
            tlib.view.showPopup("10 contacts available only.");
        } else {
            setMsgPage();
            $.mobile.changePage("#msgpage");
        }
    };
    /**
     * Log the information about progress of sending message in logDiv element
     *
     * @param {String} message Information which we want to display
     */
    var logMsg = function(message) {
        var line = $('<p></p>');
        line.html(message);
        $("#logDiv").append(line);
    };
    /**
     * Success Callback for function sendMessage()
     */
    var messageSent = function() {
        logMsg("The SMS has been sent");
    };
    /**
     * Error Callback for function sendMessage()
     */
    var messageFailed = function(error) {
        logMsg("The SMS could not be sent: " + error.message);
    };
    /**
     * Success Callback for function getMessageServices(), invoked if the
     * services are successfully retrieved
     *
     * @param services Interface, which allows a web application to manipulate
     *        and send messages
     */
    var serviceList = function(services) {
        $("#logDiv").empty();
        logMsg("Service initialized");

        var body = $("#formBody").val(), msg, phone;

        if (body.trim()) {
            for ( var i = 0; i < receivers.length; i++) {
                if (getPhoneNumber(i)) {
                    phone = getPhoneNumber(i);
                    logMsg("Message \"" + body + "\" to " + phone
                            + " will be send.");
                    try {
                        msg = new tizen.Message("messaging.sms", {
                            plainBody : body,
                            to : [ phone ]
                        });
                    } catch (err) {
                        logMsg("Creating message failed: " + err.code + ":"
                                + err.message);
                        return;
                    }
                    logMsg("Body set");
                    try {
                        services[0]
                                .sendMessage(msg, messageSent, messageFailed);
                    } catch (err2) {
                        logMsg("Sending message failed: " + err2.code + ":"
                                + err2.message);
                        return;
                    }
                    logMsg("Message send");
                } else {
                    // If the contact doesn't have proper phone number
                    // contact.phoneNumbers[0] is undefined
                    tlib.view
                            .showPopup("contact.phoneNumbers[0] is undefined!");
                }
            }
        } else {
            tlib.view.showPopup("Message cannot be empty.");
            $("#formBody").val($("#formBody").val().trim());
        }
    };
    /**
     * Function invoked after click on Send button, call function
     * getMessageServices()
     */
    var sendSms = function() {
        try {
            tizen.messaging.getMessageServices("messaging.sms", serviceList,
                    onError);
        } catch (e) {
            logMsg(e.message);
        }
    };

    var addReceiver = function(that) {
        var i, checkedNr = that.data("id");
        /**
         * Adding receivers of message
         */
        if (tmpReceivers.length > 10) {
            tlib.view
                    .showPopup("10 contacts available only. Uncheck some contacts.");
        }

        if (that.prop("checked")) {
            var wsk = 0;
            for (i = 0; i < tmpReceivers.length; i++) {
                if (tmpReceivers[i] === checkedNr) {
                    wsk = 1;
                }
            }
            (!wsk) ? tmpReceivers.push(checkedNr) : wsk = 0;
        } else {
            var idx = -1;
            for (i = 0; i < tmpReceivers.length; i++) {
                if (tmpReceivers[i] === checkedNr) {
                    idx = i;
                }
            }
            tmpReceivers.splice(idx, 1);
        }
    };

    /**
     * Change selected filter
     */
    var setFilter = function(that) {
        switch (that.data("id")) {
        case "firstN":
            if (that.prop("checked")) {
                filter += 1;
            } else {
                filter -= 1;
            }
            break;

        case "lastN":
            if (that.prop("checked")) {
                filter += 2;
            } else {
                filter -= 2;
            }
            break;

        case "phone":
            if (that.prop("checked")) {
                filter += 4;
            } else {
                filter -= 4;
            }
            break;

        default:
            break;
        }

        listResponse();
        if (filter === 0) {
            tlib.view.showPopup("Select the filter you want to use.");
        }
    };

    return {
        initialize : function() {
            tlib.visibility.onApplicationVisible(function() {
                (page === 0) ? listResponse() : fetchContacts();
            });

            $("#filter").delegate("input", "change", function() {
                var that = $(this);
                setFilter(that);
            });

            $("#search").on("keyup", function(event) {
                listResponse();
                if (filter === 0) {
                    tlib.view.showPopup("Select the filter you want to use.");
                }
            });

            $("#close").on("click", function(event) {
                exit();
            });

            gAddressbook = tizen.contact.getDefaultAddressBook();
            fetchAllContacts();

            $("#home").on("click", function(event) {
                changePage();
            });
            /**
             * Add more receivers' page functionality
             */
            $("#add").on("click", function(event) {
                addRecPage();
            });

            $("#add-to-home").unbind().click(function(event) {
                changePage();
            });

            $("#add-change").unbind().click(function(event) {
                checkReceiversCount();
            });

            $("#listAddRec").delegate("input", "change", function() {
                var that = $(this);
                addReceiver(that);
            });

            /**
             * Send message's page functionalities
             */
            $("#addRec").bind("pagebeforeshow", fetchContacts());

            $("#close-send-msg").unbind().click(function(event) {
                exit();
            });

            $("#send").unbind().click(function(event) {
                sendSms();
            });
        }
    };
}();