"use strict";
/**
 * Level constructor
 * 
 * @param {Object} options
 * @param options.duration {Number} level duration (sec)
 * @param options.period {Number} enemy appearing period (msec)
 * @param options.types {Object} available enemies types for current level
 * @param options.nextLevel {Object} next level pointer
 * @param options.name {String} level name
 */
var Level = function(options) {
	if (typeof options === "undefined") {
		game.log("ERROR: please pass options for new Level");
		return false;
	}

	var startTime;
	var intervalPtr;
	var powerUpPtr;
	var powerUpPeriod;
	var running = false;
	this.options = options;
	this.num;
	var randomPowerUp = function() {
		var rnd = Math.random();
		if (rnd < 0.2)
			return 'powerUp_double';
		else if (rnd >= 0.2 && rnd < 0.4) {
			return 'powerUp_heal';
		} else if (rnd >= 0.4 && rnd < 0.6) {
			return 'powerUp_repair';
		} else if (rnd >= 0.6 && rnd < 0.8) {
			return 'powerUp_shield';
		} else if (rnd >= 0.8) {
			return 'powerUp_bomb';
		}
	}

	var powerUpCallback = function() {
		if (!game.isPaused()) {
			var x = Math.random() * view.getWidth() - config.player.topOffset;
			var y = Math.random() * (view.getHeight() / 2);
			if (x > config.player.topOffset && y > config.player.topOffset)
				game.getBoard().add(new PowerUp(x, y, randomPowerUp(), 8), 1);
		}
		powerUpPtr = setTimeout(powerUpCallback, powerUpPeriod * 1000);
	}

	var addPowerUps = function() {
		if (!options.powerUps) {
			game.log("No powerups defined for this level");
			return false;
		}
		powerUpPeriod = options.duration / options.powerUps.number;
		powerUpCallback();
	};

	this.start = function() {
		game.log("Start Level " + this.num);
		var that = this;
		game.getHud().setLevel(this);
		running = true;
		messageBoxManager.show(new MessageBox("Level " + this.num + ": " + options.name));
		var that = this;
		startTime = new Date();
		intervalPtr = setInterval(function() {
			// for application in paused state don't add new enemies
			if (game.isPaused())
				return;
			for ( var k = 0; k < options.types.length; k++) {
				game.getBoard().add(new Enemy(options.types[k].basicType, options.types[k].overrides), config.enemy.zIndex);
			}
			var now = new Date();
			game.log("Level current duration -> " + (now.getTime() - startTime.getTime()));
			game.log("Level full duration -> " + (options.duration * 1000));
			if (now.getTime() - startTime.getTime() > options.duration * 1000) {
				game.log("Level " + that.num + " finished");
				that.stop();
			}

		}, options.period);
		addPowerUps();
	};

	this.stop = function(force) {
		game.getBoard().removeByTypes([ game.getObjectTypes().ENEMY, game.getObjectTypes().ENEMY_MISSILE, game.getObjectTypes().PLAYER_MISSILE, game.getObjectTypes().POWER_UP ]);
		clearInterval(intervalPtr);
		clearInterval(powerUpPtr);
		running = false;
		if (force)
			return false;
		if (this.num === 1)
			levelManager.storePassedLevel(this);
		if (typeof options.nextLevel === "object")
			levelManager.storePassedLevel(options.nextLevel);
		if (typeof options.nextLevel === "object")
			// start new level
			options.nextLevel.start();
		else {
			// game finished - no more levels
			messageBoxManager.show(new MessageBox("Congratulations - the game is finished!"));
			game.setOnGoing(false);
			setTimeout(function() {
				menu.open('gameover');
			}, 5000);
		}
	};

	/**
	 * Checks if level is active
	 * 
	 * @returns {Boolean} true if level is active
	 */
	this.isRunning = function() {
		return running;
	};

	this.updateStartTime = function(delta) {
		game.log("startTime before" + startTime);
		game.log("Level startTime updated by " + delta)
		var newTime = startTime.getTime() + delta;
		startTime = new Date(newTime);
		game.log("startTime after" + startTime);
	}

};