/*
 * Copyright (c) 2014 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "main-app.h"
#include <dlog.h>
#include <stdlib.h>
#include <Elementary.h>
#include <app.h>
#include <math.h>
#include <efl_extension.h>


// app event callbacks
static bool _on_create_cb(void *user_data);
static void _on_terminate_cb(void *user_data);
static void _on_pause_cb(void *user_data);
static void _on_resume_cb(void *user_data);
static void _on_app_control_cb(app_control_h app_control, void *user_data);
static void _on_low_memory_cb(void *user_data);
static void _on_low_battery_cb(void *user_data);
static void _on_dev_orientation_changed_cb(app_device_orientation_e orientation, void *user_data);
static void _on_lang_changed_cb(void *user_data);
static void _on_region_fmt_changed_cb(void *user_data);

typedef struct cymbal drum;
struct cymbal {
	int r;
	int y;
	int x;
};

app_data *app_create()
{
    app_data *app = calloc(1, sizeof(app_data));

    return app;
}

void app_destroy(app_data *app)
{
    free(app);
}

int app_run(app_data *app, int argc, char **argv)
{
    if(!app)
    {
        return -1;
    }

    app_event_callback_s cbs =
    {
        .create = _on_create_cb,
        .terminate = _on_terminate_cb,
        .pause = _on_pause_cb,
        .resume = _on_resume_cb,
        .app_control = _on_app_control_cb,
        .low_memory = _on_low_memory_cb,
        .low_battery = _on_low_battery_cb,
        .device_orientation = _on_dev_orientation_changed_cb,
        .language_changed = _on_lang_changed_cb,
        .region_format_changed = _on_region_fmt_changed_cb
    };

    return app_main(argc, argv, &cbs, app);
}

static float
get_distance(int x1, int y1, int x2, int y2) {
	int rx = (x2 - x1);
	int ry = (y2 - y1);
	rx *= rx;
	ry *= ry;
	int dist = (rx + ry);

	return sqrt(dist);
}

void play_audio(player_h player)
{
	player_state_e player_state;
    player_get_state(player, &player_state);
    if (player_state == PLAYER_STATE_PLAYING)
    {
        player_stop(player);
    }

	player_start(player);
}


static void
mouse_down_cb(void *data, Evas *e EINA_UNUSED, Evas_Object *obj EINA_UNUSED, void *event_info)
{
	app_data *ad = data;
	Evas_Event_Mouse_Out *ev = event_info;

   char buf[1024];

	drum __pRide = { 216, 1187, 556 };
	drum __pCrash = { 216, 887, 648 };
	drum __pTom = { 167, 260, 405 };
	drum __pTom2 = { 170, 605, 478 };
	drum __pHihat2 = { 157, 242, 680 };
	drum __pHihat = { 157, 48, 550 };
	drum __pFloor = { 190, 888, 223 };
	drum __pSnare = { 200, 512, 156 };
	drum __pBass = { 230, 178, 157};

	int coordX = ev->output.x;
	int coordY = ev->output.y;


	if (get_distance(coordX, coordY, __pRide.x, __pRide.y) < __pRide.r) {
		play_audio(ad->player1);
	} else {
		if (get_distance(coordX, coordY, __pCrash.x, __pCrash.y) < __pCrash.r) {
			play_audio(ad->player2);
		} else {
			if (get_distance(coordX, coordY, __pTom2.x, __pTom2.y) < __pTom2.r) {
				play_audio(ad->player3);
			}
		}
	}

	if (get_distance(coordX, coordY, __pHihat2.x, __pHihat2.y) < __pHihat2.r) {
		play_audio(ad->player4);
	} else {
		if (get_distance(coordX, coordY, __pHihat.x, __pHihat.y) < __pHihat.r) {
			play_audio(ad->player5);
		} else {
			if (get_distance(coordX, coordY, __pTom.x, __pTom.y) < __pTom.r) {
				play_audio(ad->player6);
			}
		}
	}

	if (get_distance(coordX, coordY, __pFloor.x, __pFloor.y) < __pFloor.r) {
		play_audio(ad->player7);
	} else {
		if (get_distance(coordX, coordY, __pSnare.x, __pSnare.y) < __pSnare.r) {
			play_audio(ad->player8);
		} else {
			if (get_distance(coordX, coordY, __pBass.x, __pBass.y) < __pBass.r) {
				play_audio(ad->player9);
			}
		}
	}
}



static void init_base_player(app_data *ad)
{
	player_create(&ad->player1);
	player_set_uri(ad->player1, "/opt/usr/apps/org.tizen.drums/res/sounds/ride.wav");
	player_prepare(ad->player1);

	player_create(&ad->player2);
	player_set_uri(ad->player2, "/opt/usr/apps/org.tizen.drums/res/sounds/crash.wav");
	player_prepare(ad->player2);

	player_create(&ad->player3);
	player_set_uri(ad->player3, "/opt/usr/apps/org.tizen.drums/res/sounds/tom2.wav");
	player_prepare(ad->player3);

	player_create(&ad->player4);
	player_set_uri(ad->player4, "/opt/usr/apps/org.tizen.drums/res/sounds/hihat2.wav");
	player_prepare(ad->player4);

	player_create(&ad->player5);
	player_set_uri(ad->player5, "/opt/usr/apps/org.tizen.drums/res/sounds/hihat1.wav");
	player_prepare(ad->player5);

	player_create(&ad->player6);
	player_set_uri(ad->player6, "/opt/usr/apps/org.tizen.drums/res/sounds/tom1.wav");
	player_prepare(ad->player6);

	player_create(&ad->player7);
	player_set_uri(ad->player7, "/opt/usr/apps/org.tizen.drums/res/sounds/floor.wav");
	player_prepare(ad->player7);

	player_create(&ad->player8);
	player_set_uri(ad->player8, "/opt/usr/apps/org.tizen.drums/res/sounds/snare.wav");
	player_prepare(ad->player8);

	player_create(&ad->player9);
	player_set_uri(ad->player9, "/opt/usr/apps/org.tizen.drums/res/sounds/bass.wav");
	player_prepare(ad->player9);

}

static void
win_back_cb(void *data, Evas_Object *obj, void *event_info)
{
	app_data *ad = data;
	/* Let window go to hide state. */
	elm_win_lower(ad->win);
}

static void
create_base_gui(app_data *ad)
{
	char buf[1024];

    ad->win = elm_win_add(NULL, APP_NAME, ELM_WIN_BASIC);

	eext_object_event_callback_add(ad->win, EEXT_CALLBACK_BACK, win_back_cb, ad);

    ad->bg = elm_bg_add(ad->win);
    evas_object_size_hint_weight_set(ad->bg, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
    snprintf(buf, sizeof(buf), "%s/background.png", ICON_DIR);
    elm_bg_file_set(ad->bg, buf, NULL);
	elm_win_resize_object_add(ad->win, ad->bg);
    evas_object_show(ad->bg);

	evas_object_event_callback_add(ad->bg, EVAS_CALLBACK_MOUSE_DOWN, mouse_down_cb, ad);

	init_base_player(ad);

	evas_object_show(ad->win);
}
static bool _on_create_cb(void *user_data)
{
    app_data *ad = user_data;

    if(!ad)
    {
        return false;
    }
	create_base_gui(ad);

    return true;
}

static void _on_terminate_cb(void *user_data)
{
    app_data *app = user_data;

    if(!app)
    {
        return;
    }
    player_unprepare(app->player1);
    player_unprepare(app->player2);
    player_unprepare(app->player3);
    player_unprepare(app->player4);
    player_unprepare(app->player5);
    player_unprepare(app->player6);
    player_unprepare(app->player7);
    player_unprepare(app->player8);
    player_unprepare(app->player9);

    player_destroy(app->player1);
    player_destroy(app->player2);
    player_destroy(app->player3);
    player_destroy(app->player4);
    player_destroy(app->player5);
    player_destroy(app->player6);
    player_destroy(app->player7);
    player_destroy(app->player8);
    player_destroy(app->player9);

    app->player1 = NULL;
    app->player2 = NULL;
    app->player3 = NULL;
    app->player4 = NULL;
    app->player5 = NULL;
    app->player6 = NULL;
    app->player7 = NULL;
    app->player8 = NULL;
    app->player9 = NULL;

    app->win = NULL;
}

static void _on_pause_cb(void *user_data)
{
}

static void _on_resume_cb(void *user_data)
{
}

static void _on_app_control_cb(app_control_h app_control, void *user_data)
{
}

static void _on_low_memory_cb(void *user_data)
{
}

static void _on_low_battery_cb(void *user_data)
{
}

static void _on_dev_orientation_changed_cb(app_device_orientation_e orientation, void *user_data)
{
}

static void _on_lang_changed_cb(void *user_data)
{
}

static void _on_region_fmt_changed_cb(void *user_data)
{
}
