#include "main-view.h"
#include "customorganizer.h"
#include "edit-view.h"
#include "calendar-utils.h"
#include "logger.h"

#include <utils_i18n.h>

#define ms2sec(ms) 			(long long int)(ms)/1000.0
#define SECONDS_IN_DAY		(24*60*60)

static const char *label_select        	= "Select day to see tasks";
static const char *label_completed       	= "Set As Completed";
static const char *label_edit		       	= "Edit";
static const char *label_completed_warn	= "Choose the task you want to set as completed";
static const char *label_edit_warn		= "Choose the task you want to edit";

typedef struct item_data
{
	 int index;
	 char *name;
	 Elm_Object_Item *item;
} item_data_s;

typedef struct button_data
{
	int index;
	int year, month, day;
} button_data_s;

static void
win_delete_request_cb(void *data , Evas_Object *obj , void *event_info)
{
	ui_app_exit();
}

static void
_gl_del_cb(void *data, Evas_Object *obj)
{
	/* FIXME: Unrealized callback can be called after this. */
	/* Accessing item_data_s can be dangerous on unrealized callback. */
	item_data_s *id = data;
	free(id);
}

static void
main_view_back_cb(void *data, Evas_Object *obj, void *event_info)
{
	main_view_data *view = data;

    if(view->navi_item == elm_naviframe_top_item_get(view->nf))
    {
    	/* Let window go to hide state. */
        elm_win_lower(view->win);
    }
    else
    {
        elm_naviframe_item_pop(view->nf);
    }
}

static char *
_item_label_get(void *data, Evas_Object *obj, const char *part) {
	item_data_s *id = data;
	char buf[1024];

	// Check this is text for the part we're expecting
	if (strcmp(part, "elm.text") == 0) {
		snprintf(buf, 1023, "%s", id->name);
		return strdup(buf);
	} else {
		return NULL;
	}
}

// get time in sec from input date and time [sec]
long long int _time_convert_itol(char *tzid, int y, int mon, int d)
{
   long long int lli;
   i18n_ucalendar_h ucal;
   i18n_udate date;
   int ret = I18N_ERROR_NONE;
   int len;

   i18n_uchar *_tzid = NULL;

   if (tzid == NULL) {
       tzid = "Etc/GMT";
   }
   _tzid = (i18n_uchar*)calloc(strlen(tzid) + 1, sizeof(i18n_uchar));
   if (_tzid == NULL) {
       return -1;
   }
   // converts 'tzid' to unicode string
   i18n_ustring_copy_ua(_tzid, tzid);

   // gets length of '_tzid'
   len = i18n_ustring_get_length(_tzid);
   // creates i18n_ucalendar_h
   ret = i18n_ucalendar_create(_tzid, -1, I18N_ULOCALE_UK, I18N_UCALENDAR_DEFAULT, &ucal);
   if (ret) {
       dlog_print(DLOG_ERROR, LOG_TAG, "i18n_ucalendar_create failed.\n");
       return -1;
   }

   // sets i18n_ucalendar_h's date
   i18n_ucalendar_set(ucal, I18N_UCALENDAR_YEAR, y);
   i18n_ucalendar_set(ucal, I18N_UCALENDAR_MONTH, mon);
   i18n_ucalendar_set(ucal, I18N_UCALENDAR_DATE, d);
   i18n_ucalendar_set(ucal, I18N_UCALENDAR_HOUR_OF_DAY, 0);
   i18n_ucalendar_set(ucal, I18N_UCALENDAR_MINUTE, 0);
   i18n_ucalendar_set(ucal, I18N_UCALENDAR_SECOND, 0);

   // gets i18n_ucalendar's current time and converts it from milliseconds to seconds
   i18n_ucalendar_get_milliseconds(ucal, &date);
   lli = ms2sec(date);

   // destroys i18n_ucalendar_h
   i18n_ucalendar_destroy(ucal);
   if (_tzid) {
       free(_tzid);
   }

   return lli;
}

static void _item_selected_cb(void *user_data, Evas_Object *obj, void *event_info) {
	main_view_data *view = user_data;

	 Elm_Object_Item *it = event_info;
	 item_data_s *id = elm_object_item_data_get(it);

	view->selected_item_db_index = id->index;
}


static void
get_tasks_and_events(main_view_data *view, long long int limit) {
	calendar_error_e error_code = CALENDAR_ERROR_NONE;
	long long int tasktime;
	calendar_list_h list = NULL;
	calendar_record_h recordOut;
	Elm_Genlist_Item_Class *itc;
	int index = 0;
	int status = 0;
	int record_id = 0;

	limit += SECONDS_IN_DAY;

	elm_genlist_clear(view->genlist);

	itc = elm_genlist_item_class_new();

	itc->item_style = "default";
	itc->func.text_get = _item_label_get;
	itc->func.content_get = NULL; //_item_content_get;
	itc->func.state_get = NULL; //_item_state_get;
	itc->func.del = _gl_del_cb; //_item_del;

	// Get all todos
	error_code = calendar_db_get_all_records(_calendar_todo._uri, 0, 0, &list);

	if (error_code != CALENDAR_ERROR_NONE)
	   dlog_print(DLOG_ERROR, LOG_TAG, "db get all records failed: %d\n", error_code);

	while (calendar_list_get_current_record_p(list, &recordOut) == CALENDAR_ERROR_NONE)
	{
		status = 0;
		// Get record status
		calendar_record_get_int(recordOut, _calendar_todo.todo_status, &status);

		if (status != CALENDAR_TODO_STATUS_COMPLETED && status != CALENDAR_TODO_STATUS_CANCELED) {
			calendar_time_s start_time = {0};
			calendar_record_get_caltime(recordOut, _calendar_todo.start_time, &start_time);
			tasktime = start_time.time.utime;

			if (tasktime < limit) {
				char* summary;
				calendar_record_get_str_p(recordOut, _calendar_todo.summary, &summary);
				calendar_record_get_int(recordOut, _calendar_todo.id, &record_id);

				// Create item on genlist
				item_data_s *id = calloc(sizeof(item_data_s), 1);
				id->index = record_id;
				id->name = summary;
				elm_genlist_item_append(view->genlist, itc, id,           // item data
						NULL,                 // parent item for trees, NULL if none
						ELM_GENLIST_ITEM_NONE,  // item type; this is the common one
						_item_selected_cb,       // callback on selection of the item
						view                     // data for that callback function
						);
			}
		}
		index++;

	   error_code = calendar_list_next(list);
	   if (error_code != CALENDAR_ERROR_NONE)
	      break;
	}
	elm_genlist_item_class_free(itc);
}

static void refresh_task_list(void *user_data, long long int selected_date) {
	calendar_time_s st = { 0 };
	calendar_error_e error_code = CALENDAR_ERROR_NONE;

	main_view_data *view = user_data;

	st.type = CALENDAR_TIME_UTIME;
	st.time.utime = selected_date;

	// Connect with calendar
	error_code = calendar_connect();
	if (error_code != CALENDAR_ERROR_NONE)
		dlog_print(DLOG_ERROR, LOG_TAG, "calendar_connect: %d\n", error_code);

	get_tasks_and_events(view, st.time.utime);

	error_code = calendar_disconnect();
	if (error_code != CALENDAR_ERROR_NONE)
		dlog_print(DLOG_ERROR, LOG_TAG, "calendar disconnect failed: %x\n",
				error_code);
}

static i18n_ucalendar_h getToday() {
	i18n_uchar timezone[6];
	i18n_ucalendar_h ucalendar;

	i18n_ustring_copy_ua_n(timezone, "GMT+1", 6);
	i18n_ucalendar_create(timezone, -1, I18N_ULOCALE_UK, I18N_UCALENDAR_DEFAULT, &ucalendar);

	return ucalendar;
}

static void _popup_timeout_cb(void *data, Evas_Object *obj, void *event_info)
{
	main_view_data *view = (main_view_data *)data;
    evas_object_del(view->popup);
    view->popup = NULL;
}

Evas_Object *popup_create(main_view_data *view, const char *text) {
	/* Popup */
	Evas_Object *popup = elm_popup_add(view->nf);
	elm_object_style_set(popup, "toast");
	elm_object_part_text_set(popup, "default", text);
	elm_popup_timeout_set(popup, 4.0);
	evas_object_smart_callback_add(popup, "block,clicked", _popup_timeout_cb, view);
	evas_object_show(popup);

	return popup;
}

static void _edit_task_cb(void *user_data, Evas_Object *obj, void *event_info) {
	main_view_data *view = user_data;

	Elm_Object_Item* item = elm_genlist_selected_item_get(view->genlist);

	if (item != NULL) {
		edit_view_add(view, view->selected_item_db_index);
	} else {
		view->popup = popup_create(view, label_edit_warn);
	}
}

static void _set_as_completed_cb(void *user_data, Evas_Object *obj, void *event_info) {
	main_view_data *view = user_data;
	calendar_error_e error_code = CALENDAR_ERROR_NONE;
	calendar_record_h record;

	Elm_Object_Item* item = elm_genlist_selected_item_get(view->genlist);

	if (item != NULL) {
		// Connect to calendar database
		error_code = calendar_connect();
		if (error_code != CALENDAR_ERROR_NONE)
		   dlog_print(DLOG_ERROR, LOG_TAG, "calendar_connect: %d\n", error_code);

		// Set record with record id as completed
		error_code = calendar_db_get_record(_calendar_todo._uri, view->selected_item_db_index, &record);
		if (error_code != CALENDAR_ERROR_NONE)
		   dlog_print(DLOG_ERROR, LOG_TAG, "calendar_db_get_record: %d\n", error_code);

	//	print_errors();

		calendar_record_set_int(record, _calendar_todo.todo_status, CALENDAR_TODO_STATUS_COMPLETED);

		// Update the record
		error_code = calendar_db_update_record(record);
		if (error_code != CALENDAR_ERROR_NONE)
		   dlog_print(DLOG_ERROR, LOG_TAG, "db update record failed: %x\n", error_code);

		// Destroy a record handle
		calendar_record_destroy(record, true);

		// Refresh the genlist of tasks
		refresh_task_list(view, view->selected_date);

		// Disconnect from calendar database
		error_code = calendar_disconnect();
		if (error_code != CALENDAR_ERROR_NONE)
		   dlog_print(DLOG_ERROR, LOG_TAG, "calendar disconnect failed: %x\n", error_code);
	} else {
		view->popup = popup_create(view, label_completed_warn);
	}
}

static void
_clicked_cb(void *user_data, Evas_Object *obj, void *event_info)
{
	char buf[36];
	button_data_s *id = user_data;

	if (id->month + 1 < 10)
		snprintf(buf, sizeof(buf), "%d/0%d/%d", id->day, id->month + 1, id->year);
	else
		snprintf(buf, sizeof(buf), "%d/%d/%d", id->day, id->month + 1, id->year);

	main_view_data *view = (main_view_data *)evas_object_data_get(obj, KEY_PRIORITY_DATA);
	view->selected_date = _time_convert_itol(NULL, id->year, id->month, id->day);

	refresh_task_list(view, view->selected_date);
}

static void
_pressed_cb(void *user_data, Evas_Object *obj, void *event_info)
{
	main_view_data *view = user_data;

	evas_object_color_set(view->active_btn, 255, 255, 255, 255);
	evas_object_color_set(obj, 255, 240, 170, 255);

	view->active_btn = obj;
}

static Evas_Object*
create_button(void *ad, Evas_Object *parent, char *str, void *data)
{
	Evas_Object *button = elm_button_add(parent);
	elm_object_style_set(button, "default");
	evas_object_size_hint_weight_set(button, EVAS_HINT_EXPAND, 0.0);
	evas_object_size_hint_align_set(button, EVAS_HINT_FILL, 0.0);

	elm_object_part_text_set(button, NULL, str);

	evas_object_smart_callback_add(button, "pressed", _pressed_cb, ad);
	evas_object_smart_callback_add(button, "clicked", _clicked_cb, data);

	return button;
}

static void add_item_table(void *data, Evas_Object *parent) {
	char str[3];
	int year, month, day;

	main_view_data *view = data;

	i18n_ucalendar_h ucalendar = getToday();

	// gets the current value of a field from i18n_ucalendar_h
	i18n_ucalendar_get(ucalendar, I18N_UCALENDAR_YEAR, &year);
	i18n_ucalendar_get(ucalendar, I18N_UCALENDAR_MONTH, &month);
	i18n_ucalendar_get(ucalendar, I18N_UCALENDAR_DATE, &day);

	for (int i = 0; i < 7; i++) {
		sprintf(str, "%d", day);

		button_data_s *id = calloc(sizeof(button_data_s), 1);
		id->year = year;
		id->month = month;
		id->day = day;

		Evas_Object *content = create_button(data, parent, str, id);
		evas_object_data_set(content, KEY_PRIORITY_DATA, view);
		evas_object_show(content);
		elm_table_pack(parent, content, i, 0, 1, 1);

		// Get the next day
		i18n_ucalendar_add(ucalendar, I18N_UCALENDAR_DATE, 1);
		i18n_ucalendar_get(ucalendar, I18N_UCALENDAR_YEAR, &year);
		i18n_ucalendar_get(ucalendar, I18N_UCALENDAR_MONTH, &month);
		i18n_ucalendar_get(ucalendar, I18N_UCALENDAR_DATE, &day);
	}
}

void todos_list_fill(){
		calendar_error_e error_code = CALENDAR_ERROR_NONE;
		calendar_list_h list = NULL;
		int count = 0;
		int status;
		calendar_record_h recordOut;

		// Connect with calendar
		error_code = calendar_connect();
		if (error_code != CALENDAR_ERROR_NONE)
		   dlog_print(DLOG_ERROR, LOG_TAG, "calendar_connect: %d\n", error_code);

		// Check if calendar contains any tasks...
		error_code = calendar_db_get_all_records(_calendar_todo._uri, 0, 0, &list);

		if (error_code != CALENDAR_ERROR_NONE)
		   dlog_print(DLOG_ERROR, LOG_TAG, "db get all records failed: %d\n", error_code);

		error_code = calendar_list_get_count(list, &count);

		if (error_code != CALENDAR_ERROR_NONE)
		   dlog_print(DLOG_ERROR, LOG_TAG, "calendar_list_get_count failed: %d\n", error_code);

		if (count != 0)
			while (calendar_list_get_current_record_p(list, &recordOut) == CALENDAR_ERROR_NONE)
			{
				status = 0;
				count = 0;
				// Get record status
				calendar_record_get_int(recordOut, _calendar_todo.todo_status,
						&status);

				if (status != CALENDAR_TODO_STATUS_COMPLETED
						&& status != CALENDAR_TODO_STATUS_CANCELED) {
					count++;
				}

				error_code = calendar_list_next(list);
				if (error_code != CALENDAR_ERROR_NONE)
					break;
			}

		// ...if no - create some tasks
		if (count == 0) {
			char str[2];
			int y, m, d;
			// Get actual date
			i18n_ucalendar_h ucalendar = getToday();

			// gets the current value of a field from i18n_ucalendar_h
			i18n_ucalendar_get(ucalendar, I18N_UCALENDAR_YEAR, &y);
			i18n_ucalendar_get(ucalendar, I18N_UCALENDAR_MONTH, &m);
			i18n_ucalendar_get(ucalendar, I18N_UCALENDAR_DATE, &d);

			long long int x = _time_convert_itol(NULL, y, m, d);

			for (int i = 0; i < 7; i++) {
				sprintf(str, "Task%d", i);
				create_test_todo_task(str, x);
				x += SECONDS_IN_DAY;
			}
		}

		error_code = calendar_disconnect();
		if (error_code != CALENDAR_ERROR_NONE)
		   dlog_print(DLOG_ERROR, LOG_TAG, "calendar disconnect failed: %x\n", error_code);
}

void *main_view_add(void *data) {
	appdata_s *ad = (appdata_s *) data;
	main_view_data *view = calloc(1, sizeof(main_view_data));

	view->win = ad->win;
	view->nf = ad->navi;
	ad->main_view_data = view;

	Evas_Object *box, *box2, *label;

	/* Box */
	box = elm_box_add(view->nf);
	evas_object_size_hint_weight_set(box, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(box, EVAS_HINT_FILL, EVAS_HINT_FILL);

	/* Second box with table */
	box2 = elm_box_add(box);
	evas_object_size_hint_weight_set(box2, EVAS_HINT_EXPAND, 0.1);
	evas_object_size_hint_align_set(box2, EVAS_HINT_FILL, EVAS_HINT_FILL);
	evas_object_show(box2);

	/* Instruction label */
	label = elm_label_add(box2);
	evas_object_size_hint_weight_set(label, EVAS_HINT_EXPAND, 0.0);
	elm_object_text_set(label, label_select);
	evas_object_show(label);

	elm_box_pack_end(box2, label);

	/* Table */
	view->content_table = elm_table_add(box2);
	elm_table_homogeneous_set(view->content_table, EINA_TRUE);
	evas_object_size_hint_weight_set(view->content_table, EVAS_HINT_EXPAND,
			0.0);
	evas_object_size_hint_align_set(view->content_table, EVAS_HINT_FILL,
			EVAS_HINT_FILL);
	elm_table_padding_set(view->content_table, ELM_SCALE_SIZE(5),
			ELM_SCALE_SIZE(5));
	evas_object_show(view->content_table);

	// Adds elements to the table
	add_item_table(view, view->content_table);

	evas_object_show(view->content_table);

	elm_box_pack_end(box2, view->content_table);

	elm_box_pack_end(box, box2);

	/* GenList of tasks */
	view->genlist = elm_genlist_add(box);
	evas_object_size_hint_weight_set(view->genlist, EVAS_HINT_EXPAND, 0.75);
	evas_object_size_hint_align_set(view->genlist, EVAS_HINT_FILL,
			EVAS_HINT_FILL);
	elm_box_pack_end(box, view->genlist);
	evas_object_show(view->genlist);

	/* Menu of two buttons for tasks */
	Evas_Object *toolbar = elm_toolbar_add(box);
	evas_object_size_hint_weight_set(toolbar, EVAS_HINT_EXPAND, 0.05);
	evas_object_size_hint_align_set(toolbar, EVAS_HINT_FILL, EVAS_HINT_FILL);
	elm_object_style_set(toolbar, "default");
	elm_toolbar_shrink_mode_set(toolbar, ELM_TOOLBAR_SHRINK_EXPAND);
	elm_toolbar_transverse_expanded_set(toolbar, EINA_TRUE);

	{
		Elm_Object_Item *completed_it, *edit_it;

		completed_it = elm_toolbar_item_append(toolbar, NULL, label_completed,
				_set_as_completed_cb, view);
		edit_it = elm_toolbar_item_append(toolbar, NULL, label_edit,
				_edit_task_cb, view);
	}

	evas_object_show(toolbar);
	elm_box_pack_end(box, toolbar);

	evas_object_show(box);

	view->navi_item = elm_naviframe_item_push(ad->navi, _("Custom Organizer"),
			NULL, NULL, box, "basic");

	/* Exit button */
	Evas_Object *exit_btn = elm_button_add(view->nf);
	elm_object_style_set(exit_btn, "naviframe/title_cancel");
	evas_object_show(exit_btn);

	evas_object_smart_callback_add(exit_btn, "clicked", win_delete_request_cb,
			NULL);
	elm_object_item_part_content_set(view->navi_item, "title_right_btn",
			exit_btn);

	eext_object_event_callback_add(view->win, EEXT_CALLBACK_BACK,
			main_view_back_cb, view);
	/* Show window after base gui is set up */
	evas_object_show(view->win);

	todos_list_fill();
}
