#include "edit-view.h"
#include "customorganizer.h"
#include "calendar-utils.h"
#include "logger.h"

static const char *label_view_title   = "Edit task";
static const char *label_cancel_btn   = "Cancel";
static const char *label_save_btn     = "Save";
static const char *label_task_name    = "Task Name";
static const char *label_st_date      = "Start date";
static const char *label_status       = "Status";
static const char *button_cancel		= "Cancel";
static const char *button_set			= "Set";

static void remove_context_popup(Evas_Object *ctxpopup){
	if (ctxpopup != NULL) {
		evas_object_del(ctxpopup);
		ctxpopup = NULL;
	}
}

static void remove_date_time_popup(edit_view_data *view){
	if (view->start_date_popup != NULL) {
		evas_object_del(view->start_date_popup);
		view->start_date_popup = NULL;

		evas_object_del(view->start_datetime);
		view->start_datetime = NULL;
	}
}

static void close_calendar_connection(edit_view_data *view){
	calendar_error_e error_code = CALENDAR_ERROR_NONE;

	// Destroy a record handle
	calendar_record_destroy(view->todo_record, true);

	// Disconnect from calendar database
	error_code = calendar_disconnect();
	if (error_code != CALENDAR_ERROR_NONE)
	   dlog_print(DLOG_ERROR, LOG_TAG, "calendar disconnect failed: %x\n", error_code);
}

static Eina_Bool
naviframe_pop_cb(void *data, Elm_Object_Item *it)
{
	edit_view_data *view = (edit_view_data *)data;

	close_calendar_connection(view);

	remove_date_time_popup(view);
	remove_context_popup(view->ctxpopup);

	/* By returning FALSE, you can reject the popping the item */
	return EINA_TRUE;
}

void _ctxpopup_dismissed_cb(void *data, Evas_Object *obj, void *event_info)
{
	edit_view_data *view = (edit_view_data *)data;
	remove_context_popup(view->ctxpopup);
}

static void
_ctxpopup_item_cb(void *data, Evas_Object *obj, void *event_info)
{
	calendar_todo_status_e status = (calendar_todo_status_e)data;
	edit_view_data *view = (edit_view_data *)evas_object_data_get(obj, KEY_PRIORITY_DATA);

	if (!view) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Passed NULL input data");
	}

	const char *label = elm_object_item_text_get((Elm_Object_Item *) event_info);
	elm_object_text_set((Evas_Object*)view->status_btn, label);

	view->task_status = status;

    remove_context_popup(view->ctxpopup);
}

static void
move_ctxpopup(Evas_Object *ctxpopup, Evas_Object *btn)
{
	Evas_Coord x, y, w , h;
	evas_object_geometry_get(btn, &x, &y, &w, &h);
	evas_object_move(ctxpopup, (w / 2), y + (h / 2));
}

static void
_status_clicked_cb(void *user_data, Evas_Object *obj, void *event_info)
{
	edit_view_data *view = user_data;

	remove_context_popup(view->ctxpopup);

	view->ctxpopup = elm_ctxpopup_add(view->nf);

	eext_object_event_callback_add(view->ctxpopup, EEXT_CALLBACK_BACK, eext_ctxpopup_back_cb, NULL);
	evas_object_smart_callback_add(view->ctxpopup, "dismissed", _ctxpopup_dismissed_cb, NULL);

	evas_object_data_set(view->ctxpopup, KEY_PRIORITY_DATA, view);

	for (int i = 0; i < 5; i++)
		elm_ctxpopup_item_append(view->ctxpopup, status_info[i].name, NULL, _ctxpopup_item_cb, (void *)status_info[i].type);

	move_ctxpopup(view->ctxpopup, obj);
	evas_object_show(view->ctxpopup);
}

static void format_date(struct tm set_date, char *buffer){
	if (set_date.tm_mon + 1 < 10)
		snprintf(buffer, 11, "%d/0%d/%d", set_date.tm_mday, set_date.tm_mon + 1, set_date.tm_year + 1900);
	else
		snprintf(buffer, 11, "%d/%d/%d", set_date.tm_mday, set_date.tm_mon + 1, set_date.tm_year + 1900);
}

static void _date_time_popup_set_cb(void *data, Evas_Object *obj, void *event_info) {
	edit_view_data *view = (edit_view_data *)evas_object_data_get(obj, KEY_PRIORITY_DATA);

    elm_datetime_value_get(view->start_datetime, &view->task_start_date);

	char buffer[11];
	format_date(view->task_start_date, buffer);
    elm_object_text_set(view->start_date_btn, buffer);

	remove_date_time_popup(view);
}

static void _date_time_popup_cancel_cb(edit_view_data *view, Evas_Object *obj, void *event_info) {
	remove_date_time_popup(view);
}

static void get_start_datetime(edit_view_data *view, Evas_Object *parent, const char *format)
{
    if(!view->start_datetime)
    {
        view->start_datetime = elm_datetime_add(parent);
        evas_object_size_hint_weight_set(view->start_datetime, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
        evas_object_size_hint_align_set(view->start_datetime, EVAS_HINT_FILL, EVAS_HINT_FILL);
    }

    elm_datetime_value_set(view->start_datetime, &view->task_start_date);
    elm_datetime_format_set(view->start_datetime, format);
    evas_object_show(view->start_datetime);
}

static Evas_Object *date_time_popup_add(Evas_Object *parent, void *data, Evas_Smart_Cb set_cb, Evas_Smart_Cb cancel_cb)
{
	edit_view_data *view = data;

	Evas_Object *popup = elm_popup_add(parent);
	view->start_date_popup = popup;

	elm_popup_orient_set(popup, ELM_POPUP_ORIENT_CENTER);
	evas_object_show(popup);

	Evas_Object *cancel_btn = elm_button_add(popup);
	elm_object_style_set(cancel_btn, "popup");
	elm_object_text_set(cancel_btn, button_cancel);
	elm_object_part_content_set(popup, "button1", cancel_btn);
	evas_object_smart_callback_add(cancel_btn, "clicked", cancel_cb, view);

	Evas_Object *set_btn = elm_button_add(popup);
	elm_object_style_set(set_btn, "popup");
	elm_object_text_set(set_btn, button_set);
	elm_object_part_content_set(popup, "button2", set_btn);
	evas_object_data_set(set_btn, KEY_PRIORITY_DATA, view);
	evas_object_smart_callback_add(set_btn, "clicked", set_cb, view);

	return popup;
}

static void _start_datatime_clicked_cb(void *user_data, Evas_Object *obj, void *event_info) {
	edit_view_data *view = user_data;

    Evas_Object *popup = date_time_popup_add(view->nf, (void *)view, _date_time_popup_set_cb, _date_time_popup_cancel_cb);
    evas_object_show(popup);

	get_start_datetime(view, popup, "%d %b %Y");

	elm_object_content_set(popup, view->start_datetime);
}

static void _edit_cancel (void *user_data, Evas_Object *obj, void *event_info)
{
	edit_view_data *view = user_data;
	elm_naviframe_item_pop(view->nf);
}

static void _edit_save (void *user_data, Evas_Object *obj, void *event_info)
{
	calendar_error_e error_code = CALENDAR_ERROR_NONE;
	edit_view_data *view = user_data;

	set_start_time(view->todo_record, view->task_start_date);

	set_todo_status(view->todo_record, view->task_status);

	// Update the record
	error_code = calendar_db_update_record(view->todo_record);
	if (error_code != CALENDAR_ERROR_NONE)
	   dlog_print(DLOG_ERROR, LOG_TAG, "db update record failed: %x\n", error_code);

	elm_naviframe_item_pop(view->nf);
}

Evas_Object *create_label(Evas_Object *parent, const char *label_name) {
	Evas_Object *label = elm_label_add(parent);
	evas_object_size_hint_weight_set(label, EVAS_HINT_EXPAND, 0.0);
	elm_object_text_set(label, label_name);
	evas_object_show(label);

	return label;
}

Evas_Object *create_button(Evas_Object *parent, const char *str) {
	Evas_Object *button = elm_button_add(parent);
	elm_object_style_set(button, "default");
	evas_object_size_hint_weight_set(button, EVAS_HINT_EXPAND, 0.0);
	evas_object_size_hint_align_set(button, EVAS_HINT_FILL, 0.0);

	elm_object_part_text_set(button, NULL, str);

	return button;
}

void *edit_view_add(void *data, int record_index) {
	calendar_error_e error_code = CALENDAR_ERROR_NONE;

	appdata_s *ad = (appdata_s *)data;
	edit_view_data *view = calloc(1, sizeof(edit_view_data));

	view->win = ad->win;
	view->nf = ad->navi;

	Evas_Object *box = elm_box_add(view->nf);
	evas_object_size_hint_weight_set(box, EVAS_HINT_EXPAND, 1.0);
	evas_object_size_hint_align_set(box, EVAS_HINT_FILL, EVAS_HINT_FILL);
	evas_object_show(box);

	Evas_Object *content_table = elm_table_add(box);
	elm_table_homogeneous_set(content_table, EINA_TRUE);
	evas_object_size_hint_weight_set(content_table, EVAS_HINT_EXPAND, 0.6);
	evas_object_size_hint_align_set(content_table, EVAS_HINT_FILL, EVAS_HINT_FILL);
	elm_table_padding_set(content_table, ELM_SCALE_SIZE(5), ELM_SCALE_SIZE(5));

	elm_table_pack(content_table, create_label(content_table, label_task_name), 0, 0, 1, 1);
	elm_table_pack(content_table, create_label(content_table, label_st_date), 0, 1, 1, 1);
	elm_table_pack(content_table, create_label(content_table, label_status), 0, 2, 1, 1);

	// Connect to calendar database
	error_code = calendar_connect();
	if (error_code != CALENDAR_ERROR_NONE)
		dlog_print(DLOG_ERROR, LOG_TAG, "calendar_connect: %d\n", error_code);

	// Get informations about task
	view->todo_record = get_todo_record(record_index);

	// Task name
	const char *task_name = get_name_of_task(view->todo_record);

	Evas_Object *task_name_label = create_label(content_table, task_name);
	evas_object_show(task_name_label);
	elm_table_pack(content_table, task_name_label, 1, 0, 1, 1);

	// Start date
	view->task_start_date = get_start_date_of_task(view->todo_record);

	// Format task start date & display
	char buffer[11];
	format_date(view->task_start_date, buffer);

	view->start_date_btn = create_button(content_table, buffer);
	evas_object_smart_callback_add(view->start_date_btn, "clicked", _start_datatime_clicked_cb, view);
	evas_object_show(view->start_date_btn);
	elm_table_pack(content_table, view->start_date_btn, 1, 1, 1, 4);

	// Status
//	int status_index = get_status_of_task(view->todo_record);
	view->task_status = get_status_of_task(view->todo_record);
	const char *status_name = get_status_name_of_task(view->task_status);

	view->status_btn = create_button(content_table, status_name);
	evas_object_smart_callback_add(view->status_btn, "clicked", _status_clicked_cb, view);
	evas_object_show(view->status_btn);
	elm_table_pack(content_table, view->status_btn, 1, 2, 1, 4);

	evas_object_show(content_table);

	elm_box_pack_end(box, content_table);

	Evas_Object *box2 = elm_box_add(view->nf);
	evas_object_size_hint_weight_set(box2, EVAS_HINT_EXPAND, 0.4);
	evas_object_size_hint_align_set(box2, EVAS_HINT_FILL, EVAS_HINT_FILL);
	evas_object_show(box2);

	elm_box_pack_end(box, box2);

	/* Toolbar with Save & Cancel options */
	Evas_Object *toolbar = elm_toolbar_add(box);
	evas_object_size_hint_weight_set(toolbar, EVAS_HINT_EXPAND, 0.05);
	evas_object_size_hint_align_set(toolbar, EVAS_HINT_FILL, EVAS_HINT_FILL);
	elm_object_style_set(toolbar, "default");
	elm_toolbar_shrink_mode_set(toolbar, ELM_TOOLBAR_SHRINK_EXPAND);
	elm_toolbar_transverse_expanded_set(toolbar, EINA_TRUE);

	Elm_Object_Item *cancel_it, *save_it;

	cancel_it = elm_toolbar_item_append(toolbar, NULL, label_cancel_btn, _edit_cancel, view);
	save_it = elm_toolbar_item_append(toolbar, NULL, label_save_btn, _edit_save, view);

	evas_object_show(toolbar);
	elm_box_pack_end(box, toolbar);

	/* Push a new item */
	Elm_Object_Item *nf_it = elm_naviframe_item_push(view->nf, label_view_title, NULL, NULL, box, NULL);

	elm_naviframe_item_pop_cb_set(nf_it, naviframe_pop_cb, view);
}
