#include "calendar-utils.h"
#include "logger.h"

/**
 * Create test todos with specified name and start time
 * status and calendar type are predefined
 */
void create_test_todo_task(char *summary, long long int startTime)
{
	calendar_error_e error_code = CALENDAR_ERROR_NONE;
	calendar_record_h todo = NULL;
	calendar_time_s start_time = {0};
	int calendar_type = DEFAULT_TODO_CALENDAR_BOOK_ID;
	int id = 0;
	int todo_status = CALENDAR_TODO_STATUS_IN_PROCESS;

	/* URI */
	error_code = calendar_record_create(_calendar_todo._uri, &todo);
	if (error_code != CALENDAR_ERROR_NONE)
	   dlog_print(DLOG_ERROR, LOG_TAG, "calendar_record_create: %x\n", error_code);

	/* Summary */
	error_code = calendar_record_set_str(todo, _calendar_todo.summary, summary);
	if (error_code != CALENDAR_ERROR_NONE)
	   dlog_print(DLOG_ERROR, LOG_TAG, "set summary failed : %x\n", error_code);

	/* Start time */
	start_time.type = CALENDAR_TIME_UTIME;
	// utime in seconds
	start_time.time.utime = startTime;

	error_code = calendar_record_set_caltime(todo, _calendar_todo.start_time, start_time);
	if (error_code != CALENDAR_ERROR_NONE)
	   dlog_print(DLOG_ERROR, LOG_TAG, "set due_time failed : %x\n", error_code);

	/* Status */
	error_code = calendar_record_set_int(todo, _calendar_todo.todo_status, todo_status);
	if (error_code != CALENDAR_ERROR_NONE)
	   dlog_print(DLOG_ERROR, LOG_TAG, "set todo_status failed : %x\n", error_code);

	/* Book Id */
	error_code = calendar_record_set_int(todo, _calendar_todo.calendar_book_id, calendar_type);
	if (error_code != CALENDAR_ERROR_NONE)
	   dlog_print(DLOG_ERROR, LOG_TAG, "set calendar_book_id failed: %x\n", error_code);

	/* Insert record into db... */
	error_code = calendar_db_insert_record(todo, &id);
	if (error_code != CALENDAR_ERROR_NONE)
	   dlog_print(DLOG_ERROR, LOG_TAG, "db insert record failed: %d\n", error_code);

	/* ...and destroy a record handle */
	calendar_record_destroy(todo, true);
}

void set_todo_status(calendar_record_h record, calendar_todo_status_e todo_status) {
	calendar_error_e error_code = CALENDAR_ERROR_NONE;

	// Set new status
	error_code = calendar_record_set_int(record, _calendar_todo.todo_status, todo_status);
	if (error_code != CALENDAR_ERROR_NONE)
	   dlog_print(DLOG_ERROR, LOG_TAG, "calendar_record_set_int failed: %x\n", error_code);
}

void set_start_time(calendar_record_h record, struct tm task_start_date){
	calendar_error_e error_code = CALENDAR_ERROR_NONE;
    calendar_time_s start_time;
    memset(&start_time, 0x0, sizeof(start_time));

    calendar_record_get_caltime(record, _calendar_event.start_time, &start_time);

    start_time.type = CALENDAR_TIME_UTIME;
    start_time.time.utime = mktime(&task_start_date);

	// Set new start date
	error_code = calendar_record_set_caltime(record, _calendar_todo.start_time, start_time);
	if (error_code != CALENDAR_ERROR_NONE)
	   dlog_print(DLOG_ERROR, LOG_TAG, "set due_time failed : %x\n", error_code);
}

int get_status_of_task(calendar_record_h record) {
	int status;

	calendar_record_get_int(record, _calendar_todo.todo_status, &status);

	dlog_print(DLOG_INFO, LOG_TAG, "task status: %d\n", status);

    switch(status)
    {
        case CALENDAR_TODO_STATUS_NONE: {
        	dlog_print(DLOG_INFO, LOG_TAG, "CALENDAR_TODO_STATUS_NONE");
        	status = 0;
        }
            break;
        case CALENDAR_TODO_STATUS_NEEDS_ACTION: {
        	dlog_print(DLOG_INFO, LOG_TAG, "CALENDAR_TODO_STATUS_NEEDS_ACTION");
        	status = 1;
        }
            break;
        case CALENDAR_TODO_STATUS_COMPLETED: {
        	dlog_print(DLOG_INFO, LOG_TAG, "CALENDAR_TODO_STATUS_COMPLETED");
        	status = 2;
        }
            break;
        case CALENDAR_TODO_STATUS_IN_PROCESS: {
        	dlog_print(DLOG_INFO, LOG_TAG, "CALENDAR_TODO_STATUS_IN_PROCESS");
        	status = 3;
        }
            break;
        case CALENDAR_TODO_STATUS_CANCELED: {
        	dlog_print(DLOG_INFO, LOG_TAG, "CALENDAR_TODO_STATUS_CANCELED");
        	status = 4;
        }
            break;
        default:
        	break;
    }

	return status;
}

char *get_name_of_task(calendar_record_h record) {
	char *task_name;

	calendar_record_get_str_p(record, _calendar_todo.summary, &task_name);

	return task_name;
}

struct tm get_start_date_of_task(calendar_record_h record) {
	struct tm task_date;

	calendar_time_s start_time = {0};
	calendar_record_get_caltime(record, _calendar_todo.start_time, &start_time);

	struct tm time_start_temp = {0};
    const time_t date = start_time.time.utime;
    time_start_temp = *localtime(&date);
    memcpy(&task_date, &time_start_temp, sizeof(struct tm));

    return task_date;
}

calendar_record_h get_todo_record(int record_index) {
	calendar_error_e error_code = CALENDAR_ERROR_NONE;
	calendar_record_h record;

	// Set record with record id as completed
	error_code = calendar_db_get_record(_calendar_todo._uri, record_index, &record);
	if (error_code != CALENDAR_ERROR_NONE)
		dlog_print(DLOG_ERROR, LOG_TAG, "calendar_db_get_record: %d\n", error_code);

	return record;
}


