/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

using Xamarin.Forms;

namespace CopynPasteApp
{

    /// <summary>
    /// A start page
    /// </summary>
    public class MainPage : ContentPage
    {
        static string SourceImage = "source.png";
        static string TargetImage = "target.png";
        private AbsoluteLayout _pageLayout;
        private Label _sourceLabel;
        private Image _sourceBgImage;
        private Image _targetBgImage;
        private Editor _targetEditor;
        private Image _hoverBgImage;

        private PanGestureRecognizer dragHoverPart;

        private double _hoverPositionX;
        private double _hoverPositionY;


        // 720X1280 for HD, 480X800 for WVGA
        private int ScreenWidth { get; set; }
        private int ScreenHeight { get; set; }

        /// <summary>
        /// Initializes a new instance of the <see cref="MainPage"/> class.
        /// </summary>
        /// <param name="screenWidth">The screen width</param>
        /// <param name="screenHeight">The screen height</param>
        public MainPage(int screenWidth, int screenHeight)
        {
            ScreenWidth = screenWidth;
            ScreenHeight = screenHeight;

            // created Initialization and user logic
            InitializeComponent();

            // Set the bounds of children
            SetChildBounds();

            // Add children to page
            LayoutChildren();

            this.Content = _pageLayout;

        }

        /// <summary>
        /// a method to initialize component
        /// </summary>
        private void InitializeComponent()
        {
            _hoverPositionX = 100;
            _hoverPositionY = 100;

            dragHoverPart = new PanGestureRecognizer();
            dragHoverPart.PanUpdated += DragHoverPartPanUpdated;

            // UI Logic Development
            _pageLayout = new AbsoluteLayout();

            _sourceBgImage = new Image
            {
                Source = SourceImage,
                Aspect = Aspect.Fill,
            };
            _sourceBgImage.GestureRecognizers.Add(dragHoverPart);

            _hoverBgImage = new Image
            {
                Source = SourceImage,
                Aspect = Aspect.Fill,
                IsVisible = false
            };

            _targetBgImage = new Image
            {
                Source = TargetImage,
                Aspect = Aspect.Fill,
            };

            /// <summary>
            /// Create center label.
            /// </summary>
            _sourceLabel = new Label()
            {
                Text =
                "Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua. Ut enim ad minim veniam, quis nostrud exercitation ullamco laboris nisi ut aliquip ex ea commodo consequat. Duis aute irure dolor in reprehenderit in voluptate velit esse cillum dolore eu fugiat nulla pariatur. Excepteur sint occaecat cupidatat non proident, sunt in culpa qui officia deserunt mollit anim id est laborum",
                TextColor = Color.Gray,
            };
            _sourceLabel.GestureRecognizers.Add(dragHoverPart);

            _targetEditor = new Editor
            {
                TextColor = Color.Red,
            };
        }

        private void DragHoverPartPanUpdated(object sender, PanUpdatedEventArgs e)
        {
            switch (e.StatusType)
            {
                case GestureStatus.Started:
                    _hoverBgImage.IsVisible = true;
                    _hoverBgImage.TranslationX = e.TotalX;
                    _hoverBgImage.TranslationY = e.TotalY;
                    break;
                case GestureStatus.Running:
                    _hoverPositionX = 100 + e.TotalX;
                    _hoverPositionY = 100 + e.TotalY;
                    _hoverBgImage.TranslationX = e.TotalX;
                    _hoverBgImage.TranslationY = e.TotalY;
                    break;
                case GestureStatus.Completed:
                    _hoverBgImage.IsVisible = false;
                    if (_hoverPositionX > 40 && _hoverPositionY > ScreenHeight / 2)
                    {
                        System.Diagnostics.Debug.WriteLine("########### PasteToTarget ########### !!!!!!!!!!!!!!!!");

                        PasteToTarget();
                    }
                    _hoverBgImage.TranslationX = 100;
                    _hoverBgImage.TranslationY = 100;
                    break;
            }
        }

        // Set the bounds of children
        private void SetChildBounds()
        {
            System.Diagnostics.Debug.WriteLine("###########DragHoverPartPanUpdated ScreenWidth:{0}, ScreenHeight:{1}", ScreenWidth, ScreenHeight);

            AbsoluteLayout.SetLayoutBounds(_sourceBgImage, new Rectangle(0f, 0f, ScreenWidth, ScreenHeight/2));
            AbsoluteLayout.SetLayoutBounds(_sourceLabel, new Rectangle(40f, 50f, ScreenWidth - 80, ScreenHeight / 2 - 40f));
            AbsoluteLayout.SetLayoutBounds(_targetBgImage, new Rectangle(0f, ScreenHeight / 2 - 14, ScreenWidth, ScreenHeight / 2));
            AbsoluteLayout.SetLayoutBounds(_targetEditor, new Rectangle(46f, ScreenHeight / 2 + 60f, ScreenWidth - 80, ScreenHeight / 2 - 120f));
            AbsoluteLayout.SetLayoutBounds(_hoverBgImage, new Rectangle(100f, 100f, ScreenWidth - 200, ScreenHeight / 2 - 220));
        }

        // Add children to page
        private void LayoutChildren()
        {
            _pageLayout.Children.Add(_sourceBgImage);
            _pageLayout.Children.Add(_sourceLabel);
            _pageLayout.Children.Add(_targetBgImage);
            _pageLayout.Children.Add(_targetEditor);
            _pageLayout.Children.Add(_hoverBgImage);
        }

        private bool HitTest()
        {
            System.Diagnostics.Debug.WriteLine("###########DragHoverPartPanUpdated _hoverBgImage.X:{0}, _hoverBgImage.Y:{1}", _hoverBgImage.X, _hoverBgImage.Y);

            if (_hoverBgImage.X > 60 && _hoverBgImage.Y > 700)
            {
                return true;
            }
            else
            {
                return false;
            }
        }

        private void PasteToTarget()
        {
            _targetEditor.Text = _sourceLabel.Text;
        }

    }
}