/*******************************************************************************
 * Module responsible for showing and managing texts' and images' properties
 * (excluding color picker popup)
 *
 * @author Zaneta Szymanska <<ahref='mailto:z.szymanska@samsung.com'>z.szymanska@samsung.com</a>>
 *
 * Copyright (c) 2012 Samsung Electronics All Rights Reserved.
 ******************************************************************************/
"use strict";
var messages = function() {
    // Color picker handler
    var colorPicker;

    /**
     * Sets proper values for glowProp attributes and refreshes the "glow"
     * effect connected with the element
     */
    var refreshGlow = function() {
        var selected = collage.getSelectedItem();

        if (copyGlow(glowProperties, selected.glowProp)) {
            selected.refreshGlow();
        }
    };

    /**
     * Copies attributes' values from fromObject to toObject
     *
     * @param {Object} fromObject Object, from which attributes' values should
     *        be copied
     * @param {Object} toObject Object, to which attributes' values should be
     *        copied
     *
     */
    var copyGlow = function(fromObject, toObject) {
        for ( var v in fromObject) {
            toObject[v] = fromObject[v];
        }

        return true;
    };

    /**
     * Sets attribute's value for selected textObject
     *
     * @param {String} attrName Name of the attribute, which value is changed
     * @param {String} value Adjusted value
     */
    var setSelectedTextAttr = function(attrName, value) {
        if (collage.getSelectedItem()) {
            collage.getSelectedItem().attr(attrName, value);
        }
    };

    /**
     * Adds event handlers to all controls used to set properties of the created
     * or selected textObject
     */
    var initTextPopupControls = function() {
        /** Font properties */
        $("#font-size").on("change", function() {
            font.size = $("#font-size").val();
            setSelectedTextAttr("font-size", font.size);
        });

        $("#bold").on("change", function() {
            if ($("#bold").prop("checked")) {
                font.weight = "bold";
            } else {
                font.weight = "normal";
            }

            setSelectedTextAttr("font-weight", font.weight);
        });

        $("#italic").on("change", function() {
            if ($("#italic").prop("checked")) {
                font.style = "italic";
            } else {
                font.style = "normal";
            }

            setSelectedTextAttr("font-style", font.style);
        });

        $("#font-color").on("click", function() {
            $("#color-picker").show();
        });

        $("#sample-text").on("change", function() {
            textContent = $("#sample-text").val();
        });
    };

    /**
     * Removes or adds "glow" effect to the selected element
     */
    var modifyGlow = function() {
        var selected = collage.getSelectedItem();

        if ($("#adding-glow").is(":checked")) {
            $("#glow-prop").show();

            if (selected.glowProp) {
                copyGlow(glowProperties, selected.glowProp);
            } else {
                selected.glowProp = {
                    "width" : glowProperties.width,
                    "fill" : glowProperties.fill,
                    "opacity" : glowProperties.opacity,
                    "offsetx" : glowProperties.offsetx,
                    "offsety" : glowProperties.offsety,
                    "color" : glowProperties.color
                };
            }

            selected.g = selected.glow(selected.glowProp);

        } else {
            $("#glow-prop").hide();

            if (selected.g && (selected.g !== null)) {
                selected.removeGlow();
                selected.glowProp = null;
            }
        }
    };

    /**
     * Adds event handlers to all controls used to set properties of the created
     * or selected imageObject
     */
    var initImagePopupControls = function() {
        /** Image properties */
        $("#image-opacity-slider").on("change", function() {
            elementProperties.opacity = $("#image-opacity-slider").val();

            collage.getSelectedItem().attr({
                "opacity" : elementProperties.opacity
            });
        });

        $("#adding-glow").on("change", function() {
            modifyGlow();
        });

        $("#glow-width-slider").on("slidestop", function() {
            glowProperties.width = $("#glow-width-slider").val();
            refreshGlow();
        });

        $("#glow-opacity-slider").on("change", function() {
            glowProperties.opacity = $("#glow-opacity-slider").val();
            refreshGlow();
        });

        $("#offset-x-slider").on("change", function() {
            glowProperties.offsetx = $("#offset-x-slider").val();
            refreshGlow();
        });

        $("#offset-y-slider").on("change", function() {
            glowProperties.offsety = $("#offset-y-slider").val();
            refreshGlow();
        });

        $("#glow-fill").on("change", function() {
            if ($("#glow-fill").is(":checked")) {
                glowProperties.fill = true;
            } else {
                glowProperties.fill = false;
            }

            refreshGlow();
        });

        $("#glow-color").on("click", function() {
            $("#color-picker").show();
        });

        $("#close-image-prop").bind("click", function() {
            $(".popup").hide();
        });
    };

    /**
     * Changes color of UI element and selected shape (if exist)
     *
     * @param {String} color Color to set
     */
    var changeColor = function(color) {
        var item = collage.getSelectedItem();

        if (item !== null) {
            if (item.type === "image") {
                glowProperties.color = color;
                item.glowProp.color = color;
                item.refreshGlow();
            } else {
                font.color = color;
                item.attr("fill", color);
            }
        } else {
            font.color = color;
        }
    };

    /**
     * Initializes color picker
     */
    var pickerInit = function() {
        Raphael(function() {
            // this is where colorpicker created
            colorPicker = Raphael.colorpicker(600, 100, 300, "#0f0", document
                    .getElementById("color-picker"));

            // assigning onchange event handler
            var onchange = function() {
                return function(clr) {
                    var val = clr.replace(/^#(.)\1(.)\2(.)\3$/, "#$1$2$3");
                    changeColor(val);
                };
            };

            colorPicker.onchange = onchange();

            $("#cp-close").on("click", function(e) {
                e.preventDefault();
                // close color picker
                $("#color-picker").hide();
            });
        });
    };

    return {
        /**
         * Initializes controls
         */
        initialize : function() {
            initTextPopupControls();
            initImagePopupControls();
            pickerInit();
        },

        /**
         * Sets properties of textObject's popup
         */
        setTextPopup : function() {
            var selected = collage.getSelectedItem();

            $("#font-size").val(selected.attr("font-size"));

            if (selected.attr("font-weight") === "bold") {
                $("#bold").prop("checked", true).checkboxradio("refresh");
            } else {
                $("#bold").prop("checked", false).checkboxradio("refresh");
            }

            if (selected.attr("font-style") === "italic") {
                $("#italic").prop("checked", true).checkboxradio("refresh");
            } else {
                $("#italic").prop("checked", false).checkboxradio("refresh");
            }

            colorPicker.color(selected.attr("fill"));

            $("#sample-text").val(selected.content);
        },

        /**
         * Sets properties of imageObject's popup
         */
        setImagePopup : function() {
            var selected = collage.getSelectedItem();

            $("#image-opacity-slider").val(selected.attr("opacity")).slider(
                    "refresh");

            if (selected.glowProp) {
                if (copyGlow(selected.glowProp, glowProperties)) {
                    $("#adding-glow").prop("checked", true).checkboxradio(
                            "refresh");
                    $("#glow-prop").show();

                    $("#glow-width-slider").val(glowProperties["width"])
                            .slider("refresh");
                    $("#glow-opacity-slider").val(glowProperties["opacity"])
                            .slider("refresh");
                    $("#offset-x-slider").val(glowProperties["offsetx"])
                            .slider("refresh");
                    $("#offset-y-slider").val(glowProperties["offsety"])
                            .slider("refresh");

                    colorPicker.color(selected.glowProp.color);

                    if (glowProperties["fill"]) {
                        $("#glow-fill").prop("checked", true).checkboxradio(
                                "refresh");
                    } else {
                        $("#glow-fill").prop("checked", false).checkboxradio(
                                "refresh");
                    }
                }
            } else {
                $("#adding-glow").prop("checked", false).checkboxradio(
                        "refresh");
                $("#glow-prop").hide();
            }
        },

        /**
         * @param {String}
         *            type Popup of text properties type {"create" or "update"}
         */
        defineTextProp : function(type) {
            if (type === "create") {
                collage.setItemSelected(null);
                $("#close-text-prop").attr("value", "Apply Text").button(
                        "refresh");
                $("#sample-text").attr("disabled", false);
                $("#text-prop-popup").show();

                $("#close-text-prop").unbind().bind("click", function() {
                    $(".popup").hide();
                    collage.drawText();
                });
            } else {
                $("#close-text-prop").attr("value", "Save & Close").button(
                        "refresh");
                $("#sample-text").attr("disabled", true);
                $("#text-prop-popup").show();

                $("#close-text-prop").unbind().bind("click", function() {
                    $(".popup").hide();
                });
            }
        }
    };
}();