/*******************************************************************************
 * Module responsible for managing file system, allows to find and open an image
 * which you want to add to your collage
 *
 * @author Zaneta Szymanska <<ahref='mailto:z.szymanska@samsung.com'>z.szymanska@samsung.com</a>>
 *
 * Copyright (c) 2012 Samsung Electronics All Rights Reserved.
 ******************************************************************************/
"use strict";
var mediaList = function() {
    // the path to the current location
    var src = [];
    var lockedFolders = [ 'ringtones' ];

    /**
     * Error callback function
     *
     * @param {WebAPIError} e Generic error interface
     */
    var onError = function(e) {
        tlib.logger.err("Error: " + e.message);
    };

    /**
     * Click event handler
     *
     * @param {String} id Id of selected file
     * @param {Array} files The files resulting from the asynchronous call
     */
    var clickHandle = function(id, files) {
        if (files[id].isDirectory === false) {
            var uri = files[id].toURI();
            if ((uri.indexOf(".jpg") !== -1) || (uri.indexOf(".png") !== -1)) {
                $.mobile.changePage("#mainPage");
                $("#choosenFile").removeAttr("src").attr("src", uri);
                $("#choose-file-ok").attr("disabled", false).button("refresh");
            } else {
                var msg = "<h2>Type Error</h2>You can not add this type of file! Only '.jpg' and '.png' files are allowed.";
                $('#info-popup>p').html(msg);
                $('#info-popup').show();
            }
        } else {
            src[src.length] = files[id].name;
            goInto();
        }
    };

    /**
     * File Array Success Callback for listFiles() method
     *
     * @param {Array} files The files resulting from the asynchronous call
     */
    var createList = function(files) {
        $('#fileList').empty();

        if (files.length > 0) {
            var str = "";

            for ( var i = 0; i < files.length; i++) {
                str += '<li data-id="' + i + '" >' + files[i].name + '</li>';
            }

            var elem = $(str).on("click", function() {
                clickHandle($(this).data("id"), files);
            });

            $("#fileList").append(elem);
        }
    };

    /**
     * File Success Callback function
     *
     * @param {File} dir Resolved directory
     */
    var onResolveSuccess = function(dir) {
        dir.listFiles(createList, onError);
    };

    /**
     * Resolves specified directory
     */
    var goInto = function() {
        $("#navbar").html("Media > " + src.join(" > "));
        tizen.filesystem.resolve(src.join("/"), onResolveSuccess, onError, 'r');
    };

    var upLevel = function() {
        src.pop();

        if (src.length !== 0) {
            goInto();
        } else {
            homeView();
        }
    };

    /**
     * Displays all available storages on the device
     *
     * @param {Array} storages List of the available storages
     */
    var storagesList = function(storages) {
        var i, str = "", list = [];

        $('#fileList').empty();

        for (i = 1; i < storages.length; i++) {
            if (storages[i].label !== '' && storages[i].type === 'INTERNAL'
                    && storages[i].label.substring(0, 4) !== 'wgt-'
                    && $.inArray(storages[i].label, lockedFolders) === -1) {
                list[list.length] = storages[i];
            }
        }

        for (i = 0; i < list.length; i++) {
            str += '<li data-id="' + i + '" >' + list[i].label + '</li>';
        }

        var elem = $(str).on("click", function() {
            src[src.length] = list[$(this).data("id")].label;
            goInto();
        });

        $("#fileList").append(elem);
    };

    /**
     * Retrieves the list of the available storages on the device
     */
    var homeView = function() {
        $("#navbar").html("Media");
        // Search for the storages
        tizen.filesystem.listStorages(storagesList);
    };

    return {
        /**
         * Initializes the Media view
         */
        init : function() {
            $('#homeBtn').on('click', function() {
                src = [];
                homeView();
            });

            $('#levelUpBtn').on('click', function() {
                upLevel();
            });

            $("#closeBtn").on("click", function() {
                $.mobile.changePage("#mainPage");
            });

            $('#close-info-popup').on("click", function() {
                $('#info-popup').hide();
            });
        },

        /**
         * Sets page's state before showing
         */
        pageBeforeShow : function() {
            $('#fileList').empty();
            src = [];
            homeView();
        }
    };
}();