/*******************************************************************************
 * Module responsible for showing and managing the main view
 *
 * @author Zaneta Szymanska <<ahref='mailto:z.szymanska@samsung.com'>z.szymanska@samsung.com</a>>
 *
 * Copyright (c) 2012 Samsung Electronics All Rights Reserved.
 ******************************************************************************/
"use strict";
var collage = function() {
    // Raphael paper, every Raphael element is drawn on it
    var paper;
    // DOM element, parent for Raphael drawing surface
    var container;
    // currently selected Raphael element
    var selectedItem = null;
    // array collecting Raphael elements
    var elements = [];

    /**
     * Replaces elements' locations, set properties and refreshes glow for image
     * elements
     *
     * @param {Number} up Id of element which was moving above elements[down]
     * @param {Number} down Id of element which was moving under elements[up]
     */
    var orderElements = function(up, down) {
        var tmp = elements[up];
        elements[up] = elements[down];
        elements[down] = tmp;

        if (elements[down].type === "image") {
            elements[down].refreshGlow();
        }

        elements[up].pos--;

        if (elements[up].type === "image") {
            elements[up].refreshGlow();
        }

        elements[down].pos++;
    };

    /**
     * Inserts element above the next element (due to order in elements[] array)
     *
     * @param {Number} up Id of element which should be moving up
     */
    var upElement = function(up) {
        if (up !== (elements.length - 1)) {
            // If selected item isn't on the peak: selected item insert up
            elements[up].insertAfter(elements[up + 1]);
            orderElements(up, up + 1);
        } else {
            tlib.logger.warn("up is not permitted");
        }
    };

    /**
     * Inserts element under the previous element (due to order in elements[]
     * array)
     *
     * @param {Number} down Id of element which should be moving down
     */
    var downElement = function(down) {
        if (down !== 0) {
            // If selected item isn't on the bottom: selected item insert down
            elements[down].insertBefore(elements[down - 1]);
            orderElements(down - 1, down);
        } else {
            tlib.logger.warn("down is not permitted");
        }
    };

    /**
     * Hide visible popups
     */
    var hidePopups = function() {
        if ($(".popup").is(":visible")) {
            $(".popup").hide();
        }
    };

    /**
     * Removes selected element from the Raphael's paper
     */
    var removeElement = function() {
        elements.splice(selectedItem.pos, 1);
        transforms.removeSelection(selectedItem);
        if (selectedItem.type === "image") {
            selectedItem.removeGlow();
        }
        selectedItem.remove();
        collage.setItemSelected(null);
    };

    /**
     * Closes application if possible
     */
    var exit = function() {
        if (typeof tizen !== "undefined" && tizen.application) {
            if (confirm("Exit?")) {
                var app = tizen.application.getCurrentApplication();
                app.exit();
                tlib.logger.info("Application closed");
            }
        } else {
            alert("Not supported");
        }
    };

    /**
     * Opens properties' popups
     */
    var propertiesOpen = function() {
        switch (selectedItem.type) {
        case "text":
            if (!($("#text-prop-popup").is(":visible"))
                    || ($("#close-text-prop").attr("value") !== "Save & Close")) {
                hidePopups();
                messages.defineTextProp("update");
                messages.setTextPopup();
            } else {
                hidePopups();
            }
            break;

        case "image":
            if (!($("#image-prop-popup").is(":visible"))) {
                hidePopups();
                $("#image-prop-popup").show();
                messages.setImagePopup();
            } else {
                hidePopups();
            }
            break;

        default:
            break;
        }
    };

    /**
     * Creates file and saves created collage
     *
     * @param {String} content PNG URI which should be saved to the file
     */
    var createFile = function(content) {
        var newFile = new tlib.file({
            filename : filename,
            virtualRoot : "images",
            success : function() {
                newFile.write(content, {
                    success : function() {
                    },
                    error : function() {
                        tlib.logger.err("Text not written to the file");
                    }
                });
            },
            error : function() {
                tlib.logger.err("File not created");
            }
        });
    };

    /**
     * Converts the canvas to the PNG URI, which is saved to the file
     */
    var render = function() {
        var can = document.getElementById('canvas-result');

        try {
            var dataURL = can.toDataURL("image/png").replace(
                    'data:image/png;base64,', '').replace('data:,', '');
            if (dataURL === '') {
                tlib.logger.err("No image source");
            } else {
                if (filename === '') {
                    $("#filename-popup").show();

                    $("#filename-save").unbind().bind("click", function() {
                        filename = $("#filename-input").val() + ".png";
                        $(".popup").hide();
                        createFile(dataURL);
                    });
                } else {
                    createFile(dataURL);
                }
            }
        } catch (e) {
            tlib.logger.err('canvas.toDataUrl error: ' + e.message);
            alert("Data source error: " + e.message);
            return;
        }
    };

    /**
     * Saves collage to the file
     */
    var saveCollage = function() {
        if (collage.getSelectedItem() !== null) {
            collage.setItemSelected(null);
        }

        var innerSVG = container.innerHTML;
        // Fixes the SVG generated by Raphael
        var fixedSVG = svgfix(innerSVG);

        // Convert the SVG string to canvas
        canvg("canvas-result", fixedSVG, {
            renderCallback : function() {
                render();
            }
        });
    };

    /**
     * Checks if application is launched on Tizen device/emulator or Simulator
     *
     * @returns {Boolean} true - application is launched on device or emulator,
     *          false - on Simulator
     */
    var isItMobile = function() {
        return /Tizen/.test(navigator.userAgent);
    };

    var addText = function() {
        if (!($("#text-prop-popup").is(":visible"))
                || ($("#close-text-prop").attr("value") !== "Apply Text")) {
            hidePopups();
            $("#text-prop-popup").show();
            messages.defineTextProp("create");
        } else {
            hidePopups();
        }
    };

    var addImage = function() {
        if (isItMobile()) {
            // In case of using device or Emulator
            if (!($("#open-image-file").is(":visible"))) {
                hidePopups();
                $("#open-image-file").show();
            } else {
                hidePopups();
            }
        } else {
            // In case of using Web Simulator
            collage.addImage("./css/images/lena.jpg");
        }
    };

    var baskOption = function() {
        if (selectedItem) {
            removeElement();
        } else {
            tlib.logger.warn("No image is selected to remove");
        }
    };

    /**
     * Click event handler
     *
     * @param {String} option Id of the pressed button
     */
    var chooseMenu = function(option) {
        switch (option) {
        case "text":
            addText();
            break;

        case "image":
            addImage();
            break;

        case "properties":
            propertiesOpen();
            break;

        case "up":
            upElement(selectedItem.pos);
            break;

        case "down":
            downElement(selectedItem.pos);
            break;

        case "bask":
            baskOption();
            break;

        case "clear":
            paper.clear();
            filename = '';
            $('#filename-input').val("");
            break;

        case "save-collage":
            saveCollage();
            break;

        case "close":
            exit();
            break;

        default:
            break;
        }
    };

    return {
        /**
         * Provides initialization for the app
         */
        initialize : function() {
            // 1 - means "only errors and warnings"
            tlib.logger.changeLogLevel(1);

            // Prevents scrolling effect
            $('#mainPage').bind('touchmove', function(e) {
                e.preventDefault();
            });

            $('#mainPage>*, #options>*').on("click", function(e) {
                e.stopPropagation();
            });

            $('#mainPage, #options').on("click", function(e) {
                if (collage.getSelectedItem() !== null) {
                    collage.setItemSelected(null);
                }
                hidePopups();
            });

            messages.initialize();

            hidePopups();
            $('#info-popup').hide();
            $("#canvas-result").hide();

            mediaList.init();

            // Creates Raphael paper
            container = document.getElementById('container');

            paper = new Raphael(container, container.clientWidth,
                    container.clientHeight);

            $("#container").on("click", function(e) {
                e.stopPropagation();
                e.preventDefault();

                var x0 = e.pageX, y0 = e.pageY;
                var elem = paper.getElementByPoint(x0, y0);

                if (elem === null) {
                    if (collage.getSelectedItem() !== null) {
                        collage.setItemSelected(null);
                    }
                    hidePopups();
                }
            });

            $(".option-button").on("click", function(e) {
                e.stopPropagation();
                chooseMenu(this.id);
            });

            $("#choose-file").on("click", function() {
                mediaList.pageBeforeShow();
                $.mobile.changePage("#fileMgr");
            });

            $("#choose-file-ok").on("click", function() {
                collage.addImage($("#choosenFile").attr("src"));
                $("#choosenFile").removeAttr("src");
                $("#choose-file-ok").attr("disabled", true).button("refresh");
                $("#open-image-file").hide();
            });
        },

        /**
         * Set element as selected to operate on it
         *
         * @param {Object} that Element, which should be set as selected
         */
        setItemSelected : function(that) {
            if (selectedItem !== null && selectedItem !== that) {
                transforms.removeSelection(selectedItem);
            }

            selectedItem = that;

            if (that) {
                transforms.setPoints(that);

                $("#properties").attr("disabled", false).button("refresh");
                $("#up").attr("disabled", false).button("refresh");
                $("#down").attr("disabled", false).button("refresh");
                $("#bask").attr("disabled", false).button("refresh");

                if ($("#text-prop-popup").is(":visible")) {
                    if (selectedItem.type === "text") {
                        messages.setTextPopup();
                    } else {
                        hidePopups();
                        messages.setImagePopup();
                        $("#image-prop-popup").show();
                    }
                } else {
                    if ($("#image-prop-popup").is(":visible")) {
                        if (selectedItem.type === "image") {
                            messages.setImagePopup();
                        } else {
                            hidePopups();
                            messages.setTextPopup();
                            $("#text-prop-popup").show();
                        }
                    }
                }
            } else {
                $("#properties").attr("disabled", true).button("refresh");
                $("#up").attr("disabled", true).button("refresh");
                $("#down").attr("disabled", true).button("refresh");
                $("#bask").attr("disabled", true).button("refresh");
            }
        },

        /**
         * Gets the selected element
         *
         * @returns {Object} selectedItem Selected Raphael Element
         */
        getSelectedItem : function() {
            return selectedItem;
        },

        /**
         * Embeds image element on Raphael paper
         *
         * @param {String} uri URI to the image which you want to add
         */
        addImage : function(uri) {
            var img = new Image();
            img.onload = function() {
                if (this.width > this.height) {
                    var scaledWidth = 200 / this.height * this.width;
                    var scaledHeight = 200;
                } else {
                    var scaledWidth = 200;
                    var scaledHeight = 200 / this.width * this.height;
                }

                var imgObj = new ImageObject(paper, uri, scaledWidth,
                        scaledHeight);

                imgObj.element.pos = elements.length;
                elements[elements.length] = imgObj.element;
            };
            img.src = uri;
        },

        /**
         * Draws text element on the Raphael paper
         */
        drawText : function() {
            var t = new TextObject(paper);

            t.element.pos = elements.length;
            elements[elements.length] = t.element;
        }
    };
}();