/*******************************************************************************
 * Module responsible for adding transformation to the element on
 * move/scale/rotate operation It contains event handlers for drag start, stop
 * and moving
 *
 * @author Zaneta Szymanska <<ahref='mailto:z.szymanska@samsung.com'>z.szymanska@samsung.com</a>>
 *
 * Copyright (c) 2012 Samsung Electronics All Rights Reserved.
 ******************************************************************************/
"use strict";
var transforms = function() {
    var that;

    /** Scaling */
    var scaleStart = function() {
        that.osx = that.s[0];
        that.osy = that.s[1];

        that.removeGlow();

        removePoints(true, false, false, true);
    };

    var scaleMove = function(dx, dy) {
        that.s[0] = that.osx - 2 * dx / that.scaledWidth;
        that.s[1] = that.osy - 2 * dy / that.scaledHeight;

        applyTransforms(that);
    };

    var scaleStop = function() {
        that.scaledWidth = that.attr("width") * that.s[0];
        that.scaledHeight = that.attr("height") * that.s[1];

        addPoints();

        that.restoreGlow();
    };

    /** Rotating */
    var rotateStart = function() {
        that.or = that.r;

        if (that.type === "image") {
            that.removeGlow();
        }

        removePoints(false, false, true, true);
    };

    var rotateStop = function() {
        if (that.type === "image") {
            addScalePoint();
        } else {
            addTranslatePoint();
        }

        if (that.type === "image") {
            that.restoreGlow();
        }
    };

    /** Text translations */
    var textMove = function(dx, dy) {
        that.translationPoint.attr("x", that.translationPoint.odx + dx);
        that.translationPoint.attr("y", that.translationPoint.ody + dy);

        that.t[0] = that.odx + dx;
        that.t[1] = that.ody + dy;

        applyTransforms(that);
    };

    var textStart = function() {
        that.odx = that.t[0];
        that.ody = that.t[1];

        that.translationPoint.odx = that.translationPoint.attr("x");
        that.translationPoint.ody = that.translationPoint.attr("y");

        removePoints(true, true, true, false);
    };

    var textStop = function() {
        that.translated.x = that.attr("x") + that.t[0];
        that.translated.y = that.attr("y") + that.t[1];

        addPoints();
    };

    /**
     * Adding rotate/move/translate points
     */

    var addRotatePoints = function() {
        // Remove previous rotate and center point
        removePoints(true, true, false, false);

        // Center point coordinates
        var centerPointAttr = {
            x : that.translated.x + that.attr("width") / 2,
            y : that.translated.y + that.attr("height") / 2
        };

        // Draw center point
        that.centrePoint = that.paper.circle(centerPointAttr.x,
                centerPointAttr.y, 3).attr({
            "fill" : "#fff",
            "opacity" : 0.4
        });

        // Rotate point coordinates
        if (that.scaledWidth) {
            var rotatePointAttr = {
                x : centerPointAttr.x - that.scaledWidth / 2 - 30,
                y : centerPointAttr.y
            };
        } else {
            var rotatePointAttr = {
                x : centerPointAttr.x - 150,
                y : centerPointAttr.y
            };
        }

        // Draw rotate point
        that.rotatePoint = that.paper.circle(rotatePointAttr.x,
                rotatePointAttr.y, 25).attr({
            "fill" : "#0f0",
            "opacity" : 0.4
        });

        // On rotatePoint drag
        that.rotatePoint.drag(function(dx, dy) {
            that.r = that.or
                    + Raphael.angle(centerPointAttr.x, centerPointAttr.y,
                            rotatePointAttr.x + dx, rotatePointAttr.y + dy);

            if (that.r > 360) {
                that.r -= 360;
            }

            applyTransforms(that);
        }, rotateStart, rotateStop);
    };

    var addScalePoint = function() {
        removePoints(false, false, true, false);

        var scalePointAttr = {
            x : that.translated.x - (that.scaledWidth - that.attr("width")) / 2,
            y : that.translated.y - (that.scaledHeight - that.attr("height"))
                    / 2
        };

        that.scalePoint = that.paper.circle(scalePointAttr.x, scalePointAttr.y,
                25).attr({
            "fill" : "#00f",
            "opacity" : 0.4
        });

        that.scalePoint.drag(scaleMove, scaleStart, scaleStop);
    };

    var addTranslatePoint = function() {
        removePoints(false, false, false, true);

        var translationPointAttr = {
            x : that.translated.x,
            y : that.translated.y + that.attr("font-size") / 2 + 10
        };

        that.translationPoint = that.paper.rect(translationPointAttr.x,
                translationPointAttr.y, 40, 40).attr({
            "fill" : "#00f",
            "opacity" : 0.4
        });

        that.translationPoint.drag(textMove, textStart, textStop);
    };

    /** Points adding and removing */
    var removePoints = function(rot, centre, scale, trans) {
        if (rot && that.rotatePoint && (that.rotatePoint !== null)) {
                that.rotatePoint.remove();
        }

        if (centre && that.centrePoint && (that.centrePoint !== null)) {
                that.centrePoint.remove();
        }

        if (scale && that.scalePoint && (that.scalePoint !== null)) {
                that.scalePoint.remove();
        }

        if (trans && that.translationPoint && (that.translationPoint !== null)) {
                that.translationPoint.remove();
        }
    };

    var addPoints = function() {
        addRotatePoints();

        if (that.type === "image") {
            // for images
            addScalePoint();
        } else {
            // for texts
            addTranslatePoint();
        }
    };

    /**
     * Adds specified transformation to the element
     *
     * @param that {Object} Element to which the transformation is added
     */
    var applyTransforms = function(that) {
        var str = "t" + that.t[0] + "," + that.t[1] + "r" + that.r;

        if (that.type === "image") {
            str += "s" + that.s[0] + "," + that.s[1];
        }

        that.transform(str);
    };

    return {
        /**
         * Event handler for drag start
         */
        start : function() {
            that = this;

            that.odx = that.t[0];
            that.ody = that.t[1];

            collage.setItemSelected(that);

            removePoints(true, true, true, false);

            that.removeGlow();
        },

        /**
         * Event handler for moving
         *
         * @param dx {Number} shift by x axis from the start point
         * @param dy {Number} shift by y axis from the start point
         */
        move : function(dx, dy) {
            that.t[0] = that.odx + dx;
            that.t[1] = that.ody + dy;

            applyTransforms(that);
        },

        /**
         * Event handler for drag end
         */
        up : function() {
            that.translated.x = that.attr("x") + that.t[0];
            that.translated.y = that.attr("y") + that.t[1];

            addPoints();
            that.restoreGlow();
        },

        /**
         * Adds rotation/scaling/moving points to selected element
         *
         * @param {Object} item Element, to which the selection points should be
         *        added
         */
        setPoints : function(item) {
            that = item;
            addPoints();
        },

        /**
         * Removes rotation/scaling/moving points from the element
         *
         * @param {Object} item Element, from which the selection should be
         *        removed
         * @returns
         */
        removeSelection : function(item) {
            that = item;
            removePoints(true, true, true, true);
        }
    };
}();