/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

using Xamarin.Forms;
using Clock.Styles;
using Xamarin.Forms.PlatformConfiguration.TizenSpecific;
using Image = Xamarin.Forms.Image;
using Label = Xamarin.Forms.Label;
using Switch = Xamarin.Forms.Switch;

namespace Clock.Alarm
{
    /// <summary>
    /// This class defines list cell for Alarm.
    /// This inherits from ViewCell which provides custom UI for a ListView
    /// It is commonly used for any cell for this ListView
    /// </summary>
    public class AlarmListCell : ViewCell
    {
        /// <summary>
        /// alarm item layout to layout ui components
        /// </summary>
        public RelativeLayout alarmItemLayout;

        /// <summary>
        /// time label for the alarm list
        /// </summary>
        public Label timeLabel;

        /// <summary>
        /// AM or PM label for alarm time
        /// </summary>
        public Label amPmLabel;

        /// <summary>
        /// Weekdays (M T W Th F S S) label
        /// </summary>
        public Label weekDaysLabel;

        /// <summary>
        /// Alarm name by users
        /// </summary>
        public Label alamNameLabel;

        /// <summary>
        /// Snooze switch object
        /// </summary>
        public Switch switchObj;

        /// <summary>
        /// Image to indicate repeat of this alarm
        /// </summary>
        public Image repeatImage;


        /// <summary>
        /// Draws alarm list 
        /// </summary>
        /// <returns>Returns RelativeLayout</returns>
        private RelativeLayout Draw()
        {
            int repeatOffset = 0;

            /// Need to get bindable context to assign list value
            AlarmRecord alarmData = (AlarmRecord)BindingContext;

            /// If binding context is null, can't proceed further action
            if (alarmData == null)
            {
                return null;
            }

            /// Gets the alarm state of the binding object
            AlarmStates state = alarmData.AlarmState;

            /// Alarm item layout should be set if null
            if (alarmItemLayout == null)
            {
                alarmItemLayout = new RelativeLayout
                {
                    HeightRequest = 22 + 93 + 29,
                };
                /// if state is active set the style accordingly
                if (state == Clock.Alarm.AlarmStates.Active)
                {
                    /// Style set for time label for normal case
                    timeLabel = new Label { Style = AlarmStyle.ATO001 };
                    /// Style set for apm / pm label for normal case
                    amPmLabel = new Label { Style = AlarmStyle.ATO002 };
                    /// Style set for weekday label for normal case
                    weekDaysLabel = new Label
                    {
                        Style = AlarmStyle.ATO004
                    };
                    /// Style set for alarm name label for normal case
                    alamNameLabel = new Label { Style = AlarmStyle.ATO003 };
                }
                else if (state == Clock.Alarm.AlarmStates.Inactive)
                {
                    /// Style set for time label for disabled case
                    timeLabel = new Label { Style = AlarmStyle.ATO001D };
                    /// Style set for apm / pm label for disabled case
                    amPmLabel = new Label { Style = AlarmStyle.ATO002D };
                    /// Style set for weekday label for disabled case
                    weekDaysLabel = new Label
                    {
                        Style = AlarmStyle.ATO004D
                    };
                    /// Style set for alarm name label for disabled case
                    alamNameLabel = new Label { Style = AlarmStyle.ATO003D };
                }

                /// For time label, to meet thin attribute for font, need to use custom feature
                timeLabel.On<Xamarin.Forms.PlatformConfiguration.Tizen>().SetFontWeight(FontWeight.Light);
                /// For am pm label, to meet To meet thin attribute for font, need to use custom feature
                amPmLabel.On<Xamarin.Forms.PlatformConfiguration.Tizen>().SetFontWeight(FontWeight.Light);
                /// For weekdays label, to meet To meet thin attribute for font, need to use custom feature
                weekDaysLabel.On<Xamarin.Forms.PlatformConfiguration.Tizen>().SetFontWeight(FontWeight.Normal);
                /// For alarm name label, to meet To meet thin attribute for font, need to use custom feature
                alamNameLabel.On<Xamarin.Forms.PlatformConfiguration.Tizen>().SetFontWeight(FontWeight.Normal);
                /// Needs to set binding context for scheduled time
                timeLabel.SetBinding(Label.TextProperty, "ScheduledTimeText");
                /// Add to layout
                alarmItemLayout.Children.Add(timeLabel,
                    Constraint.RelativeToParent((parent) =>
                    {
                        return 32;
                    }),
                    Constraint.RelativeToParent((parent) =>
                    {
                        return 22;
                    }));
                /// Needs  to set binding context for am pm label
                amPmLabel.SetBinding(Label.TextProperty, "ScheduledAmPmText");
                alarmItemLayout.Children.Add(amPmLabel,
                    Constraint.RelativeToView(timeLabel, (parent, sibling) => sibling.X + sibling.Width + 10),
                    Constraint.RelativeToView(timeLabel, (parent, sibling) => sibling.Y + 36));

                /// Gets formatted string based on alarm state
                if (alarmData.AlarmState == AlarmStates.Active)
                {
                    alarmData.WeekdayRepeatText = alarmData.GetFormatted(alarmData.WeekFlag, true);
                }

                /// Gets formatted string based on alarm state
                else if (alarmData.AlarmState == AlarmStates.Inactive)
                {
                    alarmData.WeekdayRepeatText = alarmData.GetFormatted(alarmData.WeekFlag, false);
                }

                /// Sets binding context for weekdays label
                weekDaysLabel.SetBinding(Label.FormattedTextProperty, new Binding("WeekdayRepeatText", BindingMode.Default));
                alarmItemLayout.Children.Add(weekDaysLabel,
                    Constraint.RelativeToParent((parent) => (720 - 104 - 32 - 268)),
                    Constraint.RelativeToParent((parent) => (22 + 93) - 43));

                /// repeat icon
                if (alarmData != null && alarmData.Repeat)
                {
                    repeatImage = new Image
                    {
                        Source = "alarm/clock_ic_repeat.png",
                        WidthRequest = 38,
                        HeightRequest = 38,
                    };

                    //if (state == (int)Clock.Alarm.AlarmStates.Active)
                    //    repeatImage.BackgroundColor = Color.FromHex("B3000000"); // TODO: Doublecheck color
                    //else
                    //    repeatImage.BackgroundColor = Color.FromHex("FF000000"); // TODO: Doublecheck color

                    alarmItemLayout.Children.Add(repeatImage,
                        Constraint.RelativeToParent((parent) => (720 - 104 - 32 - 268)),
                        Constraint.RelativeToParent((parent) => (22 + 93) - (43 + 43))); // need to check again

                    repeatOffset = 43;
                }

                /// Sets binding context for alarm name label
                alamNameLabel.SetBinding(Label.TextProperty, "AlarmName");

                /// Adds to relative layout
                alarmItemLayout.Children.Add(alamNameLabel,
                    Constraint.RelativeToParent((parent) => (720 - 104 - 32 - 268 + repeatOffset)),
                    Constraint.RelativeToParent((parent) => (22 + 93 - (43 + 43))));

                /// Switch object to indicate snooze or not
                switchObj = new Switch
                {
                    HeightRequest = 72,
                    WidthRequest = 72,
                };

                /// Sets IsToggled based on alarm state
                if (alarmData.AlarmState == AlarmStates.Active)
                {
                    switchObj.IsToggled = true;
                }
                else if (alarmData.AlarmState == AlarmStates.Inactive)
                {
                    switchObj.IsToggled = false;
                }
                else
                {
                    /// Snooze case should be handled properly
                }

                /// Adds to relative layout
                alarmItemLayout.Children.Add(switchObj,
                    Constraint.RelativeToParent((parent) =>
                    {
                        return (720 - 104);
                    }),
                    Constraint.RelativeToParent((parent) =>
                    {
                        return (22 + 93 - 72);
                    }));

                /// Adds an event 
                switchObj.Toggled += (s, e) =>
                {
                    Switch sObj = s as Switch;
                    /// Needs valid parent to proceed
                    if (sObj.Parent == null || sObj.Parent.Parent == null)
                    {
                        return;
                    }

                    /// Need binding context to check state
                    AlarmRecord am = (AlarmRecord)((AlarmListCell)sObj.Parent.Parent).BindingContext;
                    if (am == null)
                    {
                        return;
                    }

                    /// Modify state and re-draw it. Redraw must be called to redraw
                    am.AlarmState = e.Value ? AlarmStates.Active : AlarmStates.Inactive;
                    Draw();

                    AlarmRecord returnedRecord = null;
                    string alarmUniqueIdentifier = am.GetUniqueIdentifier();
                    /// In case of snooze change, it should be reflected to AlarmReord.AlarmRecordDictionary
                    if (AlarmRecord.AlarmRecordDictionary.TryGetValue(alarmUniqueIdentifier, out returnedRecord))
                    {
                        returnedRecord.AlarmState = am.AlarmState;
                        AlarmRecord.AlarmRecordDictionary[alarmUniqueIdentifier] = returnedRecord;
                        AlarmRecord.SaveDictionary(AlarmRecord.AlarmRecordDictionary);
                    }
                };
            }
            else
            {
                /// change state of model here 
                if (state == Clock.Alarm.AlarmStates.Inactive)
                {
                    /// change style of time to inactive
                    timeLabel.Style = AlarmStyle.ATO001D; 
                    /// change style of ampm label to inactive
                    amPmLabel.Style = AlarmStyle.ATO002D;
                    /// change style of alarm name label to inactive
                    alamNameLabel.Style = AlarmStyle.ATO003D;
                    /// change style of week days label to inactive
                    weekDaysLabel.Style = AlarmStyle.ATO004D;
                    /// get formatted week day label based on bitmap
                    weekDaysLabel.FormattedText = alarmData.GetFormatted(alarmData.WeekFlag, false);
                }
                else if (state == Clock.Alarm.AlarmStates.Active)
                {
                    /// change style of time to inactive
                    timeLabel.Style = AlarmStyle.ATO001;
                    /// change style of ampm label to inactive
                    amPmLabel.Style = AlarmStyle.ATO002;
                    /// change style of alarm name label to inactive
                    alamNameLabel.Style = AlarmStyle.ATO003;
                    /// change style of week days label to inactive
                    weekDaysLabel.Style = AlarmStyle.ATO004;
                    /// get formatted week day label based on bitmap
                    weekDaysLabel.FormattedText = alarmData.GetFormatted(alarmData.WeekFlag, true);
                }
            }

            // When draw() is called, alarmItemLayout is already on so just update LayoutAgain to refresh
            return alarmItemLayout; 
        }

        /// <summary>
        /// When binding context is changed, need to redraw
        /// </summary>
        protected override void OnBindingContextChanged()
        {
            base.OnBindingContextChanged();
            if (BindingContext == null)
            {
                return;
            }

            View = Draw();
        }
    }
}
