/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Collections.Specialized;

namespace Clock.Alarm
{
    /// <summary>
    /// This class defines custom ObservableCollection for AlarmRecord
    /// This compensates property update event trigger in addition to ObservableCollection 
    /// whihch only triggers event for add / remove / replace item in to the list
    /// </summary>
    public class AlarmExtendedObservableCollections<T> : ObservableCollection<T>
    {

        /// <summary>
        /// Constructor for this class
        /// </summary>
        public AlarmExtendedObservableCollections()
        {
        }

        /// <summary>
        /// Constructor for this class with enumerable list
        /// </summary>
        /// <param name="items">A collection of generic type</param>
        public AlarmExtendedObservableCollections(IEnumerable<T> items) : this()
        {
            /// Add each item into ObservableCollection
            foreach (var item in items)
            {
                this.Add(item);
            }
        }

        /// <summary>
        /// This method triggers collection change event notification
        /// </summary>
        /// <param name="item">Object of generic type</param>
        public void ReportItemChange(T item)
        {
            NotifyCollectionChangedEventArgs args =
                new NotifyCollectionChangedEventArgs(
                    NotifyCollectionChangedAction.Replace,
                    item,
                    item,
                    IndexOf(item));
            OnCollectionChanged(args);
        }
    }
}
