/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

using System;
using Xamarin.Forms;
using System.Collections.Generic;
using Clock.Controls;
using Tizen.Xamarin.Forms.Extension;
#if CROSS_PLATFORM
using Clock.Interfaces;
#endif

namespace Clock.Alarm
{
    /// <summary>
    /// This class defines Alarm Edit View page contents.
    /// This page contains table view for alarm info input.
    /// Each table cell is binded with sub-pages if necessary.
    /// </summary>
    public class AlarmEditPage : ContentPage
    {
        /// <summary>
        /// Main layout for this page.
        /// </summary>
        StackLayout mainLayout;
        /// <summary>
        /// This static field holds current alarm record which is being displayed for alarm info input.
        /// Through this record, edit view is automatically updated when users select certain option in sub pages.
        /// </summary>
        static internal AlarmRecord BindableAlarmRecord { get; set; }

        /// <summary>
        /// TableView used to hold each alarm info edit cell
        /// </summary>
        private TableView alarmEditTable;

        /// <summary>
        /// Title bar with two buttons to move on two pages (back, done for adding/updating alarm info)
        /// </summary>
        internal static AlarmTitleBar titleBar;

        /// <summary>
        /// Alarm time edit cell
        /// </summary>
        AlarmEditTableCell editCell;

        /// <summary>
        /// Alarm repeat (e.g. weekly, never, everyday) edit cell
        /// </summary>
        AlarmEditTableCell repeatCell;

        /// <summary>
        /// Alarm type (sound, vibration, etc)
        /// </summary>
        AlarmEditTableCell typeCell;

        /// <summary>
        /// Alarm sound (default, mp3, etc)
        /// </summary>
        AlarmEditTableCell soundCell;

        /// <summary>
        /// Alarm sound (default, mp3, etc)
        /// </summary>
        AlarmEditTableCell toneCell;

        /// <summary>
        /// Alarm sound (default, mp3, etc)
        /// </summary>
        AlarmEditTableCell snoozeCell;

        /// <summary>
        /// Alarm name edit cell
        /// </summary>
        AlarmEditTableCell nameCell;

        /// <summary>
        /// Table view root
        /// </summary>
        TableRoot root;


        /// <summary>
        /// Initializes a new instance of the <see cref="AlarmEditPage"/> class.
        /// </summary>
        /// <param name="alarmRecordData">AlarmRecord</param>
        public AlarmEditPage(AlarmRecord alarmRecordData)
        {
            //NewAlarmRecord = new AlarmRecord(alarmRecordData); // deep copy 
            BindableAlarmRecord = alarmRecordData;
            Draw();
        }

        /// <summary>
        /// Draws inital ui component for this page
        /// </summary>
        private void Draw()
        {
            try
            {
                if (titleBar == null)
                {
                    // Hide navigation bar (title and buttons)
                    NavigationPage.SetHasNavigationBar(this, false);
                    titleBar = new AlarmTitleBar();
                    titleBar.LeftButton.Text = "Cancel";
                    titleBar.RightButton.Text = "Done";
                    titleBar.TitleLabel.Text = "Edit";

                    // Check for a new record or editing existing one
                    if (BindableAlarmRecord.IsSerialized == false)
                    {
                        titleBar.TitleLabel.Text = "Create";
                    }
                    else
                    {
                        titleBar.TitleLabel.Text = "Edit";
                    }

                    // When button clicked, need to do followings:
                    // 1. check time set by users
                    // 2. convert TimeSpan to DateTime (since epoc time)
                    // 3. get alarm name
                    // 4. get alarm key (creation date which identify alarm uniquely)
                    // 5. save a new record or update existing alarm. 
                    titleBar.RightButton.Clicked += (s, e) =>
                    {
                        // Get date, name
                        TimeSpan ts = ((Controls.TizenTimePicker)editCell.View).Time;
                        // Should add epoc time to convert TimeSpan to DateTime. 
                        DateTime tmpDate = new DateTime(1970, 1, 1) + ts;
                        tmpDate = new DateTime(
                            tmpDate.Year,
                            tmpDate.Month,
                            tmpDate.Day,
                            tmpDate.Hour,
                            tmpDate.Minute,
                            0);
                        // Set alarm time
                        BindableAlarmRecord.ScheduledDateTime = tmpDate;
                        // Set alarm Name 
                        BindableAlarmRecord.AlarmName = ((AlarmNameRow)nameCell.View).mainEntry.Text;
                        // Use alarm created date for unique identifier for a alarm record
                        string alarmUniqueIdentifier = BindableAlarmRecord.GetUniqueIdentifier();

                        // if new record
                        if (BindableAlarmRecord.IsSerialized == false)
                        {
                            string checkAlarm;
                            // Check this BindableRecord is already available in AlarmRecordDictionary
                            if ((AlarmRecord.AlarmRecordDictionary == null) ||
                                IsAlarmTimeAlreadySet(BindableAlarmRecord.GetAlarmTimeKey(), out checkAlarm) == false)
                            {
                                // Create native alarm 
#if CROSS_PLATFORM
                                var fromDB = DependencyService.Get<IAlarm>().CreateAlarm(BindableAlarmRecord);
#else
                                var fromDB = Tizen.Impls.Alarm.GetInstance().CreateAlarm(BindableAlarmRecord);
#endif

                                if (fromDB != null)
                                {
                                    // decorate newAlarmRecord. IsSerialized is used to indicate whether new one or existing one
                                    BindableAlarmRecord.IsSerialized = true;
#if CROSS_PLATFORM
                                    BindableAlarmRecord.NativeAlarmID = DependencyService.Get<IAlarm>().GetAlarmID(fromDB);
#else
                                    BindableAlarmRecord.NativeAlarmID = Tizen.Impls.Alarm.GetInstance().GetAlarmID(fromDB);
#endif
                                    // update AlarmRecordDictionary
                                    // AlarmRecordDictionary can be null this point if no previous attempt to add one or 
                                    // no record available. This dictionary should be serialized so it can be retrieve when
                                    // newly launched
                                    if (AlarmRecord.AlarmRecordDictionary == null)
                                    {
                                        AlarmRecord.AlarmRecordDictionary = new Dictionary<string, AlarmRecord>();
                                    }
                                    // Alarm record creation date is the alarmKey
                                    AlarmRecord.AlarmRecordDictionary.Add(alarmUniqueIdentifier, BindableAlarmRecord);

                                    // Serialize (save this dicitonary to disk)
                                    AlarmRecord.SaveDictionary(AlarmRecord.AlarmRecordDictionary);

                                    // Update ListView (ItemsSource) 
                                    AlarmListPage ap = ((AlarmListPage)AlarmPageController.GetInstance(AlarmPages.ListPage));

                                    // ObservableAlarmList is ItemsSource for AlarmList view. This is bindable so this model 
                                    // should be changed to reflect record change (adding a new alarm)
                                    AlarmListUI.ObservableAlarmList.Add(BindableAlarmRecord);

                                    if (ap != null)
                                    {
                                    ap.Content = ap.alarmListUI; // [TODO] Come back later to move it to AlarmListPage's onAppearing
                                }
                            }
                            }
                            else
                            {
                                // If alarm time is already existing in alarm record dictionary, should not add
                                Toast.DisplayText("Alarm already exists. It cannot be added because it is duplicatd.", 3000);
                            }
                        }
                        else // Edit
                        {
                            // first cancel native alarm since updating
                            var obj = BindableAlarmRecord;
#if CROSS_PLATFORM
                            DependencyService.Get<IAlarm>().DeleteAlarm(ref obj);
#else
                            Tizen.Impls.Alarm.GetInstance().DeleteAlarm(ref obj);
#endif
                            BindableAlarmRecord = obj;
                            // Since it has been removed in the native alarm list, remove from alarm record dictionary also
                            AlarmRecord.AlarmRecordDictionary.Remove(BindableAlarmRecord.GetUniqueIdentifier());
                            string alarmKeyToUpdate;
                            bool showAlert = false;
                            // If updated alarm record to insert has time which already existing in the alarm record dictionary,
                            // then remove existing one and use this for this alarm time. Still can't allow two alarm with the same time.
                            if (IsAlarmTimeAlreadySet(BindableAlarmRecord.GetAlarmTimeKey(), out alarmKeyToUpdate) == true)
                            {
                                // previous alarm record should be remove from both AlarmListUI and Dictionary
                                AlarmRecord.AlarmRecordDictionary.Remove(alarmKeyToUpdate);
                                AlarmRecord alarmRecordToDelete = null;
                                foreach (var alarmItem in AlarmListUI.ObservableAlarmList)
                                {
                                    AlarmRecord alarm = alarmItem;
                                    if (alarm.GetUniqueIdentifier() == alarmKeyToUpdate) // find the same AlarmRecord in ObservableList
                                    {
                                        alarmRecordToDelete = alarm;
                                        break;
                                    }
                                }

                                if (alarmRecordToDelete != null)
                                {
                                    AlarmListUI.ObservableAlarmList.Remove(alarmRecordToDelete);
                                }

                                showAlert = true;
                            }

                            // Add this record to the alarm dictionary
                            AlarmRecord.AlarmRecordDictionary.Add(alarmUniqueIdentifier, BindableAlarmRecord);

                            // Since it is change of text of existing ObservableAlarmList item, update it
                            foreach (var alarmItem in AlarmListUI.ObservableAlarmList)
                            {
                                AlarmRecord alarm = alarmItem;
                                if (alarm.GetUniqueIdentifier() == BindableAlarmRecord.GetUniqueIdentifier()) // find the same AlarmRecord in ObservableList
                                {
                                    // Apply changed value to observable collection object
                                    alarm.DeepCopy(BindableAlarmRecord);
                                    break;
                                }
                            }

                            // Save this dictionary to file
                            AlarmRecord.SaveDictionary(AlarmRecord.AlarmRecordDictionary);
                            if (showAlert)
                            {
                                Toast.DisplayText("Alarm already exists. Updated existing alarm.");
                            }
                        }

                        Navigation.PopAsync();
                    };
                }

                // Create time edit cell
                if (editCell == null)
                {
                    editCell = new AlarmEditTableCell(AlarmEditTypes.Edit, this);
                }

                // Create repeat edit cell
                if (repeatCell == null)
                {
                    repeatCell = new AlarmEditTableCell(AlarmEditTypes.Repeat, this);
                }

                // Create alarm type cell
                if (typeCell == null)
                {
                    typeCell = new AlarmEditTableCell(AlarmEditTypes.Type, this);
                }

                // Create sound type cell
                if (soundCell == null)
                {
                    soundCell = new AlarmEditTableCell(AlarmEditTypes.Sound, this);
                }

                // Create tone selection cell
                if (toneCell == null)
                {
                    toneCell = new AlarmEditTableCell(AlarmEditTypes.Tone, this);
                }

                // Create snooze setting cell
                if (snoozeCell == null)
                {
                    snoozeCell = new AlarmEditTableCell(AlarmEditTypes.Snooze, this);
                }

                // Create name setting cell
                if (nameCell == null)
                {
                    nameCell = new AlarmEditTableCell(AlarmEditTypes.Name, this);
                }

                if (root == null)
                {
                    root = new TableRoot()
                    {
                        new TableSection()
                        {
                            editCell,
                            repeatCell,
                            typeCell,
                            soundCell,
                            toneCell,
                            snoozeCell,
                            nameCell,
                        }
                    };
                }

                if (alarmEditTable == null)
                {
                    alarmEditTable = new TableView(root);
                    alarmEditTable.HasUnevenRows = true;
                }

                if (mainLayout == null)
                {
                    mainLayout = new StackLayout
                    {
                        HorizontalOptions = LayoutOptions.FillAndExpand,
                        Children =
                        {
                            titleBar,
                            alarmEditTable
                        }
                    };
                }

                Content = mainLayout;
            }
            catch (Exception e)
            {
                System.Diagnostics.Debug.WriteLine(e.Message);
            }
        }

        /// <summary>
        /// Update the panel text info.
        /// </summary>
        /// <param name="v">The alarm time to check availability</param>
        /// <param name="key">The unique ID for the selected alarm</param>
        /// <returns>Returns true if alarm exists; otherwise, returns false.</returns>
        private bool IsAlarmTimeAlreadySet(string v, out string key)
        {
            key = "";
            foreach (var item in AlarmRecord.AlarmRecordDictionary)
            {
                AlarmRecord record = item.Value as AlarmRecord;
                if (record?.GetAlarmTimeKey() == v)
                {
                    key = item.Key;
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// This method is called when this page is showing up.
        /// </summary>
        /// <param name="o">The alarm record to show on this page. If a new one, default alarm record is passed.</param>
        internal void Update(AlarmRecord o)
        {
            BindableAlarmRecord = o;
            ((TizenTimePicker)(editCell.View)).Time = BindableAlarmRecord.ScheduledDateTime.TimeOfDay;
            ((AlarmEditRow)(repeatCell.View)).subLabel.BindingContext = BindableAlarmRecord;
            ((AlarmEditRow)(typeCell.View)).subLabel.BindingContext = BindableAlarmRecord;
            ((AlarmEditSoundRow)(soundCell.View)).slider.Value = BindableAlarmRecord.Volume;
            ((AlarmEditRow)(toneCell.View)).subLabel.BindingContext = BindableAlarmRecord;
            ((AlarmEditRow)(snoozeCell.View)).snoozeSwitch.IsToggled = BindableAlarmRecord.Snooze;
            ((AlarmNameRow)(nameCell.View)).mainEntry.Text = BindableAlarmRecord.AlarmName;
            if (BindableAlarmRecord.IsSerialized == false)
            {
                titleBar.TitleLabel.Text = "Create";
            }
            else
            {
                titleBar.TitleLabel.Text = "Edit";
            }
        }
    }
}
