/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

using Clock.Controls;
using System;
using System.Diagnostics;
using Xamarin.Forms;
using Xamarin.Forms.PlatformConfiguration.TizenSpecific;
using Label = Xamarin.Forms.Label;
using Image = Xamarin.Forms.Image;
using Button = Xamarin.Forms.Button;
using Clock.Styles;
#if CROSS_PLATFORM
#else
using Clock.Tizen;
#endif

namespace Clock.Worldclock
{
    /// <summary>
    /// The world clock page, the class is defined in 2 files
    /// One is for UI part, one is for logical process,
    /// This one is for UI part
    /// </summary>
    public partial class WorldclockPage : ContentPage
    {
        public StackLayout mainLayout;
        public RelativeLayout mapLayout;
        public WorldclockCityList cityList;

        const int DOT_IMAGE_SIZE = 7;
        const int RING_IMAGE_SIZE = 9;

        private Image[] dotImages;
        private Image[] ringImages;
        const int DOT_MAX_NUM = 8;

        private Image timezoneAreaImage;
        private Label timezoneGmtOffsetLabel;

        #region timezone.details
        private StackLayout timezoneDetailFirstLineLayout;
        private Label timeLabel, amLabel, relativeToLocalLabel;
        private Label citiesLabel;
        #endregion

        /// <summary>
        /// Create the worldclock's main view,
        /// It includes map, timezone detail information, listview and one float button
        /// </summary>
        /// <returns> The StackLayout. </returns>
        public StackLayout CreateWorldClockPage()
        {
            CreateMapArea();
            cityList = new WorldclockCityList();

            mainLayout = new StackLayout
            {
                HorizontalOptions = LayoutOptions.FillAndExpand,
                VerticalOptions = LayoutOptions.FillAndExpand,
                Spacing = 0,
                //BackgroundColor = Color.Red,
                Children =
                {
                    mapLayout,
                    cityList.emptyListAreaLayout,
                    cityList.cityListUI,
                }
            };

            OnMapViewUpdateRequest();

            App myApp = Application.Current as App;
            foreach (Location l in myApp.ClockInfo.userLocations)
            {
                cityList.AppendItemToCustomList(l);
            }

            myApp.ClockInfo.UpdateEmptyListBackground();
            return mainLayout;
        }

        /// <summary>
        /// Shows a floating button when the worldclock page is appearing
        /// </summary>
        protected override void OnAppearing()
        {
            ((App)Application.Current).ShowFloatingButton(Title);
        }

        /// <summary>
        /// Hides a floating button when the worldclock page is disappearing
        /// </summary>
        protected override void OnDisappearing()
        {
            ((App)Application.Current).HideFloatingButton(Title);
        }

        /// <summary>
        /// Creates image of world map and contents of timezone detail
        /// </summary>
        private void CreateMapArea()
        {
            CreateWorldMap();
            CreateTimezoenDetails();
        }

        /// <summary>
        /// Creates dot images for displaying city locations
        /// </summary>
        private void CreateDots()
        {
            try
            {
                dotImages = new Image[DOT_MAX_NUM];
                ringImages = new Image[DOT_MAX_NUM];
                for (int i = 0; i < DOT_MAX_NUM; i++)
                {
                    dotImages[i] = new Image();
                    dotImages[i].Source = "worldclock/clock_world_location_dot_b43.png";
                    dotImages[i].IsVisible = false;
                    mapLayout.Children.Add(dotImages[i],
                                  Constraint.RelativeToParent((parent) => { return 0; }),
                                  Constraint.RelativeToParent((parent) => { return 17; }));
                    ringImages[i] = new Image();
                    ringImages[i].Source = "worldclock/clock_world_location_ring.png";
                    ringImages[i].IsVisible = false;
                    mapLayout.Children.Add(ringImages[i],
                                  Constraint.RelativeToParent((parent) => { return 0; }),
                                  Constraint.RelativeToParent((parent) => { return 17; }));
                }
            }
            catch (Exception ex)
            {
                Debug.WriteLine("[CreateDots] Exception Occurs : " + ex.Message);
            }
        }

        /// <summary>
        /// Creates area of world map
        /// </summary>
        private void CreateWorldMap()
        {
            mapLayout = new RelativeLayout
            {
                HeightRequest = 17 + 406 + 50 + 36 + 48 + 10 + 51 + 61, /*679*/
                HorizontalOptions = LayoutOptions.FillAndExpand,
                VerticalOptions = LayoutOptions.Start,
                //BackgroundColor = Color.FromHex("#99dd00"),
            };

            Image worldMapImage = new Image
            {
                Source = "worldclock/clock_world_map_01.png",
                //BackgroundColor = Color.FromHex("#14CC88"),
            };
            mapLayout.Children.Add(worldMapImage,
              Constraint.RelativeToParent((parent) => { return 0; }),
              Constraint.RelativeToParent((parent) => { return 17; }));

            Image gtmAreaImage = new Image
            {
                Source = "worldclock/clock_world_gmt_0_AO009.png",
                IsVisible = false,
            };
            mapLayout.Children.Add(gtmAreaImage,
              Constraint.RelativeToParent((parent) => { return 0; }),
              Constraint.RelativeToParent((parent) => { return 17; }));

            Button arrowLeftImage = new Button
            {
                Image = "worldclock/clock_icon_world_clock_arrow_left.png",
                WidthRequest = 56,
                HeightRequest = 80,
                BackgroundColor = Color.FromHex("#00000000"),
                //BackgroundColor = Color.Firebrick,
            };

            mapLayout.Children.Add(arrowLeftImage,
              Constraint.RelativeToParent((parent) => { return 10; }),
              Constraint.RelativeToParent((parent) => { return 17 + 196; }));
            arrowLeftImage.Clicked += ArrowLeftImage_Clicked;

            Button arrowRightImage = new Button
            {
                Image = "worldclock/clock_icon_world_clock_arrow_right.png",
                WidthRequest = 56,
                HeightRequest = 80,
                BackgroundColor = Color.FromHex("#00000000"),
            };

            mapLayout.Children.Add(arrowRightImage,
                          Constraint.RelativeToParent((parent) => { return 720 - 56 - 10; }),
                          Constraint.RelativeToParent((parent) => { return 17 + 196; }));
            arrowRightImage.Clicked += ArrowRightImage_Clicked;

            Image nightanddayLineImage = new Image
            {
                Source = "worldclock/clock_world_nightandday_line.png",
                IsVisible = false,
            };
            mapLayout.Children.Add(nightanddayLineImage,
              Constraint.RelativeToParent((parent) => { return 0; }),
              Constraint.RelativeToParent((parent) => { return 17 + 109; }));

            Image nightanddayMaskImage = new Image
            {
                Source = "worldclock/clock_world_nightandday_mask.png",
                IsVisible = false,
            };
            mapLayout.Children.Add(nightanddayMaskImage,
              Constraint.RelativeToParent((parent) => { return 0; }),
              Constraint.RelativeToParent((parent) => { return 17 + 109; }));

            timezoneAreaImage = new Image
            {
                Source = "worldclock/clock_world_gmt_area_AO009_.png",
                Aspect = Aspect.Fill,
                WidthRequest = 15,
                HeightRequest = 320,
            };

            mapLayout.Children.Add(timezoneAreaImage,
              Constraint.RelativeToParent((parent) => { return 0; }),
              Constraint.RelativeToParent((parent) => { return 17 + 67; }));

            CreateDots();

            // textblock {timezone_offset}
            timezoneGmtOffsetLabel = new Label
            {
                FontSize = CommonStyle.GetDp(40),
                Text = "?",
                TextColor = Color.FromHex("#fafafa"),
                WidthRequest = 40 + 15 + 40,
                HeightRequest = 60 - 5,
                //BackgroundColor = Color.FromHex("#aa77cb"),
                HorizontalTextAlignment = TextAlignment.Center,
            };
            mapLayout.Children.Add(timezoneGmtOffsetLabel,
                    Constraint.RelativeToView(timezoneAreaImage, (parent, sibling) => { return sibling.X - 40; }),
                    Constraint.RelativeToView(timezoneAreaImage, (parent, sibling) => { return sibling.Y - 60; }));
        }

        /// <summary>
        /// Called when right arrow button has been clicked
        /// It updates Map view including time zone details according to GMT variation(current + 1)
        /// </summary>
        /// <param name="sender">arrowRightImage Button object</param>
        /// <seealso cref="System.object">
        /// <param name="e">Event argument for event of Button</param>
        /// <seealso cref="System.EventArgs">
        private void ArrowRightImage_Clicked(object sender, EventArgs e)
        {
            info.MoveCurrentTimezone(Direction.RIGHT);
            UpdateMapAndTimezoneDetails(info.GetCurrentTimezone());
        }

        /// <summary>
        /// Called when left arrow button has been clicked
        /// It updates Map view including time zone details according to GMT variation(current - 1)
        /// </summary>
        /// <param name="sender">arrowLeftImage Button object</param>
        /// <seealso cref="System.object">
        /// <param name="e">Event argument for event of Button</param>
        /// <seealso cref="System.EventArgs">
        private void ArrowLeftImage_Clicked(object sender, EventArgs e)
        {
            info.MoveCurrentTimezone(Direction.LEFT);
            UpdateMapAndTimezoneDetails(info.GetCurrentTimezone());
        }

        /// <summary>
        /// Updates map view and timezone details based on current timezone information
        /// </summary>
        public void OnMapViewUpdateRequest()
        {
            UpdateMapAndTimezoneDetails(info.GetCurrentTimezone());
        }

        /// <summary>
        /// Updates timezone details based on the given timezone
        /// </summary>
        /// <param name="tz">The <see cref="Timezone"/> timezone for updating map view and timezone details </param>
        private void UpdateMapAndTimezoneDetails(Timezone tz)
        {
            UpdateTimezoneDots(tz);
            UpdateTimezoneArea(tz);
            UpdateGmtOffset(tz);
            UpdateTimezoneRelativeToLocal(tz.gmtOffset);
            UpdateTimezoneCitiesList(tz);
        }

        /// <summary>
        /// Draw dot and ring images at cities in the current timezone
        /// </summary>
        /// <param name="tz">The <see cref="Timezone"/> timezone to set for map </param>
        private void UpdateTimezoneDots(Timezone tz)
        {
            try
            {
                for (int i = 0; i < tz.places.Length; i++)
                {
                    dotImages[i].TranslationX = (tz.places[i].x - DOT_IMAGE_SIZE / 2);
                    ringImages[i].TranslationX = (tz.places[i].x - RING_IMAGE_SIZE / 2);
                    dotImages[i].TranslationY = (tz.places[i].y - DOT_IMAGE_SIZE / 2);
                    ringImages[i].TranslationY = (tz.places[i].y - RING_IMAGE_SIZE / 2);
                    dotImages[i].IsVisible = true;
                    ringImages[i].IsVisible = true;
                }

                for (int i = tz.places.Length; i < DOT_MAX_NUM; i++)
                {
                    dotImages[i].IsVisible = false;
                    ringImages[i].IsVisible = false;
                }
            }
            catch (Exception ex)
            {
                Debug.WriteLine("[UpdateTimezoneDots] Exception : " + ex.ToString() + ", " + ex.Message);
            }
        }

        /// <summary>
        /// Mark the current timezone area
        /// Image : timezoneAreaImage
        /// </summary>
        /// <param name="tz">The <see cref="Timezone"/> timezone to set for map </param>
        private void UpdateTimezoneArea(Timezone tz)
        {
            timezoneAreaImage.TranslationX = tz.xCoord + (tz.zoneWidth - timezoneAreaImage.Width) / 2;
            timezoneGmtOffsetLabel.TranslationX = tz.xCoord + (tz.zoneWidth - timezoneAreaImage.Width) / 2;
            timezoneAreaImage.IsVisible = true;
        }

        private string OffsetToString(int offset)
        {
            int abGmtOffset = Math.Abs(offset);
            int remainder = (abGmtOffset % 60) * 100 / 60;
            int remainder_str = (remainder % 10 != 0) ? remainder : remainder / 10;

            var updatedGmtOffset = String.Format("{0}{1}{2}{3}",
                (offset < 0) ? "" : "+",
                offset / 60,
                ((offset % 60 != 0) ? "." : ""),
                (remainder != 0) ? remainder_str.ToString() : "");

            return updatedGmtOffset;
        }

        /// <summary>
        /// Update current GMT offset information
        /// Label : timezoneGmtOffsetLabel
        /// </summary>
        /// <param name="tz">The <see cref="Timezone"/> timezone to set for map </param>
        private void UpdateGmtOffset(Timezone tz)
        {
            int gmtOffset = tz.gmtOffset;
            string updatedGmtOffset = OffsetToString(gmtOffset);
            timezoneGmtOffsetLabel.Text = updatedGmtOffset;
        }

        private void UpdateTimezoneRelativeToLocal(int offset)
        {
            App currentApp = Application.Current as App;
            CityRecord cityRecord = currentApp.ClockInfo.GetCityRecord(offset);
            timeLabel.Text = cityRecord.CityTime;
            amLabel.Text = cityRecord.CityAmPm;
            relativeToLocalLabel.Text = cityRecord.RelativeToLocalCountry;
        }

        private void UpdateTimezoneCitiesList(Timezone tz)
        {
            string cities = tz.places[0].name;
            for (int i = 1; i < tz.places.Length; i++)
            {
                cities += ", " + tz.places[i].name;
            }

            citiesLabel.Text = cities;
        }

        private void CreateTimezoenDetails()
        {
            timeLabel = new Label
            {
                Text = "7:50",
                Style = Styles.WorldclockStyle.ATO017,
                WidthRequest = 164,
                HeightRequest = 94,
                HorizontalOptions = LayoutOptions.Center,
                //BackgroundColor = Color.Coral,
            };
            timeLabel.On<Xamarin.Forms.PlatformConfiguration.Tizen>().SetFontWeight(FontWeight.Thin);

            amLabel = new Label
            {
                Text = "a.m.",
                Style = Styles.WorldclockStyle.ATO018,
                WidthRequest = 70,
                HeightRequest = 48,
                HorizontalOptions = LayoutOptions.Center,
                //BackgroundColor = Color.BurlyWood,
                Margin = new Thickness(12, 36, 20, 10),
            };
            amLabel.On<Xamarin.Forms.PlatformConfiguration.Tizen>().SetFontWeight(FontWeight.Normal);

            relativeToLocalLabel = new Label
            {
                Text = "1h behind",
                Style = Styles.WorldclockStyle.ATO018,
                HorizontalOptions = LayoutOptions.Center,
                HeightRequest = 48,
                //BackgroundColor = Color.Pink,
                Margin = new Thickness(0, 36, 0, 10),
            };
            relativeToLocalLabel.On<Xamarin.Forms.PlatformConfiguration.Tizen>().SetFontWeight(FontWeight.Normal);

            timezoneDetailFirstLineLayout = new StackLayout
            {
                Orientation = StackOrientation.Horizontal,
                Spacing = 0,
                HeightRequest = 94,
                HorizontalOptions = LayoutOptions.Center,
                VerticalOptions = LayoutOptions.Center,
                //BackgroundColor = Color.FromHex("#22dd22"),
                Children =
                {
                    timeLabel,
                    amLabel,
                    relativeToLocalLabel
                }
            };
            StackLayout tmpStack = new StackLayout
            {
                Orientation = StackOrientation.Vertical,
                Spacing = 0,
                HeightRequest = 94,
                WidthRequest = 720,
                HorizontalOptions = LayoutOptions.Center,
                VerticalOptions = LayoutOptions.Center,
                //BackgroundColor = Color.FromHex("#aa448b"),
                Children =
                {
                    timezoneDetailFirstLineLayout
                }
            };
            mapLayout.Children.Add(tmpStack,
              Constraint.RelativeToParent((parent) => { return 0; }),
              Constraint.RelativeToParent((parent) => { return 17 + 406 + 50; }));

            citiesLabel = new Label
            {
                Text = "Beijing, Denpasar, Guangzhou, Haikou",
                Style = Styles.WorldclockStyle.ATO016,
                WidthRequest = 720 - 16 - 16,
                HeightRequest = 51,
                LineBreakMode = LineBreakMode.TailTruncation,
                //BackgroundColor = Color.Blue,
            };
            citiesLabel.On<Xamarin.Forms.PlatformConfiguration.Tizen>().SetFontWeight(FontWeight.Normal);

            mapLayout.Children.Add(citiesLabel,
              Constraint.RelativeToParent((parent) => { return 16; }),
              Constraint.RelativeToParent((parent) => { return 17 + 406 + 50 + 94; }));
        }
    }
}
