/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

using System.Collections.Generic;
using System.Collections.Specialized;
using System.ComponentModel;

namespace Clock.Worldclock
{
    /// <summary>
    /// The CityRecord class
    /// </summary>
    public class CityRecord : INotifyPropertyChanged
    {
        public string CityTime { get; set; }
        public string CityAmPm { get; set; }
        public string CityDate { get; set; }
        public string Cities { get; set; }
        public int Offset { get; set; }
        public string RelativeToLocalCountry { get; set; }

        public static CityRecordObservableCollection cityRecordList = new CityRecordObservableCollection();

        ///<summary>
        ///Event that is raised when the properties of CityRecord change
        ///</summary>
        public event PropertyChangedEventHandler PropertyChanged;
 
        public CityRecord()
        {
        }

        protected virtual void OnPropertyChanged(string propertyName)
        {
            if (PropertyChanged != null)
            {
                PropertyChanged(this,
                    new PropertyChangedEventArgs(propertyName));
            }
        }
    }

    /// <summary>
    /// The CityRecordObservableCollection class which represents a data collection that provides notifications when items get added, removed.
    /// </summary>
    public class CityRecordObservableCollection : List<CityRecord>, INotifyCollectionChanged
    {
        public CityRecordObservableCollection()
        {
        }

        /// <summary>
        /// The event that notifies dynamic changes, such as when items get added or removed
        /// </summary>
        public event NotifyCollectionChangedEventHandler CollectionChanged;

        /// <summary>
        /// Adds CityRecord object to a list
        /// </summary>
        /// <param name="item">CityRecord object to add</param>
        /// <seealso cref="CityRecord">
        public new void Add(CityRecord item)
        {
            base.Add(item);
            CollectionChanged?.Invoke(this,
                new NotifyCollectionChangedEventArgs(NotifyCollectionChangedAction.Add, item));
        }

        /// <summary>
        /// Inserts a collection of CityRecords in the list at the specified index.
        /// </summary>
        /// <param name="index">The zero-based index at which the new elements should be inserted.</param>
        /// <param name="collection">The collection of alarm records whose elements should be inserted into the CityRecordObservableCollection</param>
        public new void InsertRange(int index, IEnumerable<CityRecord> collection)
        {
            base.InsertRange(index, collection);
            CollectionChanged?.Invoke(this,
                new NotifyCollectionChangedEventArgs(NotifyCollectionChangedAction.Add, new List<CityRecord>(collection), index));
        }

        /// <summary>
        /// Removes the specified CityRecord object from the list.
        /// </summary>
        /// <param name="item">CityRecord object to remove from the CityRecordObservableCollection</param>
        /// <seealso cref="CityRecord">
        public new void Remove(CityRecord item)
        {
            base.Remove(item);
            CollectionChanged?.Invoke(this,
                new NotifyCollectionChangedEventArgs(NotifyCollectionChangedAction.Remove, item));
        }

        /// <summary>
        /// Removes all elements from the list.
        /// </summary>
        public new void Clear()
        {
            base.Clear();
            CollectionChanged?.Invoke(this,
                new NotifyCollectionChangedEventArgs(NotifyCollectionChangedAction.Reset));
        }
    }
}
