/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

using Clock.Alarm;
using Clock.Common;
#if CROSS_PLATFORM
using Clock.Interfaces;
#endif
using Native = Tizen.Applications;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using Tizen.Multimedia;
using Tizen.System;

#if CROSS_PLATFORM
[assembly: Xamarin.Forms.Dependency(typeof(Alarm))]
#endif
namespace Clock.Tizen.Impls
{
    /// <summary>
    /// The Alarm class
    /// </summary>
#if CROSS_PLATFORM
    class Alarm : IAlarm
#else
    class Alarm
#endif
    {
        AudioStreamPolicy audioStreamPolicy;
        Feedback feedback;
        Player player;
        AlarmRecord _bindableRecord;
#if CROSS_PLATFORM
        public Alarm()
        {
        }
#else
        static Alarm singletonObj;

        private Alarm()
        {
        }

        public static Alarm GetInstance()
        {
            return singletonObj ?? (singletonObj = new Impls.Alarm());
        }
#endif

        /// <summary>
        /// Gets system alarm ID from alarm object
        /// </summary>
        /// <param name="alarm">AlarmRecord object to get system alarm ID from</param>
        /// <seealso cref="AlarmRecord">
        /// <returns> Returns system alarm ID </returns>
        public int GetAlarmID(object alarm)
        {
            Native.Alarm nativeAlarm = alarm as Native.Alarm;
            return nativeAlarm.AlarmId;
        }

        private Native.AlarmWeekFlag Convert(AlarmWeekFlag flag)
        {
            Native.AlarmWeekFlag ret = Native.AlarmWeekFlag.AllDays;
            switch (flag)
            {
                case AlarmWeekFlag.Monday:
                    ret = Native.AlarmWeekFlag.Monday;
                    break;
                case AlarmWeekFlag.Tuesday:
                    ret = Native.AlarmWeekFlag.Tuesday;
                    break;
                case AlarmWeekFlag.Wednesday:
                    ret = Native.AlarmWeekFlag.Wednesday;
                    break;
                case AlarmWeekFlag.Thursday:
                    ret = Native.AlarmWeekFlag.Thursday;
                    break;
                case AlarmWeekFlag.Friday:
                    ret = Native.AlarmWeekFlag.Friday;
                    break;
                case AlarmWeekFlag.WeekDays:
                    ret = Native.AlarmWeekFlag.WeekDays;
                    break;
                case AlarmWeekFlag.AllDays:
                    ret = Native.AlarmWeekFlag.AllDays;
                    break;
                default:
                    break;
            }

            return ret;
        }

        /// <summary>
        /// Creates a system alarm of AlarmRecord
        /// </summary>
        /// <param name="binableAlarmRecord">AlarmRecord object to create a system alarm based on</param>
        /// <seealso cref="AlarmRecord">
        /// <returns> Returns native alarm object </returns>
        public object CreateAlarm(AlarmRecord bindableAlarmRecord)
        {
            Native.AppControl appControl = new Native.AppControl()
            {
                ApplicationId = Native.Application.Current.ApplicationInfo.ApplicationId
            };
            appControl.ExtraData.Add("AlarmRecord.UniqueIdentifier", bindableAlarmRecord.GetUniqueIdentifier());
            appControl.ExtraData.Add("RingType", RingType.RING_TYPE_ALARM.ToString());
            Native.AlarmWeekFlag nativeFlag = Convert(bindableAlarmRecord.WeekFlag);
            return Native.AlarmManager.CreateAlarm(bindableAlarmRecord.ScheduledDateTime, nativeFlag, appControl);
        }

        /// <summary>
        /// Cancel a system alarm of AlarmReco
        /// </summary>
        /// <param name="binableAlarmRecord">AlarmRecord object relevant to a system alarm to cancel</param>
        /// <seealso cref="AlarmRecord">
        public void DeleteAlarm(ref AlarmRecord bindableAlarmRecord)
        {
            var allAlarms = Native.AlarmManager.GetAllSceduledAlarms();
            foreach (Native.Alarm item in allAlarms)
            {
                if (item.AlarmId == bindableAlarmRecord.NativeAlarmID)
                {
                    item.Cancel();
                    Native.AppControl appControl = new Native.AppControl()
                    {
                        ApplicationId = Native.Application.Current.ApplicationInfo.ApplicationId
                    };
                    appControl.ExtraData.Add("AlarmRecord.UniqueIdentifier", bindableAlarmRecord.GetUniqueIdentifier());
                    appControl.ExtraData.Add("RingType", RingType.RING_TYPE_ALARM.ToString());
                    var fromDB = Native.AlarmManager.CreateAlarm(bindableAlarmRecord.ScheduledDateTime, Convert(bindableAlarmRecord.WeekFlag), appControl);
                    bindableAlarmRecord.NativeAlarmID = fromDB.AlarmId;

                    break;
                }
            }
        }

        /// <summary>
        /// Activate to ring
        /// </summary>
        /// <param name="delay">The duration of the alarm delay period before an alarm will activate</param>
        /// <seealso cref="int">
        /// <param name="type">RingType for ringing</param>
        /// <seealso cref="RingType">
        public void ActivateAlarm(int delay, RingType type)
        {
            Native.AppControl appControl = new Native.AppControl();
            appControl.ApplicationId = Native.Application.Current.ApplicationInfo.ApplicationId;
            appControl.ExtraData.Add("RingType", type.ToString());
            Native.AlarmManager.CreateAlarm(delay, appControl);
        }

        /// <summary>
        /// Cancel to ring
        /// </summary>
        public void DeactivateAlarm()
        {
            IEnumerable<Native.Alarm> alarmList = Native.AlarmManager.GetAllSceduledAlarms();

            foreach (Native.Alarm alarm in alarmList)
            {
                Native.AppControl appControl = alarm.AlarmAppControl;
                try
                {
                    string type = (string)appControl.ExtraData.Get("RingType");
                    if (type == RingType.RING_TYPE_TIMER.ToString())
                    {
                        alarm.Cancel();
                    }
                }
                catch (Exception e)
                {
                    Debug.WriteLine("[DeactivateAlarm] Exception - Message:{0}", e.Message);
                }
            }
        }

        /// <summary>
        /// This method launches another tone setting app and set proper type based on selection.
        /// </summary>
        /// <param name="alarmRecord">AlarmRecord object</param>
        public void LaunchAlarmToneAppControl(AlarmRecord alarmRecord)
        {
            try
            {
                /// Sets app id
                Native.AppControl appControl = new Native.AppControl()
                {
                    ApplicationId = "org.tizen.setting-ringtone"
                };
                /// Sets operation for app control
                appControl.Operation = "http://tizen.org/appcontrol/operation/pick";
                /// Adds selection mode
                appControl.ExtraData.Add("http://tizen.org/appcontrol/data/selection_mode", "single");
                /// Prepares tone resources
                List<string> toneResources = new List<string>()
                {
                    Native.Application.Current.DirectoryInfo.SharedResource + "ringtones/alarm.mp3",
                    Native.Application.Current.DirectoryInfo.SharedResource + "ringtones",
                };
                /// Sets launch mode
                appControl.LaunchMode = Native.AppControlLaunchMode.Group;
                /// Sets tone resource collections
                appControl.ExtraData.Add("http://tizen.org/appcontrol/data/path", toneResources);
                /// Sets alarm record to update based on app control result
                _bindableRecord = alarmRecord;
                /// Requests launch
                Native.AppControl.SendLaunchRequest(appControl, ReplyAfterLaunching);
            }
            catch (Exception e)
            {
                Debug.WriteLine("[LaunchAlarmToneAppControl] Exception - Message:{0}", e.Message);
            }
        }

        private void ReplyAfterLaunching(Native.AppControl launchRequest, Native.AppControl replyRequest, Native.AppControlReplyResult result)
        {
            //_bindableRecord.AlarmToneType = AlarmToneTypes.RingtoneSdk;
            if (result == Native.AppControlReplyResult.Succeeded)
            {
                List<string> data = (List<string>)replyRequest.ExtraData.Get("http://tizen.org/appcontrol/data/selected");
                if (data.Count == 1)
                {
                    Debug.WriteLine("[LaunchAlarmToneAppControl] Type {0}", data[0]);
                    switch (data[0])
                    {
                        case "/opt/usr/data/settings/Ringtones/rintone_sdk.mp3":
                            _bindableRecord.AlarmToneType = AlarmToneTypes.RingtoneSdk;
                            break;
                        default:
                            break;
                    }
                }
            }
        }

#if CROSS_PLATFORM
        void IAlarm.DeactivateAlarm(AlarmRecord alarmRecord)
#else
        public void DeactivateAlarm(AlarmRecord alarmRecord)
#endif
        {
            // Deserialize AlarmRecordDictionary
            if (AlarmRecord.AlarmRecordDictionary == null)
            {
#if CROSS_PLATFORM
                IAlarmPersistentHandler serizliazer = DependencyService.Get<IAlarmPersistentHandler>();
                AlarmRecord.AlarmRecordDictionary = serizliazer.DeserializeAlarmRecord();
                // Need to retrieve at the page creation time
#else
                AlarmRecord.AlarmRecordDictionary = AlarmPersistentHandler.GetInstance().DeserializeAlarmRecord();
#endif
            }

            AlarmRecord alarmToUpdate = AlarmRecord.AlarmRecordDictionary[alarmRecord.GetUniqueIdentifier()];
            if (alarmToUpdate != null)
            {
                AlarmRecord.AlarmRecordDictionary.Remove(alarmRecord.GetUniqueIdentifier());
                /// Need to implement Snooze when deactivating alarm
            }

            AlarmRecord.AlarmRecordDictionary.Add(alarmRecord.GetUniqueIdentifier(), alarmToUpdate);

            foreach (var alarmItem in AlarmListUI.ObservableAlarmList)
            {
                AlarmRecord alarm = alarmItem;
                if (alarm.GetUniqueIdentifier() == alarmToUpdate.GetUniqueIdentifier()) // find the same AlarmRecord in ObservableList
                {
                    // Apply changed value to observable collection object
                    alarm.DeepCopy(alarmToUpdate);
                    break;
                }
            }

            AlarmRecord.SaveDictionary(AlarmRecord.AlarmRecordDictionary);
        }

        /// <summary>
        /// Starts to vibrate
        /// </summary>
        public void StartVibration()
        {
            feedback = new Feedback();
            if (feedback.IsSupportedPattern(FeedbackType.Vibration, "Timer"))
            {
                feedback.Play(FeedbackType.Vibration, "Timer");
            }
        }

        /// <summary>
        /// Stops to vibrate
        /// </summary>
        public void StopVibration()
        {
            if (feedback != null)
            {
                feedback.Stop();
            }
        }

        /// <summary>
        /// Asynchronously plays ringing sound
        /// </summary>
        async public void PlaySound()
        {
            if (player == null)
            {
                audioStreamPolicy = new AudioStreamPolicy(AudioStreamType.Alarm);
                audioStreamPolicy.AcquireFocus(AudioStreamFocusOptions.Playback, AudioStreamBehavior.None, null);

                player = new Player();
                MediaUriSource soudSource = new MediaUriSource(SystemSettings.IncomingCallRingtone);
                player.SetSource(soudSource);

                player.ApplyAudioStreamPolicy(audioStreamPolicy);
                await player.PrepareAsync();
                player.IsLooping = true;
                player.Volume = 1;
                if (player.State == PlayerState.Ready)
                {
                    player.Start();
                }
            }
        }

        /// <summary>
        /// Asynchronously plays ringing sound
        /// </summary>
        /// <param name="targetVolume">The volume level at playing</param>
        /// <param name="toneTypes">AlarmToneTypes to play</param>
        async public void PlaySound(float targetVolume, AlarmToneTypes toneTypes)
        {
            audioStreamPolicy = new AudioStreamPolicy(AudioStreamType.Alarm);
            audioStreamPolicy.AcquireFocus(AudioStreamFocusOptions.Playback, AudioStreamBehavior.None, null);

            player = new Player();
            MediaUriSource soudSource = new MediaUriSource(SystemSettings.IncomingCallRingtone);
            player.SetSource(soudSource);

            player.ApplyAudioStreamPolicy(audioStreamPolicy);
            await player.PrepareAsync();
            player.IsLooping = true;
            player.Volume = targetVolume;
            if (player.State == PlayerState.Ready)
            {
                player.Start();
            }
        }

        /// <summary>
        /// Asynchronously stops ringing
        /// </summary>
        public void StopSound()
        {
            if (player != null)
            {
                player.Stop();
                player.Unprepare();
                audioStreamPolicy.ReleaseFocus(AudioStreamFocusOptions.Playback, AudioStreamBehavior.None, null);
                player.Dispose();
                player = null;
            }

            if (audioStreamPolicy != null)
            {
                audioStreamPolicy.Dispose();
            }
        }
    }
}
