/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
//#define NATIVE_TONE_AVALIABLE
using System.Windows.Input;
using Xamarin.Forms;
using Clock.Controls;
using Clock.Interfaces;

namespace Clock.Alarm
{
    /// <summary>
    /// This class defines common edit table cell
    /// All rows in this edit table use this table cell
    /// </summary>
    public class AlarmEditTableCell : ViewCell
    {

        /// <summary>
        /// This property is used for sending command (page push)
        /// </summary>
        public static readonly BindableProperty CommandProperty = BindableProperty.Create("Command", typeof(ICommand), typeof(AlarmEditTableCell));

        /// <summary>
        /// To distinguish edit type to have different row UIs based on this type
        /// </summary>
        AlarmEditTypes Type {  get; set; }

        /// <summary>
        /// The containing page of this table
        /// </summary>
        ContentPage Container { get; set; }

        /// <summary>
        /// Internal AlarmRecord to keep AlarmRecord temporarily
        /// </summary>
        internal AlarmRecord AlarmRecordData { get; set; }

        /// <summary>
        /// Command to be used for page redirection
        /// </summary>
        public ICommand Command
        {
            get { return (ICommand)GetValue(CommandProperty); }
            set { SetValue(CommandProperty, value); }
        }

        /// <summary>
        /// AlarmEditTable contstructor to build UI controls in this table cell
        /// </summary>
        /// <param name="type">Type to decide UI controls in this table cell</param>
        /// <seealso cref="AlarmEditTypes">
        /// <param name="page">The containing page for the table cell</param>
        /// <seealso cref="ContentPage">
        public AlarmEditTableCell(AlarmEditTypes type, ContentPage page)
        {
            Type = type;
            Container = page;

            switch (type)
            {
                case AlarmEditTypes.Edit:
                    View = new TizenTimePicker(AlarmEditPage.BindableAlarmRecord);
                break;

                case AlarmEditTypes.Sound: // Need to enum type to string
                    View = new AlarmEditSoundRow();
                    break;

                case AlarmEditTypes.Repeat: //alarmRecordData.Repeat // Need to bitwise operation to string representation
                case AlarmEditTypes.Type: // Need to enum type to string
                case AlarmEditTypes.Tone: // Need to enum type to string
                case AlarmEditTypes.Snooze: // Need to convert bool to string
                    View = new AlarmEditRow(type);
                    break;
                case AlarmEditTypes.Name: // Need to convert bool to string
                    View = new AlarmNameRow(type);
                    break;
                default:
                break;
            }

            /// add tap gesture
            this.Tapped += (s, e) =>
            {
                if (Command == null) // Come up with setting only one time
                {
                    AlarmEditTableCell atc = (AlarmEditTableCell)s;
                    switch (atc.Type)
                    {
                        case AlarmEditTypes.Repeat:
                        Command = new Command(async () =>
                        {
                            await page.Navigation.PushAsync(AlarmPageController.GetInstance(AlarmPages.RepeatPage, atc.AlarmRecordData));
                        });
                        break;

                        case AlarmEditTypes.Type:
                        Command = new Command(async () =>
                        {
                            await page.Navigation.PushAsync(AlarmPageController.GetInstance(AlarmPages.TypePage, atc.AlarmRecordData));
                        });
                        break;

                        /// Alarm tone should be controlled by another app.
                        /// However, in preview 2 release emulator, there is no such app so this compilation condition
                        /// should be only enabled when the tone setting app is available.
                        case AlarmEditTypes.Tone:
#if NATIVE_TONE_AVALIABLE
#if CROSS_PLATFORM
                            DependencyService.Get<IAlarm>().LaunchAlarmToneAppControl(AlarmEditPage.BindableAlarmRecord);
#else
                            Clock.Tizen.Impls.Alarm.GetInstance().LaunchAlarmToneAppControl(AlarmEditPage.BindableAlarmRecord);
#endif
#else
                            Command = new Command(async () =>
                            {
                                await page.Navigation.PushAsync(AlarmPageController.GetInstance(AlarmPages.TonePage, atc.AlarmRecordData));
                            });
#endif
                            break;

                        default:
                        break;
                    }
                }

                /// invoke command only it is assigned
                if (Command != null)
                {
                    Command.Execute(null);
                }
            };
        }
    }
}
