/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

using Clock.Styles;
using Xamarin.Forms;
using Xamarin.Forms.PlatformConfiguration.TizenSpecific;
using Page = Xamarin.Forms.Page;
using Label = Xamarin.Forms.Label;
using Switch = Xamarin.Forms.Switch;

namespace Clock.Alarm
{
    /// <summary>
    /// This class defines Alarm Edit row for Alarm Edit View page contents.
    /// This row is commonly used for many types of row cell. 
    /// </summary>
    class AlarmEditRow : RelativeLayout
    {
        /// <summary>
        /// Main label (bigger)
        /// </summary>
        internal Label mainLabel;

        /// <summary>
        /// Sub label (smaller)
        /// </summary>
        internal Label subLabel;

        /// <summary>
        /// Label for multiline
        /// </summary>
        internal Label multi2Label;

        /// <summary>
        /// Snooze switch
        /// </summary>
        internal Switch snoozeSwitch;

        /// <summary>
        /// Page to redirect
        /// </summary>
        public Page MoveTo {  get; set; }

        /// <summary>
        /// Gesture for tap action
        /// </summary>
        private TapGestureRecognizer tapAction;

        /// <summary>
        /// Type to decide row contents
        /// </summary>
        private AlarmEditTypes type;

        public AlarmEditRow(AlarmEditTypes editType)
        {
            this.type = editType;
            HeightRequest = 144;
            VerticalOptions = LayoutOptions.Start;
            tapAction = new TapGestureRecognizer();
            tapAction.Tapped += (s, e) =>
            {
                if (true)
                {
                    AlarmEditRow t = (AlarmEditRow)s;
                    if (t.MoveTo != null)
                    {
                        Navigation.PushAsync(MoveTo, true);
                    }
                }
            };
            GestureRecognizers.Add(tapAction);
            if (editType != AlarmEditTypes.Snooze)
            {
                if (mainLabel == null)
                {
                    mainLabel = new Label
                    {
                        HeightRequest = 54,
                        Style = AlarmStyle.T023,
                    };
                    mainLabel.On<Xamarin.Forms.PlatformConfiguration.Tizen>().SetFontWeight(FontWeight.Light);
                }

                switch (editType)
                {
                    case AlarmEditTypes.Repeat:
                        mainLabel.Text = "Repeat weekly";
                        break;
                    case AlarmEditTypes.Type:
                        mainLabel.Text = "Alarm type";
                        break;
                    case AlarmEditTypes.Tone:
                        mainLabel.Text = "Alarm tone";
                        break;
                    default:
                        mainLabel.Text = "";
                        break;
                }

                Children.Add(mainLabel,
                    Constraint.RelativeToParent((parent) => { return 32; }),
                    Constraint.RelativeToParent((parent) => { return (144 - (54 + 43)) / 2; }));
                if (subLabel == null)
                {
                    subLabel = new Label
                    {
                        HeightRequest = 43,
                        Style = AlarmStyle.T024,
                    };
                    subLabel.On<Xamarin.Forms.PlatformConfiguration.Tizen>().SetFontWeight(FontWeight.Normal);
                }

                switch (editType)
                {
                    case AlarmEditTypes.Repeat:
                    subLabel.SetBinding(Label.TextProperty, new Binding("WeekFlag", mode: BindingMode.OneWay, converter: new AlarmValueConverter()));
                    subLabel.BindingContext = AlarmEditPage.BindableAlarmRecord;
                    break;
                    case AlarmEditTypes.Type:
                    subLabel.SetBinding(Label.TextProperty, new Binding("AlarmType", mode: BindingMode.OneWay, converter: new AlarmValueConverter()));
                    subLabel.BindingContext = AlarmEditPage.BindableAlarmRecord;
                    break;
                    case AlarmEditTypes.Tone:
                    subLabel.SetBinding(Label.TextProperty, new Binding("AlarmToneType", mode: BindingMode.OneWay, converter: new AlarmValueConverter()));
                    subLabel.BindingContext = AlarmEditPage.BindableAlarmRecord;
                    break;
                    default:
                    subLabel.Text = "";
                    break;
                }

                Children.Add(subLabel,
                    Constraint.RelativeToView(mainLabel, (parent, sibling) => { return sibling.X; }),
                    Constraint.RelativeToView(mainLabel, (parent, sibling) => { return sibling.Y + sibling.Height; }));
            }
            else
            {
                if (mainLabel == null)
                {
                    mainLabel = new Label
                    {
                        HeightRequest = 54,
                        Style = AlarmStyle.T023,
                    };
                    mainLabel.On<Xamarin.Forms.PlatformConfiguration.Tizen>().SetFontWeight(FontWeight.Light);
                }

                mainLabel.Text = "Snooze";
                Children.Add(mainLabel,
                    Constraint.RelativeToParent((parent) => { return 32; }),
                    Constraint.RelativeToParent((parent) => { return (144 - (54 + 43 + 43)) / 2; }));
                subLabel = new Label
                {
                    HeightRequest = 43,
                    Text = "Sound the alarm 3 times, at 5-mintue",
                    Style = AlarmStyle.T024,
                };
                subLabel.On<Xamarin.Forms.PlatformConfiguration.Tizen>().SetFontWeight(FontWeight.Normal);

                Children.Add(subLabel,
                    Constraint.RelativeToView(mainLabel, (parent, sibling) => { return sibling.X; }),
                    Constraint.RelativeToView(mainLabel, (parent, sibling) => { return sibling.Y + sibling.Height; }));
                multi2Label = new Label
                {
                    HeightRequest = 43,
                    Text = "intervals.",
                    Style = AlarmStyle.T024,
                };
                multi2Label.On<Xamarin.Forms.PlatformConfiguration.Tizen>().SetFontWeight(FontWeight.Normal);
                Children.Add(multi2Label,
                    Constraint.RelativeToView(subLabel, (parent, sibling) => { return sibling.X; }),
                    Constraint.RelativeToView(subLabel, (parent, sibling) => { return sibling.Y + sibling.Height; }));

                snoozeSwitch = new Switch
                {
                    HeightRequest = 72,
                    WidthRequest = 72,
                };
                snoozeSwitch.IsToggled = AlarmEditPage.BindableAlarmRecord.Snooze;
                snoozeSwitch.Toggled += (s, e) =>
                {
                    AlarmEditPage.BindableAlarmRecord.Snooze = e.Value;
                };
                Children.Add(snoozeSwitch,
                    Constraint.RelativeToParent((parent) => (parent.X + parent.Width - (104))),
                    Constraint.RelativeToParent((parent) => parent.Y + ((parent.Y + 22 + 93 + 29) - 72) / 2));
            }
        }
    }
}
