/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

using Clock.Interfaces;
#if CROSS_PLATFORM
#else
using Clock.Tizen;
#endif
using System;
using System.Collections.Generic;
using System.Globalization;
using Tizen.Xamarin.Forms.Extension;
using Xamarin.Forms;

namespace Clock.Worldclock
{
    public enum Direction
    {
        LEFT,
        RIGHT,
    }

    public struct Timezone
    {
        public int xCoord;
        public int zoneWidth;
        public int gmtOffset;
        public Location[] places;

        public Timezone(int _xCoord, int _zoneWidth, int _gmtOffset, Location[] _places)
        {
            xCoord = _xCoord;
            zoneWidth = _zoneWidth;
            gmtOffset = _gmtOffset;
            places = _places;
        }
    }

    public struct Location
    {
        public string name;
        public string country;
        public int gmtOffset;
        public int x;
        public int y;
        public bool summerTime;
        public string tzpath;
        public Location(string _name, string _country, int _gmtOffset, int _x, int _y, bool _summerTime = false, string _tzpath = null)
        {
            name = _name;
            country = _country;
            gmtOffset = _gmtOffset;
            x = _x;
            y = _y;
            summerTime = _summerTime;
            tzpath = _tzpath;
        }
    }

    public class WorldclockInfo
    {
        private WorldclockPage page;
        private int currentTimezoneNo = 0;
        private Timezone currentTimezone;
        private int localTimezoneOffset;
        public List<Location> userLocations = new List<Location>();

        public WorldclockInfo(WorldclockPage _page)
        {
            page = _page;
            LoadItemsList();
            UpdateLocalTime();
        }

        public void Dispose()
        {
            Save();
#if CROSS_PLATFORM
            DependencyService.Get<IPreference>().SetInt("DOTNET_WORLDCLOCK_MAP_CURRENT_TIMEZONE", GetCurrentTimezoneNo());
#else
            Clock.Tizen.Impls.Preference.GetInstance().SetInt("DOTNET_WORLDCLOCK_MAP_CURRENT_TIMEZONE", GetCurrentTimezoneNo());
#endif
        }

        public void UpdateLocalTime()
        {
#if CROSS_PLATFORM
            bool exist = DependencyService.Get<IPreference>().Exist("DOTNET_WORLDCLOCK_MAP_CURRENT_TIMEZONE");
#else
            bool exist = Clock.Tizen.Impls.Preference.GetInstance().Exist("DOTNET_WORLDCLOCK_MAP_CURRENT_TIMEZONE");
#endif
            int storedTimezoneNo;
            if (exist)
            {
#if CROSS_PLATFORM
                storedTimezoneNo = DependencyService.Get<IPreference>().GetInt("DOTNET_WORLDCLOCK_MAP_CURRENT_TIMEZONE");
#else
                storedTimezoneNo = Clock.Tizen.Impls.Preference.GetInstance().GetInt("DOTNET_WORLDCLOCK_MAP_CURRENT_TIMEZONE");
#endif
                SetCurrentTimezone(storedTimezoneNo);
            }
#if CROSS_PLATFORM
            string tz = DependencyService.Get<ISystemSetting>().GetString("LOCALE_TIMEZONE");
#else
            string tz = Clock.Tizen.Impls.SystemSetting.GetInstance().GetString("LOCALE_TIMEZONE");
#endif
            string[] str = tz.Split('/');

            int _offset = 100000;
            foreach (Location l in L)
            {
                if (l.name == str[1])
                {
                    _offset = l.gmtOffset;
                }
            }

            if (_offset != 100000)
            {
                localTimezoneOffset = _offset;
            }
            else
            {
                TimeZoneInfo localZone = TimeZoneInfo.Local;
                int local_hour = Math.Abs(localZone.BaseUtcOffset.Hours);
                int local_minute = Math.Abs(localZone.BaseUtcOffset.Minutes);

                if (localZone.BaseUtcOffset >= TimeSpan.Zero)
                {
                    localTimezoneOffset = (local_hour + (local_minute / 60)) * 60;
                }
                else
                {
                    localTimezoneOffset = -((local_hour + (local_minute / 60)) * 60);
                }
            }

            if (!exist)
            {
                SetCurrentTimezone(GetTimezoneByOffset(localTimezoneOffset));
            }
        }

        private bool Exist(Location l)
        {
            if (userLocations.Count != 0)
            {
                foreach (Location item in userLocations)
                {
                    if ((l.name == item.name) && (l.country == item.country))
                    {
                        return true;
                    }
                }
            }

            return false;
        }

        public void OnItemAdded(Location l)
        {
            if (Exist(l))
            {
                Toast.DisplayText(l.name + " already exists.");
                return;
            }

            bool added = AddUserLocation(l);
            if (added)
            {
                Timezone tz = GetTimezoneByOffset(l.gmtOffset);
                SetCurrentTimezone(tz);
                page.OnMapViewUpdateRequest();
                page.cityList.AppendItemToCustomList(l);
            }

            UpdateEmptyListBackground();
        }

        bool AddUserLocation(Location l)
        {
            if (Exist(l))
            {
                // TODO: SHOW POPUP
                // %s already exists. <- content
                return false;
            }

            userLocations.Add(l);
            return true;
        }

        void RemoveUserLocations(List<Location> del)
        {
            foreach (Location item in del)
            {
                foreach (Location l in userLocations)
                {
                    if ((item.name == l.name) && (item.country == l.country))
                    {
                        userLocations.Remove(l);
                    }
                }
            }
        }

        public void Load()
        {
            LoadItemsList();
        }

        private void Save()
        {
            SaveItemsList();
        }

        void SaveItem(Location item, int i)
        {
#if CROSS_PLATFORM
            DependencyService.Get<IPreference>().SaveItem(item, i);
#else
            Clock.Tizen.Impls.Preference.GetInstance().SaveItem(item, i);
#endif
        }

        private void SaveItemsList()
        {
#if CROSS_PLATFORM
            DependencyService.Get<IPreference>().Save(userLocations);
#else
            Clock.Tizen.Impls.Preference.GetInstance().Save(userLocations);
#endif
        }

        Location LoadItem(int i)
        {
#if CROSS_PLATFORM
            return DependencyService.Get<IPreference>().LoadItem(i);
#else
            return Clock.Tizen.Impls.Preference.GetInstance().LoadItem(i);
#endif
        }

        private void LoadItemsList()
        {
#if CROSS_PLATFORM
            DependencyService.Get<IPreference>().Load(ref userLocations);
#else
            Clock.Tizen.Impls.Preference.GetInstance().Load(ref userLocations);
#endif
        }

        public int GetLocalTimeGmtOffset()
        {
            return localTimezoneOffset;
        }

        public int GetCurrentTimezoneNo()
        {
            return currentTimezoneNo;
        }

        public Timezone GetCurrentTimezone()
        {
            return timezone_[currentTimezoneNo];
        }

        public Timezone GetTimezoneByOffset(int offset)
        {
            foreach (Timezone tz in timezone_)
            {
                if (tz.gmtOffset == offset)
                {
                    return tz;
                }

            }

            return timezone_[0];
        }

        public CityRecord GetCityRecord(int offset)
        {
            int localTimezoneOffset = GetLocalTimeGmtOffset();
            int offset_integer = (Math.Abs(offset - localTimezoneOffset)) / 60;
            int offset_remainder = (Math.Abs(offset - localTimezoneOffset)) % 60;
            DateTime dateNow = DateTime.Now;
            DateTime newTime;
            string relative;

            if (offset < localTimezoneOffset)
            {
                if (offset_remainder > 0)
                {
                    relative = String.Format("{0}h {1}m behind", offset_integer, offset_remainder);
                    newTime = dateNow.Add(new TimeSpan(-(offset_integer), -(offset_remainder), 0));
                }
                else
                {
                    relative = String.Format("{0}h behind", offset_integer);
                    newTime = dateNow.Add(new TimeSpan(-(offset_integer), 0, 0));
                }
            }
            else if (offset > localTimezoneOffset)
            {
                if (offset_remainder > 0)
                {
                    relative = String.Format("{0}h {1}m ahead", offset_integer, offset_remainder);

                    newTime = dateNow.Add(new TimeSpan(offset_integer, offset_remainder, 0));
                }
                else
                {
                    relative = String.Format("{0}h ahead", offset_integer);

                    newTime = dateNow.Add(new TimeSpan(offset_integer, 0, 0));
                }
            }
            else
            {
                relative = "Same as local time";
                newTime = dateNow;
            }

            string h, m;
#if CROSS_PLATFORM
            if (DependencyService.Get<ISystemSetting>().Is24HourTimeFormatted())
#else
            if (Clock.Tizen.Impls.SystemSetting.GetInstance().Is24HourTimeFormatted())
#endif
            {
                h = String.Format("{0:HH}", newTime);
                m = String.Format("{0:MM}", newTime);
            }
            else
            {
                h = String.Format("{0:hh}", newTime);
                m = String.Format("{0:mm}", newTime);
            }

            if (h.IndexOf("0") == 0)
            {
                h = h.Substring(1);
            }

            CityRecord record = new CityRecord
            {
                CityTime = String.Format("{0}:{1}", h, m),
                CityAmPm = newTime.ToString("tt", CultureInfo.InvariantCulture),
                CityDate = String.Format("{0:ddd, d MMM}", newTime),
                RelativeToLocalCountry = relative,
                Offset = offset,
            };
            return record;
        }

        public void MoveCurrentTimezone(Direction direction)
        {
            int current = GetCurrentTimezoneNo();
            if (direction == Direction.LEFT)
            {
                if (current == 0)
                {
                    SetCurrentTimezone(timezone_.Length - 1);
                }
                else
                {
                    SetCurrentTimezone(current - 1);
                }
            }
            else
            {
                if (current == timezone_.Length - 1)
                {
                    SetCurrentTimezone(0);
                }
                else
                {
                    SetCurrentTimezone(current + 1);
                }
            }
        }

        public void SetCurrentTimezone(int no/*Timezone no*/)
        {
            currentTimezoneNo = no;
            currentTimezone = timezone_[no];
        }

        public void SetCurrentTimezone(Timezone tz)
        {
            currentTimezone = tz;
            for (int i = 0; i < timezone_.Length; i++)
            {
                if (timezone_[i].Equals(tz))
                {
                    currentTimezoneNo = i;
                }
            }
        }

        public void UpdateEmptyListBackground()
        {
            if (userLocations.Count == 0)
            {
                page.cityList.emptyListAreaLayout.IsVisible = true;
                page.cityList.cityListUI.IsVisible = false;
            }
            else
            {
                page.cityList.cityListUI.IsVisible = true;
                page.cityList.emptyListAreaLayout.IsVisible = false;
            }
        }

        public static Location[] L =
            {
            //  city, country,  , offset, x, y
            new Location("Midway Atoll", "USA", -11 * 60, 43, 218),               // GMT -11
            new Location("Pago Pago", "American Samoa", -11 * 60, 54, 230),       // GMT -11

            new Location("Honolulu", "USA", -10 * 60, 57, 223),                   // GMT -10

            new Location("Anchorage", "USA", -9 * 60, 74, 144),                   // GMT -9

            new Location("Los Angeles", "USA", -8 * 60, 131, 196),                // GMT -8
            new Location("San Francisco", "USA", -8 * 60, 126, 191),              // GMT -8
            new Location("Vancouver", "Canada", -8 * 60, 122, 166),               // GMT -8

            new Location("Denver", "USA", -7 * 60, 156, 185),                     // GMT -7

            new Location("Austin", "USA", -6 * 60, 168, 199),                     // GMT -6
            new Location("Chicago", "USA", -6 * 60, 187, 181),                    // GMT -6
            new Location("Guatemala City", "Guatemala", -6 * 60, 181, 233),       // GMT -6
            new Location("Mexico City", "Mexico", -6 * 60, 167, 244),             // GMT -6

            new Location("Bogota", "Colombia", -5 * 60, 212, 253),                // GMT -5
            new Location("Lima", "Peru", -5 * 60, 206, 286),                     // GMT -5
            new Location("Miami", "USA", -5 * 60, 200, 211),                      // GMT -5
            new Location("New York", "USA", -5 * 60, 211, 184),                   // GMT -5
            new Location("Ottawa", "Canada", -5 * 60, 208, 173),                  // GMT -5
            new Location("Washington DC", "USA", -5 * 60, 205, 189),              // GMT -5

            new Location("Santiago", "Chile", -4 * 60, 217, 327),                 // GMT -4
            new Location("Santo Domingo", "Dominican Republic", -4 * 60, 218, 226),   // GMT -4

            new Location("Brasilia", "Brazil", -3 * 60, 257, 290),                    // GMT -3
            new Location("Buenos Aires", "Argentina", -3 * 60, 239, 329),             // GMT -3
            new Location("Nuuk", "Greenland", -3 * 60, 256, 143),                     // GMT -3
            new Location("Sao Paulo", "Brazil", -3 * 60, 260, 308),                   // GMT -3

            new Location("Mid-Atlantic", "", -2 * 60, 296, 266),                       // GMT -2
            new Location("Grytvieken", "South Georgia", -2 * 60, 276, 364),

            new Location("Azores", "Portugal", -1 * 60, 308, 188),                    // GMT -1
            new Location("Ponta Delaga", "Portugal", -1 * 60, 300, 190),

            new Location("Accra", "Ghana", 0, 343, 250),                               // GMT 0
            new Location("Dakar", "Senegal", 0, 314, 234),
            new Location("Lisbon", "Portugal", 0, 327, 187),
            new Location("London", "United Kingdom", 0, 343, 162),
            new Location("Reykjavik", "Iceland", 0, 306, 138),

            new Location("Algiers", "Algeria", 1 * 60, 349, 190),                       // GMT +1
            new Location("Bracelona", "Spain", 1 * 60, 346, 181),
            new Location("Berlin", "Germany", 1 * 60, 367, 162),
            new Location("Luanda", "Angola", 1 * 60, 367, 278),
            new Location("Madrid", "Spain", 1 * 60, 337, 185),
            new Location("Paris", "France", 1 * 60, 349, 168),
            new Location("Rome", "Italy", 1 * 60, 365, 180),
            new Location("Stockholm", "Sweden", 1 * 60, 375, 147),

            new Location("Athens", "Greece", 2 * 60, 385, 188),                       // GMT +2
            new Location("Cairo", "Egypt", 2 * 60, 400, 205),
            new Location("Cape Town", "South Africa", 2 * 60, 376, 326),
            new Location("Harare", "Zimbabwe", 2 * 60, 403, 293),
            new Location("Istanbul", "Turkey", 2 * 60, 395, 183),

            new Location("Doha", "Qatar", 3 * 60, 437, 213),                       // GMT +3
            new Location("Nairobi", "Kenia", 3 * 60, 409, 263),
            new Location("Moscow", "Russia", 3 * 60, 410, 155),
            new Location("Tehran", "Iran", 3 * 60 + 30, 435, 192),


            new Location("Izhevsk", "Russia", 4 * 60, 437, 152),                       // GMT +4
            new Location("Dubai", "United Arab Emirates", 4 * 60, 441, 213),
            new Location("Kabul", "Afghanistan", 4 * 60 + 30, 464, 194),


            new Location("Islamabad", "Pakistan", 5 * 60, 467, 196),                       // GMT +5
            new Location("Tashkent", "Uzbekistan", 5 * 60, 467, 182),
            new Location("Delhi", "India", 5 * 60 + 30, 481, 203),
            new Location("Kathmandu", "Nepal", 5 * 60 + 45, 494, 204),


            new Location("Almaty", "Kazakhstan", 6 * 60, 481, 177),                       // GMT +6
            new Location("Bishkek", "Kyrgyzstan", 6 * 60, 477, 179),
            new Location("Dhaka", "Bangladesh", 6 * 60, 504, 213),
            new Location("Omsk", "Russia", 6 * 60, 474, 156),


            new Location("Bangkok", "Thailand", 7 * 60, 522, 235),                       // GMT +7
            new Location("Hanoi", "Vietnam", 7 * 60, 532, 220),
            new Location("Jakarta", "Indonesia", 7 * 60, 532, 274),


            new Location("Beijing", "China", 8 * 60, 550, 185),                       // GMT +8
            new Location("Kuala Lumpur", "Malaysia", 8 * 60, 523, 255),
            new Location("Manila", "Philipines", 8 * 60, 559, 234),
            new Location("Perth", "Australia", 8 * 60, 550, 322),
            new Location("Singapore", "Republic of Singapore", 8 * 60, 528, 258),
            new Location("Ulan Bator", "Mongolia", 8 * 60, 535, 168),


            new Location("Seoul", "South Korea", 9 * 60, 570, 190),                       // GMT +9
            new Location("Tokyo", "Japan", 9 * 60, 592, 193),

            new Location("Sydney", "Australia", 10 * 60, 613, 326),                       // GMT +10

            new Location("Noumea", "New Caledonia", 11 * 60, 639, 304),                       // GMT +11
            new Location("Khabarowsk", "Russia", 11 * 60, 584, 169),

            new Location("Auckland", "New Zealand", 12 * 60, 654, 328),                       // GMT +12

            new Location("Nuku'alofa", "Tonga", 13 * 60, 668, 304),
            new Location("Caracas", "Venezuela", -4 * 60 - 30, 219, 247),
            new Location("St John's", "Newfoundland", -3 * 60 - 30, 249, 166),
            new Location("Yangon", "Myanmar", 6 * 60 + 30, 516, 232),
            new Location("Adelaide", "Australia", 9 * 60 + 30, 603, 332),
        };

        // total 33 timezones
        internal static Timezone[] timezone_ =
        {
            //x_coord, zone_width(DOT), gmt_offset, places
            new Timezone(44, 24, -11 * 60, new Location[] { L[0], L[1] }),                                         // GMT -11
            new Timezone(44, 24, -10 * 60, new Location[] { L[2] }),                                               // GMT -10
            new Timezone(68, 24, -9 * 60, new Location[] { L[3] }),                                                // GMT -9
            new Timezone(116, 30, -8 * 60, new Location[] { L[4], L[5], L[6] }),                                     // GMT -8
            new Timezone(146, 30, -7 * 60, new Location[] { L[7] }),                                              // GMT -7
            new Timezone(146, 47, -6 * 60, new Location[] { L[8], L[9], L[10], L[11] }),                           // GMT -6
            new Timezone(193, 32, -5 * 60, new Location[] { L[12], L[13], L[14], L[15], L[16], L[17] }),
            new Timezone(193, 61, -270, new Location[] { L[77] }),
            new Timezone(193, 61, -4 * 60, new Location[] { L[18], L[19] }),
            new Timezone(225, 59, -210, new Location[] { L[78] }),
            new Timezone(225, 59, -3 * 60, new Location[] { L[20], L[21], L[22], L[23] }),
            new Timezone(254, 54, -2 * 60, new Location[] { L[24], L[25] }),
            new Timezone(284, 24, -1 * 60, new Location[] { L[26], L[27] }),
            new Timezone(308, 41, 0 * 60, new Location[] { L[28], L[29], L[30], L[31], L[32] }),
            new Timezone(332, 45, 1 * 60, new Location[] { L[33], L[34], L[35], L[36], L[37], L[38], L[39], L[40] }),
            new Timezone(349, 85, 2 * 60, new Location[] { L[41], L[42], L[43], L[44], L[45] }),
            new Timezone(377, 81, 3 * 60, new Location[] { L[46], L[47], L[48] }),
            new Timezone(402, 56, 210 /*(3.5 * 60)*/, new Location[] { L[49] }),
            new Timezone(402, 56, 4 * 60, new Location[] { L[50], L[51] }),
            new Timezone(458, 28, 270 /* (4.5 * 60) */, new Location[] { L[52] }),
            new Timezone(458, 28, 5 * 60, new Location[] { L[53], L[54] }),
            new Timezone(458, 28, 330 /*(5.5 * 60)*/, new Location[] { L[55] }),
            new Timezone(486, 28, 345 /* (5.75 * 60) */, new Location[] { L[56] }),
            new Timezone(458, 56, 6 * 60, new Location[] { L[57], L[58], L[59], L[60] }),
            new Timezone(498, 24, 390, new Location[] { L[79] }),
            new Timezone(514, 24, 7 * 60, new Location[] { L[61], L[62], L[63] }),
            new Timezone(514, 54, 8 * 60, new Location[] { L[64], L[65], L[66], L[67], L[68], L[69] }),
            new Timezone(538, 84, 9 * 60, new Location[] { L[70], L[71] }),
            new Timezone(538, 84, 570, new Location[] { L[80] }),
            new Timezone(595, 27, 10 * 60, new Location[] { L[72] }),
            new Timezone(568, 76, 11 * 60, new Location[] { L[73], L[74] }),
            new Timezone(644, 24, 12 * 60, new Location[] { L[75] }),
            new Timezone(655, 24, 13 * 60, new Location[] { L[76] })
        };

    }
}
