/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

using Xamarin.Forms;

namespace Clock.Alarm
{
    /// <summary>
    /// This class is to define alarm tone setting page
    /// This inherits from ContentPage
    /// Users can select proper alarm tone in this page
    /// </summary>
    public class AlarmTonePage : ContentPage
    {
        /// <summary>
        /// Main StackLayout 
        /// </summary>        
        StackLayout mainLayout;

        /// <summary>
        /// Table view to show each tone type
        /// </summary>        
        private TableView alarmToneTable;

        /// <summary>
        /// Tile bar area
        /// </summary>        
        private AlarmTitleBar titleBar;

        /// <summary>
        /// Default tone cell
        /// </summary>        
        AlarmToneTableCell defaultTone;

        /// <summary>
        /// Alarm MP3 tone cell
        /// </summary>        
        AlarmToneTableCell alarmMP3Tone;

        /// <summary>
        /// Alarm ringtone tone cell
        /// </summary>        
        AlarmToneTableCell ringtoneMP3Tone;

        /// <summary>
        /// Constructor for this page
        /// </summary>     
        public AlarmTonePage()
        {
            NavigationPage.SetHasNavigationBar(this, false);
            Draw();
        }

        /// <summary>
        /// This method is called when this page is being shown after creation
        /// </summary>     
        internal void Update()
        {
            Draw();
        }

        /// <summary>
        /// Actual drawing operations are performed in this method
        /// </summary>    
        internal void Draw()
        {
            /// Sets title bar
            titleBar = new AlarmTitleBar();
            /// Sets left button text
            titleBar.LeftButton.Text = "Cancel";
            /// Sets right button text
            titleBar.RightButton.Text = "OK";
            /// Sets title label
            titleBar.TitleLabel.Text = "Select";

            /// Sets event callback when right button is called (update the tone info)
            titleBar.RightButton.Clicked += (s, e) =>
            {
                AlarmEditPage.BindableAlarmRecord.AlarmToneType = AlarmToneRow.newValue;
                Navigation.PopAsync();
            };

            /// Sets default tone table cell
            defaultTone = new AlarmToneTableCell(AlarmToneTypes.Default, this);
            /// Sets alarm mp3 tone table cell
            alarmMP3Tone = new AlarmToneTableCell(AlarmToneTypes.AlarmMp3, this);
            /// Sets ringtone tone table cell
            ringtoneMP3Tone = new AlarmToneTableCell(AlarmToneTypes.RingtoneSdk, this);

            /// Sets table root to hold all three table cells
            TableRoot root = new TableRoot()
            {
                new TableSection()
                {
                    defaultTone,
                    alarmMP3Tone,
                    ringtoneMP3Tone,
                }
            };

            /// Sets TableView
            alarmToneTable = new TableView(root);

            /// Sets titlebar and alarmToneTable
            mainLayout = new StackLayout
            {
                HorizontalOptions = LayoutOptions.FillAndExpand,
                Children =
                {
                    titleBar,
                    alarmToneTable
                }
            };
            Content = mainLayout;
        }
    }
}
