﻿/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

using Clock.Styles;
using Xamarin.Forms;
using Xamarin.Forms.PlatformConfiguration.TizenSpecific;
using Label = Xamarin.Forms.Label;
using Entry = Xamarin.Forms.Entry;

namespace Clock.Alarm
{
    /// <summary>
    /// Alarm name row class.
    /// This class defines layout of alarm name row
    /// </summary>
    class AlarmNameRow : RelativeLayout
    {
        /// <summary>
        /// Main label
        /// </summary>
        private Label mainLabel;

        /// <summary>
        /// Entry to get alarm name from users. If there is a name already set by users, then 
        /// this name should be set as placeholder.
        /// If no placeholder is set then, it shows "Alarm"
        /// </summary>
        internal Entry mainEntry;

        /// <summary>
        /// Construct alarm name row UI
        /// </summary>
        /// <param name="editType">type indicator</param>
        /// <seealso cref="AlarmEditTypes">
        public AlarmNameRow(AlarmEditTypes editType)
        {
            /// Fills horizontally
            HorizontalOptions = LayoutOptions.FillAndExpand;
            /// Fills vertically
            VerticalOptions = LayoutOptions.Start;
            HeightRequest = 144;

            if (mainLabel == null)
            {
                mainLabel = new Label
                {
                    HeightRequest = 54,
                    Style = AlarmStyle.T023,
                };
                /// Needs custom SetFontWeight to represent thin font
                mainLabel.On<Xamarin.Forms.PlatformConfiguration.Tizen>().SetFontWeight(FontWeight.Light);
            }

            /// Sets main label
            mainLabel.Text = "Alarm Name";
            /// Adds main label
            Children.Add(mainLabel,
                Constraint.RelativeToParent((parent) => { return 32; }),
                Constraint.RelativeToParent((parent) => { return (144 - (54 + 43)) / 2; }));
            /// Creates entry
            mainEntry = new Entry();
            /// Sets width for entry
            mainEntry.WidthRequest = 720 - (32 + 10) * 2;
            /// Checks whether already set alarm name
            if (string.IsNullOrEmpty(AlarmEditPage.BindableAlarmRecord.AlarmName))
            {
                mainEntry.Placeholder = "Alarm";
            }
            else
            {
                mainEntry.Text = AlarmEditPage.BindableAlarmRecord.AlarmName;
            }

            /// Sets font size (DP base)
            mainEntry.FontSize = CommonStyle.GetDp(40);
            /// Adds to layout
            Children.Add(mainEntry,
                Constraint.RelativeToView(mainLabel, (parent, sibling) => { return sibling.X + 10; }),
                Constraint.RelativeToView(mainLabel, (parent, sibling) => { return sibling.Y + sibling.Height; }));
        }
    }
}
