/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

using System;
using System.Collections.Generic;
using System.Collections.Specialized;
using Xamarin.Forms;
using System.ComponentModel;
using Clock.Interfaces;
using Clock.Styles;

namespace Clock.Alarm
{
    /// <summary>
    /// The enumeration of AlarmStates
    /// </summary>
    public enum AlarmStates
    {
        /// <summary>
        /// Identifier for Active
        /// </summary>
        Active,
        /// <summary>
        /// Identifier for Snooze
        /// </summary>
        Snooze,
        /// <summary>
        /// Identifier for Inactive
        /// </summary>
        Inactive,
    }

    /// <summary>
    /// The enumeration of AlarmTone(media files to ring by playing)
    /// </summary>
    public enum AlarmToneTypes
    {
        /// <summary>
        /// Identifier for default alarm tone type
        /// </summary>
        Default,
        /// <summary>
        /// Identifier for alarm mp3 type
        /// </summary>
        AlarmMp3,
        /// <summary>
        /// Identifier for sdk ringtone type
        /// </summary>
        RingtoneSdk,
    }

    /// <summary>
    /// The enumeration of alarm type
    /// </summary>
    public enum AlarmTypes
    {
        /// <summary>
        /// Identifier for sound
        /// </summary>
        Sound,
        /// <summary>
        /// Identifier for vibration without making a sound
        /// </summary>
        Vibration,
        /// <summary>
        /// Identifier for vibration and sound
        /// </summary>
        SoundVibration
    }

    /// <summary>
    /// The enumeration of alarm editing type
    /// </summary>
    public enum AlarmEditTypes
    {
        /// <summary>
        /// Identifier for alarm edit
        /// </summary>
        Edit,
        /// <summary>
        /// Identifier for editing alarm repeat
        /// </summary>
        Repeat,
        /// <summary>
        /// Identifier for editing alarm type
        /// </summary>
        Type,
        /// <summary>
        /// Identifier for editing alarm sound
        /// </summary>
        Sound,
        /// <summary>
        /// Identifier for editing alarm tone
        /// </summary>
        Tone,
        /// <summary>
        /// Identifier for editing alarm snooze
        /// </summary>
        Snooze,
        /// <summary>
        /// Identifier for editing alarm name
        /// </summary>
        Name
    }

    /// <summary>
    /// The AlarmRecord class
    /// </summary>
    public class AlarmRecord : INotifyPropertyChanged
    {
        private AlarmTypes _alarmType;
        private AlarmToneTypes _alarmTone;
        private AlarmWeekFlag _weekFlag;
        private string _alarmName;
        // this dictionary should be filled at the starting time

        /// <summary>
        /// Gets/Sets AlarmRecordDictionary
        /// </summary>
        public static IDictionary<string, AlarmRecord> AlarmRecordDictionary {  get; set; }
        private FormattedString _weekdayRepeatText;
        private string _scheduledTimeText;
        private string _scheduledAmPmText;
        private DateTime _scheduledDateTime;

        ///<summary>
        ///Event that is raised when the properties of AlarmRecord change
        ///</summary>
        public event PropertyChangedEventHandler PropertyChanged;
        //public Tizen.Applications.Alarm Alarm { get; set; }

        /// <summary>
        /// Gets/Sets whether the AlarmRecord object has been serialized
        /// </summary>
        public bool IsSerialized { get; set; }

        /// <summary>
        /// Gets/Sets ScheduledDateTime
        /// </summary>
        public DateTime ScheduledDateTime
        {
            get
            {
                return _scheduledDateTime;
            }

            set
            {
                _scheduledDateTime = value;
                FillTextPropertiesFromDate();
            }
        }

        /// <summary>
        /// Gets/Sets the day of week when an alarm recurs
        /// </summary>
        public AlarmWeekFlag WeekFlag
        {
            get
            {
                return _weekFlag;
            }

            set
            {
                if (_weekFlag != value)
                {
                    _weekFlag = value;
                    FillTextPropertiesFromDate();
                    PropertyChanged?.Invoke(this, new PropertyChangedEventArgs("WeekFlag"));
                }
            }
        }

        /// <summary>
        /// Gets/Sets the scheduled time (hour and minute) of alarm
        /// </summary>
        public string ScheduledTimeText
        {
            get
            {
                return _scheduledTimeText;
            }

            set
            {
                if (_scheduledTimeText != value)
                {
                    _scheduledTimeText = value;
                    PropertyChanged?.Invoke(this, new PropertyChangedEventArgs("ScheduledTimeText"));
                }
            }
        }

        /// <summary>
        /// Gets/Sets AM/PM settings on alarm
        /// </summary>
        public string ScheduledAmPmText
        {
            get
            {
                return _scheduledAmPmText;
            }

            set
            {
                if (_scheduledAmPmText != value)
                {
                    _scheduledAmPmText = value;
                    PropertyChanged?.Invoke(this, new PropertyChangedEventArgs("ScheduledAmPmText"));
                }
            }
        }

        /// <summary>
        /// Gets/Sets whether alarm repeats or not
        /// </summary>
        public bool Repeat { get; set; }

        /// <summary>
        /// Gets/Sets a formatted string about weekday alarm repeat information
        /// </summary>
        public FormattedString WeekdayRepeatText
        {
            get
            {
                return _weekdayRepeatText;
            }

            set
            {
                if (_weekdayRepeatText != value)
                {
                    _weekdayRepeatText = value;
                    PropertyChanged?.Invoke(this, new PropertyChangedEventArgs("WeekdayRepeatText"));
                }
            }
        }

        /// <summary>
        /// Gets/Sets the name of alarm
        /// </summary>
        public string AlarmName
        {
            get
            {
                return _alarmName;
            }

            set
            {
                if (_alarmName != value)
                {
                    _alarmName = value;
                    PropertyChanged?.Invoke(this, new PropertyChangedEventArgs("AlarmName"));
                }
            }
        }

        /// <summary>
        /// Gets/Sets the state of alarm
        /// </summary>
        public AlarmStates AlarmState { get; set; }

        /// <summary>
        /// Gets/Sets the tone of alarm
        /// </summary>
        public AlarmToneTypes AlarmToneType
        {
            get
            {
                return _alarmTone;
            }

            set
            {
                if (_alarmTone != value)
                {
                    _alarmTone = value;
                    PropertyChanged?.Invoke(this, new PropertyChangedEventArgs("AlarmToneType"));
                }
            }
        }

        /// <summary>
        /// Gets/Sets alarm creation time
        /// </summary>
        public TimeSpan DateCreated { get; set; }

        /// <summary>
        /// Serializes alarm record dictionary object and save it to file
        /// </summary>
        /// <param name="alarmRecordDictionary">AlarmRecordDictionary object to serialize</param>
        public static void SaveDictionary(object alarmRecordDictionary)
        {
#if CROSS_PLATFORM
            IAlarmPersistentHandler serizliazer =  DependencyService.Get<IAlarmPersistentHandler>();
            serizliazer.SerializeAlarmRecordAsync((IDictionary<string, AlarmRecord>)alarmRecordDictionary).Wait();
#else
            Tizen.Impls.AlarmPersistentHandler.GetInstance().SerializeAlarmRecordAsync((IDictionary<string, AlarmRecord>)alarmRecordDictionary).Wait();
#endif
        }

        /// <summary>
        /// Gets/Sets alarm type
        /// </summary>
        public AlarmTypes AlarmType
        {
            get
            {
                return _alarmType;
            }

            set
            {
                if (_alarmType != value)
                {
                    _alarmType = value;
                    PropertyChanged?.Invoke(this, new PropertyChangedEventArgs("AlarmType"));
                }
            }
        }

        /// <summary>
        /// Gets/Sets alarm volume
        /// </summary>
        public float Volume { get; set; }

        /// <summary>
        /// Gets/Sets whether alarm snoozes or not
        /// </summary>
        public bool Snooze { get; set; }

        /// <summary>
        /// Gets/Sets system alarm ID
        /// </summary>
        public int NativeAlarmID { get; set; }

        /// <summary>
        /// Initializes a new instance of the AlarmRecord class
        /// </summary>
        public AlarmRecord()
        {
            SetDefault();
            FillTextPropertiesFromDate();
        }

        /// <summary>
        /// Converts the string of the creation date time
        /// </summary>
        /// <returns> Returns a string representation of value of the alarm creation time</returns>
        public string GetUniqueIdentifier()
        {
            //return ScheduledDateTime.Hour + ":" +
            //       ScheduledDateTime.Minute;
            return DateCreated.ToString();
        }

        /// <summary>
        /// Returns dictionary key
        /// </summary>
        /// <returns> Returns a string of scheduled time </returns>
        public string GetAlarmTimeKey()
        {
            return ScheduledDateTime.Hour + ":" +
                   ScheduledDateTime.Minute;
        }

        /// <summary>
        /// Sets the default value for AlarmRecord object
        /// </summary>
        private void SetDefault()
        {
            /// Gets current time
            DateTime tmpNow = System.DateTime.Now;
            /// Sets alarm name to empty
            AlarmName = "";
            /// Sets default state to active
            AlarmState = AlarmStates.Active;
            /// Sets default alarm to alarm mp3
            AlarmToneType = AlarmToneTypes.AlarmMp3;
            /// Sets default alarm type to vibration
            AlarmType = AlarmTypes.Vibration;
            /// This indicates whether it is newly created or not.
            /// This value should set to false at default creation time
            IsSerialized = false;
            /// Sets repeat to false
            Repeat = false;
            /// Sets scheduled date time for the tmpNow
            ScheduledDateTime = tmpNow;
            /// Sets time text to empty
            ScheduledTimeText = "";
            /// Sets ap pm to empty
            ScheduledAmPmText = "";
            /// Sets snooze to false
            Snooze = false;
            /// Sets volume to 0.1
            Volume = 0.1f;
            /// Sets weekRepeatText to empty
            WeekdayRepeatText = "";
            /// Sets weekday flag to all days
            WeekFlag = AlarmWeekFlag.AllDays;
            /// This field is used to uniquely identify alarm by created time
            DateCreated = tmpNow.TimeOfDay;
        }

        /// <summary>
        /// Deep copy alarm record
        /// </summary>
        /// <param name="bindableAlarmRecord">Alarm record to deep copy</param>
        /// <seealso cref="AlarmRecord">
        public void DeepCopy(AlarmRecord bindableAlarmRecord)
        {
            // All properties should be copied to this alarm record object
            AlarmName = bindableAlarmRecord.AlarmName;
            AlarmState = bindableAlarmRecord.AlarmState;
            AlarmToneType = bindableAlarmRecord.AlarmToneType;
            AlarmType = bindableAlarmRecord.AlarmType;
            IsSerialized = bindableAlarmRecord.IsSerialized;
            Repeat = bindableAlarmRecord.Repeat;
            ScheduledDateTime = bindableAlarmRecord.ScheduledDateTime;
            ScheduledTimeText = bindableAlarmRecord.ScheduledTimeText;
            ScheduledAmPmText = bindableAlarmRecord.ScheduledAmPmText;
            Snooze = bindableAlarmRecord.Snooze;
            Volume = bindableAlarmRecord.Volume;
            WeekdayRepeatText = bindableAlarmRecord.WeekdayRepeatText;
            WeekFlag = bindableAlarmRecord.WeekFlag;
            DateCreated = bindableAlarmRecord.DateCreated;
        }

        /// <summary>
        /// Sets observable alarm record list from alarm record dictionary
        /// If no alarm record dictionary set yet, then do nothing
        /// </summary>
        /// <param name="alarmListLocal">A collection of AlarmRecord</param>
        internal static void GetAlarmRecordsFromAlarmRecordDictionary(AlarmExtendedObservableCollections<AlarmRecord> alarmListLocal)
        {
            // Null check for AlarmRecordDictionary
            if (AlarmRecordDictionary == null)
            {
                return;
            }

            /// Iterates alarm record dictionary to add AlarmListUI
            foreach (var alarmItem in AlarmRecordDictionary)
            {
                // Key is DateCreated
                var keyString = alarmItem.Key;
                // Retrieve alarm record
                AlarmRecord retrieved = alarmItem.Value; //AlarmRecord does not have 
                // Add to alarmListLocal which is source for AlarmListUI
                alarmListLocal.Add(retrieved);
            }
        }

        /// <summary>
        /// Fill properties from date.
        /// AM/PM, ScheduledTime, Weekdayrepeat can get set based on AlarmRecord time and weekdayrepeat info
        /// </summary>
        private void FillTextPropertiesFromDate()
        {
            /// AM/PM
            ScheduledAmPmText = (ScheduledDateTime.Hour > 11) ? "PM" : "AM";
            /// Hour, min
            int hour = ScheduledDateTime.Hour;

            /// Hour is always 12 or less
            if (ScheduledAmPmText == "PM" && hour > 12)
            {
                hour -= 12;
            }

            /// Sets hour 
            ScheduledTimeText = hour.ToString();
            /// Sets :
            ScheduledTimeText += ":";
            /// Sets Minute
            string minText = (ScheduledDateTime.Minute).ToString();
            /// Sets following 0 if minute text length is 1
            if (minText.Length == 1)
            {
                minText = "0" + minText;
            }

            // Adds to time text
            ScheduledTimeText += minText;

            //formatted WeekDayRepeat
            WeekdayRepeatText = GetFormatted(WeekFlag, AlarmState < AlarmStates.Inactive ? true : false);
        }

        /// <summary>
        /// When property is changed, registered event callback is invoked
        /// </summary>
        /// <param name="propertyName">Property name of the change event occured</param>
        protected virtual void OnPropertyChanged(string propertyName)
        {
            if (PropertyChanged != null)
            {
                PropertyChanged(this,
                    new PropertyChangedEventArgs(propertyName));
            }
        }

        /// <summary>
        /// Gets week day string
        /// </summary>
        /// <param name="i">weekday in integer (1 is Mon, 6 is Sat, 0 is Sun).</param>
        /// <returns>Returns string about week day</returns>
        private string GetWeekDay(int i)
        {
            string s = "";
            switch (i)
            {
                case 1:
                s = "M ";
                break;
                case 2:
                s = "T ";
                break;
                case 3:
                s = "W ";
                break;
                case 4:
                s = "T ";
                break;
                case 5:
                s = "F ";
                break;
                case 6:
                s = "S ";
                break;
                case 0:
                s = "S ";
                break;
            }

            return s;
        }

        /// <summary>
        /// Gets formatted string based on week flag and state (active or inactive)
        /// </summary>
        /// <param name="weekFlag">bit flag to indicate alarm repeat days </param>
        /// <param name="isNormal">whether it is active or inactive (true or false)</param>
        /// <returns>Returns formatted string for alarm time information</returns>
        internal FormattedString GetFormatted(AlarmWeekFlag weekFlag, bool isNormal)
        {
            FormattedString newStr = new FormattedString();

            for (int i = 1; i <= 7; i++) // Mon to Saturday
            {
                if (i == 7)
                {
                    i = 0;
                }

                int mask = 1 << i;

                if (((int)weekFlag & mask) > 0)
                {
                    if (isNormal)
                    {
                        newStr.Spans.Add(new Span
                        {
                            ForegroundColor = Color.FromHex("FF59B03A"),
                            Text = GetWeekDay(i),
                            FontSize = CommonStyle.GetDp(32)
                        });
                    }
                    else
                    {
                        newStr.Spans.Add(new Span
                        {
                            ForegroundColor = Color.FromHex("6659B03A"),
                            Text = GetWeekDay(i),
                            FontSize = CommonStyle.GetDp(32)
                        });
                    }
                }
                else
                {
                    if (isNormal)
                    {
                        newStr.Spans.Add(new Span
                        {
                            ForegroundColor = Color.FromHex("FFB3B3B3"),
                            Text = GetWeekDay(i),
                            FontSize = CommonStyle.GetDp(32)
                        });
                    }
                    else
                    {
                        newStr.Spans.Add(new Span
                        {
                            ForegroundColor = Color.FromHex("66B3B3B3"),
                            Text = GetWeekDay(i),
                            FontSize = CommonStyle.GetDp(32)
                        });
                    }
                }

                if (i == 0)
                {
                    break;
                }
            }

            return newStr;
        }
    }

    /// <summary>
    /// The AlarmRecordObservableCollection class which represents a data collection that provides notifications when items get added, removed.
    /// </summary>
    public class AlarmRecordObservableCollection : List<AlarmRecord>, INotifyCollectionChanged
    {

        public AlarmRecordObservableCollection()
        {
        }

        /// <summary>
        /// The event that notifies dynamic changes, such as when items get added or removed
        /// </summary>
        public event NotifyCollectionChangedEventHandler CollectionChanged;

        /// <summary>
        /// Adds alarm record object to a list
        /// </summary>
        /// <param name="item">AlarmRecord object to add</param>
        /// <seealso cref="AlarmRecord">
        public new void Add(AlarmRecord item)
        {
            base.Add(item);
            CollectionChanged?.Invoke(this,
                new NotifyCollectionChangedEventArgs(NotifyCollectionChangedAction.Add, item));
        }

        /// <summary>
        /// Inserts a collection of alarm records in the list at the specified index.
        /// </summary>
        /// <param name="index">The zero-based index at which the new elements should be inserted.</param>
        /// <param name="collection">The collection of alarm records whose elements should be inserted into the AlarmRecordObservableCollection. </param>
        public new void InsertRange(int index, IEnumerable<AlarmRecord> collection)
        {
            base.InsertRange(index, collection);
            CollectionChanged?.Invoke(this,
                new NotifyCollectionChangedEventArgs(NotifyCollectionChangedAction.Add, new List<AlarmRecord>(collection), index));
        }

        /// <summary>
        /// Removes the specified alarm record object from the list.
        /// </summary>
        /// <param name="item">AlarmRecord object to remove from the AlarmRecordObservableCollection</param>
        /// <seealso cref="AlarmRecord">
        public new void Remove(AlarmRecord item)
        {
            base.Remove(item);
            CollectionChanged?.Invoke(this,
                new NotifyCollectionChangedEventArgs(NotifyCollectionChangedAction.Remove, item));
        }
    }
}
